/*******************************************************************************
 * See COPYRIGHT.txt & LICENSE.txt for copyright and licensing details.
 *******************************************************************************/

/*
 * qfle3f_io.c: QLogic 10 Gigabit ESX Ethernet FCoE offload driver.
 *
 * IO manager and SCSI IO processing.
 */

#include "qfle3f.h"
#include "ql_fcoe_mgmt/FC_FCP.h"

#define SUCCESS		0
#define FAILED		1

static int qfle3f_splitBD(struct qfle3fCommand *ioRequest, vmk_uint64 addr, int sg_len,
			   int bd_index);
static int qfle3f_mapSG(struct qfle3fCommand *ioRequest);
static void qfle3f_buildBDListFromSG(struct qfle3fCommand *ioRequest);
int qfle3f_postIORequest(struct qfle3f_rport *target,
			       struct qfle3fCommand *ioRequest);
static void qfle3f_unmapSGList(struct qfle3fCommand *ioRequest);
static void qfle3f_freeMPResources(struct qfle3fCommand *ioRequest);
static void qfle3f_parseFCPResponse(struct qfle3fCommand *ioRequest,
				 struct fcoe_fcp_rsp_payload *fcp_rsp,
				 vmk_uint8 num_rq);
static void qfle3f_commandTimeout(void *data);
int qfle3f_explicitLogout(struct qfle3fCommand *ioRequest, vmk_Bool *pDroppedIoRef);
struct qfle3fCommand *qfle3f_getCmdSatisfyingACondition(struct qfle3fFCLun *fclun,
        struct vmk_ScsiTaskMgmt *taskMgmt,
        vmk_Bool (*checkCondition)(struct qfle3fCommand *, struct qfle3fFCLun *, struct vmk_ScsiTaskMgmt *));
vmk_Bool checkIoTmDoneCalled(void *data);

vmk_Bool checkIoTmDoneCalled(void *data) {
    struct qfle3fCommand *ioRequest = (struct qfle3fCommand *) data;
    return (vmk_Bool)(vmk_AtomicRead64(&ioRequest->tm_done_invoked));
}

void qfle3f_commandTimerSet(struct qfle3fCommand *ioRequest,
			  vmk_uint32 timer_msec)
{
	struct qfle3fHBA *hba = ioRequest->vhba;
	VMK_ReturnStatus status;

	status = ql_vmk_queue_delayed_work(qfle3fDriverInfo.delayedTQ,
				hba->timerWorkQueue,
				&ioRequest->timeout_work,
				qfle3f_commandTimeout, ioRequest, NULL,
				timer_msec);

	if(status == VMK_OK) {
		ql_vmk_ref_get(&ioRequest->refcount);	/*ID: 002 */
    } else {
        qfle3f_log(hba, LOG_IOERR, "scheduling of delayed timer failed: %s",
                    vmk_StatusToString(status));
    }
}

static void qfle3f_commandTimeout(void *data)
{
    ql_vmk_singlethread_workitem_t *work = (ql_vmk_singlethread_workitem_t *) data;
	struct qfle3fCommand *ioRequest = (struct qfle3fCommand *) work->data1;
	vmk_uint8 cmd_type = ioRequest->cmd_type;
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba   = target->hba;
	int logo_issued;
	/* int rc; */
	VMK_ReturnStatus status;

	qfle3f_log(hba, LOG_IOERR, "cmd_type = %d, req_flags = 0x%x oxid = 0x%x",
		cmd_type, ioRequest->req_flags->vector[0], ioRequest->xid);

	vmk_SpinlockLock(target->targetLock);

	if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_RRQ)) {
		vmk_BitVectorClear(ioRequest->req_flags, QFLE3F_FLAG_RETIRE_OXID);
		/*
		 * ideally we should hold the ioRequest until RRQ complets,
		 * and release ioRequest from timeout hold.
		 */
		vmk_SpinlockUnlock(target->targetLock);
		qfle3f_send_rrq(ioRequest);
		return;
	}
	if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags, QFLE3F_FLAG_RETIRE_OXID)) {
		qfle3f_log(hba, LOG_IOERR, "IO with oxid = 0x%x is "
				"ready for reuse now", ioRequest->xid);
		goto done;
	}

	switch(cmd_type) {

	case QFLE3F_SCSI_CMD:
		if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags, QFLE3F_FLAG_EH_ABORT)) {
			/* Handle eh_abort timeout */
			qfle3f_log(hba, LOG_IOERR, "eh_abort for IO with oxid = 0x%x"
					" timed out", ioRequest->xid);
			qfle3f_notice(hba, "tm_compl - wake up the waiter = %p", &(ioRequest->tm_done));
			vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
			status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
			if(status != VMK_OK)
				qfle3f_log(hba, LOG_IOERR, "world wakeup on "
							"event tm_done: %s", vmk_StatusToString(status));
		} else if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_ABTS)) {
			/* Handle internally generated ABTS timeout */
			qfle3f_log(hba, LOG_IOERR, "ABTS for IO with oxid = 0x%x timed out",
					ioRequest->xid);
			if (!(vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags, QFLE3F_FLAG_ABTS_DONE))) {

				/* If flush is in progress, don't do anything.
				 * flushActiveIO's would clean this IO.
				 */
				if(target->flushInProgress) {
					qfle3f_log(hba, LOG_IOERR, "Flush is progress. Do Nothing.");
					goto done;
				}

				logo_issued = vmk_BitVectorAtomicTestAndSet( target->flags,
						QFLE3F_FLAG_EXPL_LOGO);
				ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
				vmk_SpinlockUnlock(target->targetLock);

				/* Explicitly logo the target */
				if (!logo_issued) {
					if(!vmk_BitVectorAtomicTestAndClear(target->flags, QFLE3F_FLAG_SESSION_READY)) {
						vmk_BitVectorClear(target->flags, QFLE3F_FLAG_EXPL_LOGO);
						qfle3f_log(hba, LOG_IOERR, "Session not ready. No need to do Explicit Logout.");
						return;
					}
					qfle3f_log(hba, LOG_IOERR, "Explicit logo"
						   "(scsi) xid - 0x%x",
						   ioRequest->xid);

					qfle3f_notice(hba, "DoPortLogout: targetPortID=%0x, WWNN=%lx, WWPN=%lx",
						target->targetPortID, target->nodeName, target->portName);
					DoPortLogout(target->Sess, VMK_TRUE);
				}
				return;
			}
		} else {
			/* Handle IO timeout */
			qfle3f_log(hba, LOG_IOERR, "IO with xid = 0x%x timed out."
				    " issue ABTS", ioRequest->xid);
			if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
						QFLE3F_FLAG_IO_COMPL)) {
				qfle3f_log(hba, LOG_IOERR, "IO with xid = 0x%x "
					   "completed just before timer expiry",
					   ioRequest->xid);
				goto done;
			}

			/* If flush is in progress, don't do anything.
			 * flushActiveIO's would clean this IO.
			 */
			if(target->flushInProgress) {
				qfle3f_log(hba, LOG_IOERR, "Flush is progress. Do Nothing.");
				goto done;
			}

			if (!vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
							QFLE3F_FLAG_ISSUE_ABTS)) {
				VMK_ReturnStatus status = qfle3f_initiateABTS(ioRequest);
				if (status == VMK_OK)
					goto done;

				/*
				 * Explicitly logo the target if
				 * abts initiation fails
				 */
				logo_issued = vmk_BitVectorAtomicTestAndSet(
						target->flags,
						QFLE3F_FLAG_EXPL_LOGO);
				ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
				vmk_SpinlockUnlock(target->targetLock);

				if (!logo_issued) {
					if(!vmk_BitVectorAtomicTestAndClear(target->flags, QFLE3F_FLAG_SESSION_READY)) {
						vmk_BitVectorClear(target->flags, QFLE3F_FLAG_EXPL_LOGO);
						qfle3f_log(hba, LOG_IOERR, "Session not ready. No need to do Explicit Logout.");
						return;
					}
					qfle3f_log(hba, LOG_IOERR, "Explicit logo"
						   "(scsi) xid - 0x%x.",
						   ioRequest->xid);

					qfle3f_notice(hba, "DoPortLogout: targetPortID=%0x, WWNN=%lx, WWPN=%lx",
						target->targetPortID, target->nodeName, target->portName);
					DoPortLogout(target->Sess, VMK_TRUE);
				}
				return;
			} else {
				qfle3f_log(hba, LOG_IOERR, "IO with xid = 0x%x"
				   "already in ABTS processing", ioRequest->xid);
			}
		}
		break;
	case QFLE3F_ELS:

		if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_ABTS)) {
			qfle3f_log(hba, LOG_IOERR, "ABTS for ELS with oxid = 0x%x "
					"timed out", ioRequest->xid);

			/* If flush is in progress, don't do anything.
			 * flushActiveIO's would clean this IO.
			 */
			if(target->flushInProgress) {
				qfle3f_log(hba, LOG_IOERR, "Flush is progress. Do Nothing");
				goto done;
			}

			if (!vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
							QFLE3F_FLAG_ABTS_DONE)) {
				logo_issued = vmk_BitVectorAtomicTestAndSet(
						target->flags,
						QFLE3F_FLAG_EXPL_LOGO);
				ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
				vmk_SpinlockUnlock(target->targetLock);

				/* Explicitly logo the target */
				if (!logo_issued) {
					if(!vmk_BitVectorAtomicTestAndClear(target->flags, QFLE3F_FLAG_SESSION_READY)) {
						vmk_BitVectorClear(target->flags, QFLE3F_FLAG_EXPL_LOGO);
						qfle3f_log(hba, LOG_IOERR, "Returning");
						return;
					}
					qfle3f_log(hba, LOG_IOERR, "Explicitly logo"
						   "(els) 0x%x", ioRequest->xid);
					DoPortLogout(target->Sess, VMK_TRUE);
				}
				return;
			}
		} else
		/*
		 * For a LOGO Request timing out this is what the sequence of event would look like.
		 *
		 * 1. LOGO Request send out, ql_fcoe_mgmt has a 2 sec timer set and qfle3f has a r_a_tov(typicallly 10) timer set.
		 * 2. After 2 sec, ql_fcoe_mgmt timer kicks in and we start the process of onloading the session.
		 * 3. As part of onloading the session, in qfle3f_flushActiveIOs, we would cancel the qfle3f timer(qfle3f_commandTimeout)
		 *    and initiate cleanup for this LOGO Request.
		 */
		{
			/*
			 * Handle ELS timeout.
			 * targetLock is used to sync compl path and timeout
			 * path. If els compl path is processing this IO, we
			 * have nothing to do here, just release the timer hold
			 */
			qfle3f_log(hba, LOG_IOERR, "ELS with oxid = 0x%x "
					"timed out", ioRequest->xid);
			if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
							QFLE3F_FLAG_ELS_DONE))
				goto done;

			/* Indicate the cb_func that this ELS is timed out */
			vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_ELS_TIMEOUT);

			if ((ioRequest->cb_func) && (ioRequest->cb_arg)) {
				ioRequest->cb_func(ioRequest->cb_arg);
				ioRequest->cb_arg = NULL;
			}
		}
		break;
	default:
		qfle3f_err(hba, "cmd_timeout: invalid cmd_type %d", cmd_type);
		break;
	}

done:
	/* release the reference that was held when timer was set */
	ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest); /* ID: 002 */
	vmk_SpinlockUnlock(target->targetLock);
}

static void qfle3f_scsiDone(struct qfle3fCommand *ioRequest, int host_status,
						int device_status)
{
	/* Called with host lock held */
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
	struct qfle3fHBA *hba   = ioRequest->vhba;

	/*
	 * active_cmd may have other command types as well,
	 * and during flush operation,  we want to error back only
	 * scsi commands.
	 */
	if (ioRequest->cmd_type != QFLE3F_SCSI_CMD)
		return;

	qfle3f_log(hba, LOG_IOERR, "scsi_done for xid = 0x%x host_status = 0x%x",
		   ioRequest->xid, host_status);
	qfle3f_unmapSGList(ioRequest);
	ioRequest->sc_cmd = NULL;
	if (!sc_cmd) {
		qfle3f_err(hba, "scsi_done - sc_cmd NULL. "
			   "IO(0x%x) already cleaned up", ioRequest->xid);
		return;
	}
	sc_cmd->status.host = host_status;
	sc_cmd->status.device = device_status;

	qfle3f_log(hba, LOG_IOERR, "sc=%p, host_status=0x%x device_status=0x%x",
		sc_cmd, sc_cmd->status.host, sc_cmd->status.device);

	if (sc_cmd->done) {
		vmk_ScsiSchedCommandCompletion(sc_cmd);
	}
	else {
		qfle3f_err(hba, "ERROR! scsi_done is NULL");
	}

}

struct qfle3fCommandManager *qfle3f_commandManagerAlloc(struct qfle3fHBA *hba,
						vmk_uint16 min_xid, vmk_uint16 max_xid)
{
	struct qfle3fCommandManager *cmgr;
	struct io_bdt *bdt_info;
	struct qfle3fCommand *ioRequest;
	vmk_uint64 len;
	vmk_uint32 mem_size;
	vmk_uint16 xid;
	int i;
	int num_ios;
	vmk_uint64 bd_tbl_sz;
	vmk_Name name;

	if (max_xid <= min_xid || max_xid == INVALID_EXCHANGE_IDENTIFIER) {
		qfle3f_err(hba, "commandManager_alloc: Invalid min_xid 0x%x "
			   "and max_xid 0x%x", min_xid, max_xid);
		return NULL;
	}
	qfle3f_log(hba, LOG_INIT, "xid: min 0x%x, max = 0x%x", min_xid, max_xid);

	num_ios = max_xid - min_xid + 1;
	len = (num_ios * (sizeof(struct qfle3fCommand *)));
	len += sizeof(struct qfle3fCommandManager);

	cmgr = qfle3f_alloc(len);
	if (!cmgr) {
		qfle3f_err(hba, "failed to alloc cmgr");
		return NULL;
	}

	cmgr->hba = hba;
	cmgr->cmds = (struct qfle3fCommand **)(cmgr + 1);
	vmk_ListInit(&cmgr->freeList);

	/* Pre-allocated pool of qfle3fCommands */
	mem_size = num_ios * sizeof(struct qfle3fCommand);
	cmgr->qfle3fCommandPool = qfle3f_alloc(mem_size);
	if (!cmgr->qfle3fCommandPool) {
		qfle3f_err(hba, "failed to alloc cmd pool");
		qfle3f_free(cmgr);
		return NULL;
	}

	xid = QFLE3F_MIN_XID;
	ioRequest = (struct qfle3fCommand *)cmgr->qfle3fCommandPool;
	for (i = 0; i < num_ios; i++) {
		vmk_ListInitElement(&ioRequest->link);
		//INIT_DELAYED_WORK(&ioRequest->timeout_work, qfle3f_commandTimeout);

		ioRequest->xid = xid++;
		if (ioRequest->xid >= QFLE3F_MAX_OUTSTANDING_CMNDS)
			qfle3f_err(hba, "ERROR allocating xids - 0x%x",
				   ioRequest->xid);
		vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&cmgr->freeList));
        ioRequest->vhba = hba;

        ioRequest->req_flags = vmk_BitVectorAlloc(qfle3fDriverInfo.moduleHeapID,
                                    QFLE3F_FLAG_REQ_FLAGS_MAX);
        if(!ioRequest->req_flags)
            qfle3f_err(hba, "Failed to allocate req_flags");

		ioRequest++;
	}

	/* Allocate pool of io_bdts - one for each qfle3fCommand */
	mem_size = num_ios * sizeof(struct io_bdt *);
	cmgr->io_bdt_pool = qfle3f_alloc(mem_size);
	if (!cmgr->io_bdt_pool) {
		qfle3f_err(hba, "failed to alloc io_bdt_pool");
		goto mem_err;
	}

	mem_size = sizeof(struct io_bdt);
	for (i=0; i < num_ios; i++) {
		cmgr->io_bdt_pool[i] = qfle3f_alloc(mem_size);
		if (!cmgr->io_bdt_pool[i]) {
			qfle3f_err(hba, "failed to alloc io_bdt_pool[%d]", i);
			while (--i >= 0) {
				qfle3f_free(cmgr->io_bdt_pool[i]);
				cmgr->io_bdt_pool[i] = NULL;
			}
			goto mem_err;
		}
	}

	/* Allocate an map fcoe_bdt_ctx structures */
	bd_tbl_sz = QFLE3F_MAX_BDS_PER_CMD * sizeof(struct fcoe_bd_ctx);
	for (i = 0; i < num_ios; i++) {
		bdt_info = cmgr->io_bdt_pool[i];
		bdt_info->bd_tbl = qfle3f_dma_alloc(hba,
						      &bdt_info->bd_tbl_dma,
						      bd_tbl_sz);
		if (!bdt_info->bd_tbl) {
			qfle3f_err(hba, "failed to alloc bdt_tbl[%d]", i);
			goto mem_err;
		}
	}

	vmk_NameFormat(&name, "commandManagerLock-%u", hba->instance);
	qfle3f_vmk_spin_lock_init(&cmgr->commandManagerLock, LOCK_RANK_HIGHEST,
                                (const char *)&name);

	return cmgr;

mem_err:
	qfle3f_commandManagerFree(cmgr);
	return NULL;
}

void qfle3f_commandManagerFree(struct qfle3fCommandManager *cmgr)
{
	struct io_bdt *bdt_info;
	struct qfle3fHBA *hba = cmgr->hba;
	vmk_uint64 bd_tbl_sz;
	vmk_uint16 min_xid = QFLE3F_MIN_XID;
	vmk_uint16 max_xid = QFLE3F_MAX_XID;
	struct qfle3fCommand *ioRequest;
	int num_ios;
	int i;

	num_ios = max_xid - min_xid + 1;

	/* Free fcoe_bdt_ctx structures */
	if (!cmgr->io_bdt_pool)
		goto free_cmd_pool;

	bd_tbl_sz = QFLE3F_MAX_BDS_PER_CMD * sizeof(struct fcoe_bd_ctx);
	for (i = 0; i < num_ios; i++) {
		bdt_info = cmgr->io_bdt_pool[i];
		if (bdt_info->bd_tbl) {
			qfle3f_dma_free(hba, &bdt_info->bd_tbl_dma);
			bdt_info->bd_tbl = NULL;
		}
	}

	ioRequest = (struct qfle3fCommand *)cmgr->qfle3fCommandPool;
	for (i = 0; i < num_ios; i++) {
		vmk_BitVectorFree(qfle3fDriverInfo.moduleHeapID, ioRequest->req_flags);
		ioRequest++;
	}

	/* Destroy io_bdt pool */
	for (i = 0; i < num_ios; i++) {
		if (cmgr->io_bdt_pool[i]) {
			qfle3f_free(cmgr->io_bdt_pool[i]);
			cmgr->io_bdt_pool[i] = NULL;
		}
	}

	if (cmgr->io_bdt_pool) {
		qfle3f_free(cmgr->io_bdt_pool);
		cmgr->io_bdt_pool = NULL;
	}

free_cmd_pool:
	/* Destroy cmd pool */
	if (cmgr->qfle3fCommandPool) {
		qfle3f_free(cmgr->qfle3fCommandPool);
		cmgr->qfle3fCommandPool = NULL;
	}

    /* Dstroying the cmgr lock */
    vmk_SpinlockDestroy(cmgr->commandManagerLock);

	/* Free command manager itself */
	qfle3f_free(cmgr);
}

struct qfle3fCommand *qfle3f_elstmAlloc(struct qfle3f_rport *target, int type)
{
	//struct qfle3f_port *port = target->port;
	struct qfle3fHBA *hba = target->hba;
	struct qfle3fCommandManager *commandManager = hba->commandManager;
	struct qfle3fCommand *ioRequest;
	struct io_bdt *bd_tbl;
	vmk_uint64 freeSendQueues;
	vmk_uint32 maxSendQueues;
	vmk_uint16 xid;

	maxSendQueues = target->maxSendQueues;
	switch (type) {
	case QFLE3F_TASK_MGMT_CMD:
		maxSendQueues = QFLE3F_TM_MAX_SQES;
		break;
	case QFLE3F_ELS:
		maxSendQueues = QFLE3F_ELS_MAX_SQES;
		break;
	default:
		break;
	}

	/*
	 * NOTE: Free list insertions and deletions are protected with
	 * cmgr lock
	 */
	vmk_SpinlockLock(commandManager->commandManagerLock);
	freeSendQueues = vmk_AtomicRead64(&target->freeSendQueues);
	if ((vmk_ListIsEmpty(&commandManager->freeList)) ||
	    (vmk_AtomicRead64(&target->num_active_ios) >= maxSendQueues) ||
	    (freeSendQueues + maxSendQueues <= QFLE3F_SQ_WQES_MAX)) {
		qfle3f_log(hba, LOG_IOERR, "No free els_tm cmds available "
			"ios(%ld):sqes(%d)",
			vmk_AtomicRead64(&target->num_active_ios), target->maxSendQueues);
		if (vmk_ListIsEmpty(&commandManager->freeList))
			qfle3f_err(hba, "elstm_alloc: list_empty");
		vmk_SpinlockUnlock(commandManager->commandManagerLock);
		return NULL;
	}

	ioRequest = VMK_LIST_ENTRY(vmk_ListFirst(&commandManager->freeList),
								struct qfle3fCommand, link);
	vmk_ListRemove(&ioRequest->link);
	xid = ioRequest->xid;
	if (xid > QFLE3F_MAX_OUTSTANDING_CMNDS - 1) {
		qfle3f_err(hba, "ERROR in elstm_alloc xid = 0x%x", xid);
	}
	commandManager->cmds[xid] = ioRequest;
	vmk_AtomicReadInc64(&target->num_active_ios);
	vmk_AtomicReadDec64(&target->freeSendQueues);
	vmk_SpinlockUnlock(commandManager->commandManagerLock);

    vmk_ListInitElement(&ioRequest->link);

	ioRequest->target = target;
    ioRequest->vhba = hba;
	ioRequest->commandManager = commandManager;
    vmk_BitVectorZap(ioRequest->req_flags);
	ioRequest->cmd_type = type;

	/* Bind io_bdt for this ioRequest */
	/* Have a static link between ioRequest and io_bdt_pool */
	bd_tbl = ioRequest->bd_tbl = commandManager->io_bdt_pool[xid];
	bd_tbl->ioRequest = ioRequest;

	/* Hold the ioRequest  against deletion */
	ql_vmk_ref_init(&ioRequest->refcount);
	return ioRequest;
}
struct qfle3fCommand *qfle3f_commandAlloc(struct qfle3f_rport *target)
{
	struct qfle3fHBA *hba = target->hba;
	struct qfle3fCommandManager *commandManager = hba->commandManager;
	struct qfle3fCommand *ioRequest;
	struct io_bdt *bd_tbl;
	vmk_uint64 freeSendQueues;
	vmk_uint32 maxSendQueues;
	vmk_uint16 xid;

	maxSendQueues = QFLE3F_SCSI_MAX_SQES;
	/*
	 * NOTE: Free list insertions and deletions are protected with
	 * cmgr lock
	 */
	vmk_SpinlockLock(commandManager->commandManagerLock);
	freeSendQueues = vmk_AtomicRead64(&target->freeSendQueues);
	if ((vmk_ListIsEmpty(&commandManager->freeList) == VMK_TRUE) ||
	    (vmk_AtomicRead64(&target->num_active_ios) >= maxSendQueues) ||
	    (freeSendQueues + maxSendQueues <= QFLE3F_SQ_WQES_MAX)) {
		vmk_SpinlockUnlock(commandManager->commandManagerLock);
		return NULL;
	}

	//listp = (vmk_ListLinks *) commandManager->freeList.next;
	//list_del_init(listp);
	//ioRequest = (struct qfle3fCommand *) listp;

	ioRequest = VMK_LIST_ENTRY(vmk_ListFirst(&commandManager->freeList),
								struct qfle3fCommand, link);
	vmk_ListRemove(&ioRequest->link);

	xid = ioRequest->xid;
	commandManager->cmds[xid] = ioRequest;
	vmk_AtomicReadInc64(&target->num_active_ios);
	vmk_AtomicReadDec64(&target->freeSendQueues);
	vmk_SpinlockUnlock(commandManager->commandManagerLock);

	vmk_ListInitElement(&ioRequest->link);
	ql_vmk_init_singlethread_delayed_workitem(&ioRequest->timeout_work);

	ioRequest->vhba = hba;
	ioRequest->target = target;
	ioRequest->commandManager = commandManager;
    vmk_BitVectorZap(ioRequest->req_flags);

	/* Bind io_bdt for this ioRequest */
	/* Have a static link between ioRequest and io_bdt_pool */
	bd_tbl = ioRequest->bd_tbl = commandManager->io_bdt_pool[xid];
	bd_tbl->ioRequest = ioRequest;

	/* Hold the ioRequest  against deletion */
	ql_vmk_ref_init(&ioRequest->refcount);	/* ID: 001 */
	return ioRequest;
}

void qfle3f_commandRelease(void *arg)
{
	struct qfle3fCommand *ioRequest = (struct qfle3fCommand *) arg;
	struct qfle3fCommandManager *commandManager = ioRequest->commandManager;
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba = target->hba;
	vmk_uint16 xid = ioRequest->xid;

#ifdef IOERROR_DEBUGGING
	if(ioRequest->printIORelease == VMK_TRUE)
	qfle3f_err(hba, "Releasing cmd: %p:0x%x", ioRequest, xid);
#endif

    qfle3f_log(hba, LOG_INFO, "Releasing cmd: 0x%x", xid);

	vmk_SpinlockLock(commandManager->commandManagerLock);
	if (ioRequest->cmd_type != QFLE3F_SCSI_CMD)
		qfle3f_freeMPResources(ioRequest);

	commandManager->cmds[xid] = NULL;
	/* Delete IO from retire queue */
    if(!vmk_ListIsUnlinkedElement(&ioRequest->link))
	    vmk_ListRemove(&ioRequest->link);
	/* Add it to the free list */
	vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&commandManager->freeList));
	if(vmk_AtomicReadDec64(&target->num_active_ios) == 0)
		qfle3f_err(hba, "ERROR! active_ios < 0");
	vmk_SpinlockUnlock(commandManager->commandManagerLock);
#ifdef IOERROR_DEBUGGING
	ioRequest->printIORelease = VMK_FALSE;
#endif

}

static void qfle3f_freeMPResources(struct qfle3fCommand *ioRequest)
{
	struct qfle3f_mp_req *mp_req = &(ioRequest->mp_req);
	struct qfle3fHBA *hba = ioRequest->vhba;

	/* clear tm flags */
	mp_req->tm_flags = 0;
	if (mp_req->mp_req_bd) {
		qfle3f_dma_free(hba,
				     &mp_req->mp_req_bd_dma);
		mp_req->mp_req_bd = NULL;
	}
	if (mp_req->mp_resp_bd) {
		qfle3f_dma_free(hba,
				     &mp_req->mp_resp_bd_dma);
		mp_req->mp_resp_bd = NULL;
	}
	if (mp_req->req_buf) {
		qfle3f_dma_free(hba,
				     &mp_req->req_buf_dma);
		mp_req->req_buf = NULL;
	}
	if (mp_req->resp_buf) {
		qfle3f_dma_free(hba,
				     &mp_req->resp_buf_dma);
		mp_req->resp_buf = NULL;
	}
}

VMK_ReturnStatus qfle3f_initMPRequest(struct qfle3fCommand *ioRequest)
{
	struct qfle3f_mp_req *mp_req;
	struct fcoe_bd_ctx *mp_req_bd;
	struct fcoe_bd_ctx *mp_resp_bd;
	struct qfle3fHBA *hba = ioRequest->vhba;
	vmk_IOA addr;
	vmk_uint64 sz;

	qfle3f_log(hba, LOG_IOERR, "Entered qfle3f_initMPRequest");

	mp_req = (struct qfle3f_mp_req *)&(ioRequest->mp_req);
	vmk_Memset(mp_req, 0, sizeof(struct qfle3f_mp_req));

	if (ioRequest->cmd_type != QFLE3F_ELS) {
		mp_req->req_len = sizeof(FCP_CMND_PAYLOAD);
		ioRequest->data_xfer_len = mp_req->req_len;
	} else
		 mp_req->req_len = ioRequest->data_xfer_len;

	mp_req->req_buf = qfle3f_dma_alloc(hba,
					       &mp_req->req_buf_dma,
							VMK_PAGE_SIZE);
	if (!mp_req->req_buf) {
		qfle3f_err(hba, "unable to alloc MP req buffer");
		qfle3f_freeMPResources(ioRequest);
		return VMK_FAILURE;
	}

	mp_req->resp_buf = qfle3f_dma_alloc(hba,
					       &mp_req->resp_buf_dma,
							VMK_PAGE_SIZE);
	if (!mp_req->resp_buf) {
		qfle3f_err(hba, "unable to alloc TM resp buffer");
		qfle3f_freeMPResources(ioRequest);
		return VMK_FAILURE;
	}
	vmk_Memset(mp_req->req_buf, 0, VMK_PAGE_SIZE);
	vmk_Memset(mp_req->resp_buf, 0, VMK_PAGE_SIZE);

	/* Allocate and map mp_req_bd and mp_resp_bd */
	sz = sizeof(struct fcoe_bd_ctx);
	mp_req->mp_req_bd = qfle3f_dma_alloc(hba,
						 &mp_req->mp_req_bd_dma,
						sz);
	if (!mp_req->mp_req_bd) {
		qfle3f_err(hba, "unable to alloc MP req bd");
		qfle3f_freeMPResources(ioRequest);
		return VMK_FAILURE;
	}

	mp_req->mp_resp_bd = qfle3f_dma_alloc(hba,
						 &mp_req->mp_resp_bd_dma, sz);
	if (!mp_req->mp_resp_bd) {
		qfle3f_err(hba, "unable to alloc MP resp bd");
		qfle3f_freeMPResources(ioRequest);
		return VMK_FAILURE;
	}
	/* Fill bd table */
	addr = GET_DMA_ADDR(mp_req->req_buf_dma);
	mp_req_bd = mp_req->mp_req_bd;
	mp_req_bd->buf_addr_lo = U64_LO(addr);
	mp_req_bd->buf_addr_hi = U64_HI(addr);
	mp_req_bd->buf_len = VMK_PAGE_SIZE;
	mp_req_bd->flags = 0;

	/*
	 * MP buffer is either a task mgmt command or an ELS.
	 * So the assumption is that it consumes a single bd
	 * entry in the bd table
	 */
	mp_resp_bd = mp_req->mp_resp_bd;
	addr = GET_DMA_ADDR(mp_req->resp_buf_dma);
	mp_resp_bd->buf_addr_lo = U64_LO(addr);
	mp_resp_bd->buf_addr_hi = U64_HI(addr);
	mp_resp_bd->buf_len = VMK_PAGE_SIZE;
	mp_resp_bd->flags = 0;

	return VMK_OK;
}

static int qfle3f_executeTMF(struct qfle3fHBA *hba,
								struct qfle3fFCLun *fclun,
								vmk_ScsiCommand *sc_cmd,
								vmk_uint8 tm_flags)
{
	struct qfle3fCommand *ioRequest;
	struct qfle3f_rport *target = fclun->target;
	struct qfle3f_mp_req *tm_req;
	struct fcoe_task_ctx_entry *task;
	struct fcoe_task_ctx_entry *task_page;
	FC_FRAME_HEADER *fc_hdr;
	FCP_CMND_PAYLOAD *fcp_cmnd;
	int task_idx, index;
	int rc = SUCCESS;
	vmk_uint16 xid;
	vmk_uint32 initiatorPortID, did;
	unsigned long start = vmk_GetTimerCycles();
	unsigned long currentTime;
	VMK_ReturnStatus status = VMK_OK;

	if (!target) {
		qfle3f_log(hba, LOG_IOERR, "execute_tmf invalid arg");
		return FAILED;
	}

	if (!(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY))) {
		qfle3f_err(hba, "execute_tmf: target not offloaded");
		rc = FAILED;
		goto reset_tmf_err;
	}

	qfle3f_log(hba, LOG_IOERR, "execute_tmf: conn:%d tm_flags = %d",
			target->vmk_targetID, tm_flags);

retry_tmf:
	ioRequest = qfle3f_elstmAlloc(target, QFLE3F_TASK_MGMT_CMD);
	if (!ioRequest) {
		currentTime = vmk_GetTimerCycles();
		if (currentTime > (start + vmk_TimerCyclesPerSecond())) { 
			qfle3f_err(hba, "tmf: Failed TMF");
			rc = FAILED;
			goto reset_tmf_err;
		}
		//msleep(20);
		vmk_WorldSleep(20 * VMK_USEC_PER_MSEC);
		goto retry_tmf;
	}
	/* Initialize rest of ioRequest fields */
	ioRequest->sc_cmd = sc_cmd;
    ioRequest->fclun = fclun;

	tm_req = (struct qfle3f_mp_req *)&(ioRequest->mp_req);

	status = qfle3f_initMPRequest(ioRequest);
	if (status != VMK_OK) {
		qfle3f_err(hba, "Task mgmt MP request init failed");
        rc = FAILED;
		vmk_SpinlockLock(target->targetLock);
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
		vmk_SpinlockUnlock(target->targetLock);
		goto reset_tmf_err;
	}

	/* Set TM flags */
	ioRequest->ioRequest_flags = 0;
	tm_req->tm_flags = tm_flags;

	/* Fill FCP_CMND */
	qfle3f_buildFCPCommand(ioRequest, (FCP_CMND_PAYLOAD *)tm_req->req_buf);
	fcp_cmnd = (FCP_CMND_PAYLOAD *)tm_req->req_buf;
	vmk_Memset(fcp_cmnd->Cdb, 0,  sc_cmd->cdbLen);

    vmk_uint32 data_xfer_len = 0;
	vmk_Memcpy(fcp_cmnd->DataLength, &data_xfer_len, 4);

	/* Fill FC header */
	fc_hdr = &(tm_req->req_fc_hdr);
	initiatorPortID = target->initiatorPortID;
	did = target->targetPortID;
	qfle3f_fill_fc_hdr(fc_hdr, FC_R_CTL_FC4_DEVICE_DATA_UNSOLICITED_COMMAND,
				initiatorPortID, did,
				FC_TYPE_FCP, FC_F_CTL0_FirstSequence | FC_F_CTL0_EndSequence |
				FC_F_CTL0_SequenceInitiativeTransfer, 0);
	/* Obtain exchange id */
	xid = ioRequest->xid;

	qfle3f_log(hba, LOG_IOERR, "TMF ioRequest xid = 0x%x", xid);
	task_idx = xid/QFLE3F_TASKS_PER_PAGE;
	index = xid % QFLE3F_TASKS_PER_PAGE;

	/* Initialize task context for this IO request */
	task_page = (struct fcoe_task_ctx_entry *) hba->taskContext[task_idx];
	task = &(task_page[index]);
	qfle3f_initializeMPTask(ioRequest, task);

	//sc_cmd->SCp.ptr = (char *)ioRequest;

	/* Obtain free SQ entry */
	vmk_SpinlockLock(target->targetLock);
	qfle3f_addToSQ(target, xid);

	/* Enqueue the ioRequest to activeTaskManagementQueue */
	ioRequest->on_tmf_queue = 1;
	vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&target->activeTaskManagementQueue));

	ioRequest->wait_for_comp = 1;
	vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_FALSE);

	/* Ring doorbell */
	qfle3f_ringDoorbell(target);

	qfle3f_notice(hba, "waiting for tm_done = %p", &(ioRequest->tm_done));
	status = qfle3f_sleepWithCondition(hba, ioRequest,
								(vmk_WorldEventID)&ioRequest->tm_done,
								target->targetLock, QFLE3F_TM_TIMEOUT,
								"Waiting from tm_done-1",
								checkIoTmDoneCalled);
	if(status != VMK_OK) {
		qfle3f_log(hba, LOG_IOERR, "TMF: returned from tm_done wait:%s",
					vmk_StatusToString(status));
		vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
	}

	ioRequest->wait_for_comp = 0;

	if (!(vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_TM_COMPL))) {
		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_TM_TIMEOUT);
		if (ioRequest->on_tmf_queue) {
			vmk_ListRemove(&ioRequest->link);
			ioRequest->on_tmf_queue = 0;
		}
		ioRequest->wait_for_comp = 1;
		vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_FALSE);

		qfle3f_initiateCleanup(ioRequest);
		qfle3f_notice(hba, "waiting for tm_done = %p", &(ioRequest->tm_done));
		status = qfle3f_sleepWithCondition(hba, ioRequest,
								(vmk_WorldEventID)&ioRequest->tm_done,
								target->targetLock, QFLE3F_TM_TIMEOUT,
								"Waiting from tm_done-2",
								checkIoTmDoneCalled);
		if(status != VMK_OK) {
			qfle3f_log(hba, LOG_IOERR, "TMF: returned from tm_done wait:%s",
						vmk_StatusToString(status));
			vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
		}

		ioRequest->wait_for_comp = 0;
		if (status != VMK_OK) {
			/* TODO:
			qfle3f_scsiDone(ioRequest, VMK_SCSI_HOST_ERROR, 0);
			*/
			ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
		}
	}
	vmk_SpinlockUnlock(target->targetLock);

	if (status != VMK_OK) {
		qfle3f_err(hba, "task mgmt command failed...");
		rc = FAILED;
	} else {
		qfle3f_err(hba, "task mgmt command success...");
		rc = SUCCESS;
	}
reset_tmf_err:
	return rc;
}

static int qfle3f_initiateTMF(struct qfle3fHBA *hba,
								struct qfle3fFCLun *fclun,
								vmk_ScsiCommand *sc_cmd,
								vmk_uint8 tm_flags)
{
	int rc = SUCCESS;

	rc = qfle3f_executeTMF(hba, fclun, sc_cmd, tm_flags);
	return rc;
}

/* called with targetLock held */
VMK_ReturnStatus qfle3f_initiateABTS(struct qfle3fCommand *ioRequest)
{
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fFCLun *fclun = ioRequest->fclun;
	struct qfle3fHBA *hba;
	struct qfle3fCommand *abts_ioRequest;
	struct fcoe_task_ctx_entry *task;
	struct fcoe_task_ctx_entry *task_page;
	FC_FRAME_HEADER *fc_hdr;
	struct qfle3f_mp_req *abts_req;
	int task_idx, index;
	vmk_uint32 initiatorPortID, did;
	vmk_uint16 xid;
	VMK_ReturnStatus status = VMK_OK;
	vmk_uint32 r_a_tov = 0;

	hba = target->hba;

	if(qfle3f_enable_r_a_tov == 1)
		r_a_tov = qfle3f_r_a_tov*VMK_MSEC_PER_SEC;
	else
		r_a_tov = target->Sess->Fabric->r_a_tov;

#ifdef IOERROR_DEBUGGING
	ioRequest->printIORelease = VMK_TRUE;
#endif
	qfle3f_log(hba, LOG_IOERR, "Entered ox_id 0x%x: r_a_tov = %d",
			ioRequest->xid, r_a_tov);

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
		qfle3f_err(hba, "initiate_abts: target not offloaded");
		status = VMK_FAILURE;
		goto abts_err;
	}

	abts_ioRequest = qfle3f_elstmAlloc(target, QFLE3F_ABTS);
	if (!abts_ioRequest) {
		qfle3f_err(hba, "abts: couldnt allocate cmd");
		status = VMK_FAILURE;
		goto abts_err;
	}

	vmk_ListInsert(&abts_ioRequest->link, vmk_ListAtRear(&target->abtsQueue));
	/* Initialize rest of ioRequest fields */
	abts_ioRequest->sc_cmd = NULL;
	abts_ioRequest->vhba = hba;
	abts_ioRequest->target = target;
	abts_ioRequest->fclun = fclun;
	abts_ioRequest->data_xfer_len = 0; /* No data transfer for ABTS */

#ifdef IOERROR_DEBUGGING
	abts_ioRequest->printIORelease = VMK_TRUE;
#endif
	abts_req = (struct qfle3f_mp_req *)&(abts_ioRequest->mp_req);
	vmk_Memset(abts_req, 0, sizeof(struct qfle3f_mp_req));

	/* Fill FC header */
	fc_hdr = &(abts_req->req_fc_hdr);

    /* Obtain oxid and rxid for the original exchange to be aborted */
	//fc_hdr->ox_id = vmk_CPUToBE16(ioRequest->xid);
    fc_hdr->ox_id[0] = ioRequest->xid >> 8;
    fc_hdr->ox_id[1] = (ioRequest->xid & 0xFF);

	//fc_hdr->rx_id = vmk_CPUToBE16(ioRequest->task->rxwr_txrd.var_ctx.rx_id);
	fc_hdr->rx_id[0] = ioRequest->task->rxwr_txrd.var_ctx.rx_id >> 8;
    fc_hdr->rx_id[1] = (ioRequest->task->rxwr_txrd.var_ctx.rx_id & 0xFF);

	initiatorPortID = target->initiatorPortID;
	did = target->targetPortID;

	qfle3f_fill_fc_hdr(fc_hdr, FC_R_CTL_BASIC_LINK_SERVICE_ABTS,
				initiatorPortID, did,
			   FC_TYPE_BLS, FC_F_CTL0_FirstSequence | FC_F_CTL0_EndSequence |
			   FC_F_CTL0_SequenceInitiativeTransfer, 0);

	xid = abts_ioRequest->xid;
	qfle3f_log(hba, LOG_IOERR, "ABTS ioRequest xid = 0x%x", xid);
	task_idx = xid/QFLE3F_TASKS_PER_PAGE;
	index = xid % QFLE3F_TASKS_PER_PAGE;

	/* Initialize task context for this IO request */
	task_page = (struct fcoe_task_ctx_entry *) hba->taskContext[task_idx];
	task = &(task_page[index]);
	qfle3f_initializeMPTask(abts_ioRequest, task);

	/*
	 * ABTS task is a temporary task that will be cleaned up
	 * irrespective of ABTS response. We need to start the timer
	 * for the original exchange, as the CQE is posted for the original
	 * IO request.
	 *
	 * Timer for ABTS is started only when it is originated by a
	 * TM request. For the ABTS issued as part of ULP timeout,
	 * scsi-ml maintains the timers.
	 */

	/* if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_ABTS))*/
	qfle3f_commandTimerSet(ioRequest, 2 * r_a_tov);

	/* Obtain free SQ entry */
	qfle3f_addToSQ(target, xid);

	/* Ring doorbell */
	qfle3f_ringDoorbell(target);

abts_err:
	return status;
}


void qfle3f_cleanupTmoHandler(struct qfle3fCommand *cleanup_ioRequest)
{
	struct qfle3fCommand *ioRequest = NULL;
	struct qfle3f_rport *target = NULL;
	struct qfle3fHBA *hba = NULL;
	vmk_uint16 orig_xid = 0;

	qfle3f_err(hba, "Cleanup compl not rcvd!! cleanup-xid=0x%x ioreq=0x%x\n",
			cleanup_ioRequest->xid, cleanup_ioRequest->orig_xid);

	if (cleanup_ioRequest->target == NULL || cleanup_ioRequest->vhba == NULL) {
		qfle3f_err(hba, "ERROR: rport is NULL\n");
		return;
	}

	hba = cleanup_ioRequest->vhba;
	orig_xid = cleanup_ioRequest->orig_xid;
	target = cleanup_ioRequest->target;

	vmk_SpinlockLock(target->targetLock);
	if (cleanup_ioRequest->cleanup_tmr_active == CLEANUP_TIMER_ACTIVE) {
		cleanup_ioRequest->cleanup_tmr_active = CLEANUP_TIMER_INPROCESS;
	} else {
		qfle3f_err(hba, "cleanup req already in process/completed.");
		cleanup_ioRequest->cleanup_tmr_active = CLEANUP_TIMER_INACTIVE;
		/* Put cleanup_tmr reference. */
		ql_vmk_ref_put(&cleanup_ioRequest->refcount, qfle3f_commandRelease,
				cleanup_ioRequest);
		vmk_SpinlockUnlock(target->targetLock);
		return;
	}

	/* Put cleanup_tmr reference. */
	ql_vmk_ref_put(&cleanup_ioRequest->refcount, qfle3f_commandRelease,
			cleanup_ioRequest);
	cleanup_ioRequest->cleanup_tmr_active = CLEANUP_TIMER_INACTIVE;
	/* Put cleanup_ioreq reference. */
	ql_vmk_ref_put(&cleanup_ioRequest->refcount, qfle3f_commandRelease,
			cleanup_ioRequest);

	/* Process original ioreq scsi-cmd completion and cmd release. */
	ioRequest = (struct qfle3fCommand *)hba->commandManager->cmds[orig_xid];

	if (VMK_UNLIKELY(ioRequest == NULL)) {
		qfle3f_err(hba, "ERROR: cleanup_tmo - io_req is NULL\n");
	} else if (ioRequest->cmd_type == QFLE3F_SCSI_CMD) {
		qfle3f_err(hba, "cleanup_tmo - io_req = %p\n", ioRequest);
		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_IO_COMPL);
		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_IO_CLEANUP);
		qfle3f_processCleanupCompletion(ioRequest, ioRequest->task, 0);
	}
	vmk_SpinlockUnlock(target->targetLock);

	qfle3f_err(hba, "cleanup_tmo - done cleanup_ioreq->xid=%x, "
			"cleanup_ioreq->orig_xid=%x\n",
			cleanup_ioRequest->xid, cleanup_ioRequest->orig_xid);

	return;
}


int qfle3f_initiateCleanup(struct qfle3fCommand *ioRequest)
{
	struct qfle3f_rport *target = ioRequest->target;
    struct qfle3fFCLun *fclun = ioRequest->fclun;
	struct qfle3fHBA *hba = NULL;
	struct qfle3fCommand *cleanup_ioRequest;
	struct fcoe_task_ctx_entry *task;
	struct fcoe_task_ctx_entry *task_page;
	int task_idx, index;
	vmk_uint16 xid = 0;
	int rc = 0;

	/* ASSUMPTION: called with targetLock held */
	hba = ioRequest->vhba;

#ifdef IOERROR_DEBUGGING
	ioRequest->printIORelease = VMK_TRUE;
#endif
	qfle3f_log(hba, LOG_IOERR, "Entered xid = 0x%x", ioRequest->xid);

	cleanup_ioRequest = qfle3f_elstmAlloc(target, QFLE3F_CLEANUP);
	if (!cleanup_ioRequest) {
		qfle3f_err(hba, "cleanup: couldnt allocate cmd");
		rc = -1;
		goto cleanup_err;
	}

	/* Initialize rest of ioRequest fields */
	cleanup_ioRequest->sc_cmd = NULL;
	cleanup_ioRequest->target = target;
	cleanup_ioRequest->fclun = fclun;
    cleanup_ioRequest->vhba = hba;
	cleanup_ioRequest->data_xfer_len = 0; /* No data transfer for cleanup */

	qfle3f_log(hba, LOG_IOERR, "cleanup_tmr_active prev=0x%x now=0x%x\n",
			cleanup_ioRequest->cleanup_tmr_active, CLEANUP_TIMER_ACTIVE);
	cleanup_ioRequest->cleanup_tmr_active = CLEANUP_TIMER_ACTIVE;
	/* Get cleanup_tmr reference. */
	ql_vmk_ref_get(&cleanup_ioRequest->refcount);
	if (vmk_TimerIsPending(cleanup_ioRequest->cleanup_tmr) == VMK_FALSE) {
		qfle3f_log(hba, LOG_IOERR,
			"Scheduling cleanup-tmr (%d seconds) for cleanup tmo.",
			QFLE3F_CLEANUP_TMO);
		rc = vmk_TimerSchedule(qfle3fDriverInfo.timerQueue,
				(void *)qfle3f_cleanupTmoHandler,
				(void *) cleanup_ioRequest,
				(QFLE3F_CLEANUP_TMO) * VMK_USEC_PER_SEC,
				VMK_TIMER_DEFAULT_TOLERANCE,
				VMK_TIMER_ATTR_NONE,
				VMK_LOCKDOMAIN_INVALID,
				VMK_SPINLOCK_UNRANKED,
				&cleanup_ioRequest->cleanup_tmr);
	}

	xid = cleanup_ioRequest->xid;

	task_idx = xid/QFLE3F_TASKS_PER_PAGE;
	index = xid % QFLE3F_TASKS_PER_PAGE;

	/* Initialize task context for this IO request */
	task_page = (struct fcoe_task_ctx_entry *) hba->taskContext[task_idx];
	task = &(task_page[index]);
	cleanup_ioRequest->orig_xid = ioRequest->xid;

	qfle3f_log(hba, LOG_IOERR, "CLEANUP ioRequest xid = 0x%x orig_xid = 0x%x",
		xid, cleanup_ioRequest->orig_xid);

	qfle3f_initializeCleanupTask(cleanup_ioRequest, task,
			cleanup_ioRequest->orig_xid);

	/* Obtain free SQ entry */
	qfle3f_addToSQ(target, xid);

	/* Ring doorbell */
	qfle3f_ringDoorbell(target);

cleanup_err:
	return rc;
}

/**
 * Function Name : qfle3f_internalAbortIO
 *
 * DESCRIPTION:
 *  Abort a IO.
 *  This is called from the ehAbort and virtualReset
 *  contexts.
 *
 * PARAMETERS:
 *  hba:       	The host port.
 *  target:    	The associated target.
 *  ioRequest: 	The IO to abort.
 *  shouldExit:	This is a output. Because of error condition faced while doing
 *  			this particular Abort, do not call this function again(from
 *  			the current context). This is currently relevant in the
 *  			virtualReset context.
 *
 * RETURN:
 *  status:		The return status of the Abort request.
 *
 * NOTE:
 * Caller **must** be holding target->targetLock lock.
 * Caller **must** be holding a reference on the io.
 */
VMK_ReturnStatus qfle3f_internalAbortIO(struct qfle3fHBA *hba, struct qfle3f_rport *target,
									struct qfle3fCommand * ioRequest, vmk_Bool *shouldExit,
									vmk_Bool *pDroppedIoRef)
{
	int rc = SUCCESS;
	VMK_ReturnStatus status = VMK_OK;
	qfle3f_err(hba, "xid = 0x%x", ioRequest->xid);
	*shouldExit = VMK_FALSE;
	*pDroppedIoRef = VMK_FALSE;

	if (hba->cnic->uplinkLinkState.state == VMK_LINK_STATE_DOWN) {
	    qfle3f_log(hba, LOG_SCSI_TM, "LINK DOWN, not sending abort");
		*shouldExit = VMK_TRUE;
	    return VMK_OK;
	}

	/* Remove the ioRequest from the active_q. */
	/*
	 * Task Mgmt functions (LUN RESET & TGT RESET) will not
	 * issue an ABTS on this particular IO req, as the
	 * ioRequest is no longer in the active_q.
	 */
	if (target->flushInProgress) {
		qfle3f_log(hba, LOG_IOERR, "ioRequest (xid = 0x%x) "
			"flush in progress", ioRequest->xid);
		*shouldExit = VMK_TRUE;
		return VMK_OK;
	}

	qfle3f_log(hba, LOG_IOERR, "on active queue = %d", ioRequest->on_active_queue);

	if (ioRequest->on_active_queue == 0) {
		qfle3f_log(hba, LOG_IOERR, "ioRequest (xid = 0x%x) "
				"not on active_q", ioRequest->xid);
		/*
		 * The IO is still with the FW.
		 * Return failure and let scsi-ml retry the abort.
		 */
		*shouldExit = VMK_TRUE;
		return VMK_FAILURE;
	}

	/*
	 * Only eh_abort processing will remove the IO from
	 * active_cmd_q before processing the request. this is
	 * done to avoid race conditions between IOs aborted
	 * as part of task management completion and eh_abort
	 * processing
	 */
	vmk_ListRemove(&ioRequest->link);
	ioRequest->on_active_queue = 0;
	/* Move IO req to retire queue */
	vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&target->ioRetireQueue));

	if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_ABTS)) {
		qfle3f_err(hba, "ioRequest (xid = 0x%x) "
				"already in abts processing", ioRequest->xid);
		if(ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK)
			ql_vmk_ref_put(&ioRequest->refcount,
				 qfle3f_commandRelease, ioRequest); /* drop timer hold */
		rc = qfle3f_explicitLogout(ioRequest, pDroppedIoRef);
		/* reference is released, unlock and exit */
        if(rc == SUCCESS)
            status = VMK_OK;
		*shouldExit = VMK_TRUE;
		goto out;
	}

	qfle3f_log(hba, LOG_INIT, "Trying to cancel the current timer");
	/* Cancel the current timer running on this ioRequest */
	if(ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK)
		ql_vmk_ref_put(&ioRequest->refcount,
			 qfle3f_commandRelease, ioRequest); /* drop timer hold */

	vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_EH_ABORT);
	ioRequest->wait_for_comp = 1;
	vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_FALSE);

	status = qfle3f_initiateABTS(ioRequest);
	if (status != VMK_OK) {
		qfle3f_initiateCleanup(ioRequest);
		qfle3f_err(hba, "Waiting for tm_done = %p", &(ioRequest->tm_done));
		status = qfle3f_sleepWithCondition(hba, ioRequest,
							(vmk_WorldEventID)&ioRequest->tm_done,
							target->targetLock, VMK_TIMEOUT_UNLIMITED_MS,
							"Waiting from tm_done-6",
							checkIoTmDoneCalled);
		if(status != VMK_OK) {
			qfle3f_log(hba, LOG_IOERR, "TMF: returned from tm_done wait:%s",
					vmk_StatusToString(status));
			vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
		}

		ioRequest->wait_for_comp = 0;
		/* reference is released, unlock and exit */
		goto out;
	}

	qfle3f_err(hba, "Waiting for tm_done = %p", &(ioRequest->tm_done));
	status = qfle3f_sleepWithCondition(hba, ioRequest,
						(vmk_WorldEventID)&ioRequest->tm_done,
						target->targetLock, VMK_TIMEOUT_UNLIMITED_MS,
						"Waiting from tm_done-7",
						checkIoTmDoneCalled);
	if(status != VMK_OK) {
		qfle3f_log(hba, LOG_IOERR, "TMF: returned from tm_done wait:%s",
				vmk_StatusToString(status));
		vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
	}

	ioRequest->wait_for_comp = 0;
	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_IO_COMPL)) {
		qfle3f_err(hba, "IO completed from diff context, xid = 0x%x",
			   ioRequest->xid);
		hba->io_cmpl_abort_race++;
		status = VMK_OK;
	} else if (!(vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
					QFLE3F_FLAG_ABTS_DONE))) {
		/* Let the scsi-ml try to recover this command */
		qfle3f_err(hba, "abort failed, xid = 0x%x", ioRequest->xid);
		rc = qfle3f_explicitLogout(ioRequest, pDroppedIoRef);
        if(rc == SUCCESS)
            status = VMK_OK;
		*shouldExit = VMK_TRUE;
		goto out;
	} else {
		/*
		 * We come here even when there was a race condition
		 * between timeout and abts completion, and abts
		 * completion happens just in time.
		 */
		qfle3f_notice(hba, "abort succeeded");
		status = VMK_OK;
		qfle3f_scsiDone(ioRequest, VMK_SCSI_HOST_RESET, 0);
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
	}

out:
	return status;
}

/**
 * Function Name : qfle3f_getCmdSatisfyingACondition
 *
 * DESCRIPTION:
 *  Loop through all the commands in the active cmd list
 *  and return the first IO for which checkCondition()
 *  returns VMK_TRUE.
 *
 * PARAMETERS:
 *  fclun:          The LUN under consideration.
 *  taskMgmt:       TaskMgmt being performed.
 *  checkCondition: A function checking condition for each I/O.
 *
 * RETURN:
 *  The pointer to the found ioRequest or NULL.
 *
 * NOTE:
 * Caller **must** be holding target->targetLock lock.
 * We obtain a reference for the IORequest, it's caller responsibility
 * to release the reference.
 */
struct qfle3fCommand *qfle3f_getCmdSatisfyingACondition(struct qfle3fFCLun *fclun,
        struct vmk_ScsiTaskMgmt *taskMgmt,
        vmk_Bool (*checkCondition)(struct qfle3fCommand *, struct qfle3fFCLun *, struct vmk_ScsiTaskMgmt *))
{
    struct qfle3f_rport *target = fclun->target;
    struct qfle3fHBA *hba = target->hba;
    struct qfle3fCommand *ioRequest;
    vmk_ListLinks *cur_cmd, *next_cmd;

	VMK_LIST_FORALL_SAFE(&target->activeCommands, cur_cmd, next_cmd)
	{
		ioRequest = VMK_LIST_ENTRY(cur_cmd, struct qfle3fCommand, link);

		if (!ioRequest || !ioRequest->sc_cmd)
			continue;

        if(checkCondition(ioRequest, fclun, taskMgmt) == VMK_FALSE)
			continue;

		/* another eh thread working on this already possibly */
		if (ioRequest->on_active_queue == 0)
			continue;

		/* Hold IO request across abort processing */
		if (!ql_vmk_ref_get(&ioRequest->refcount))
			continue;

		qfle3f_notice(hba, "found the I/O to abort %p:0x%x", ioRequest, ioRequest->xid);
		return ioRequest;
	}

    qfle3f_err(hba, "No IO to Abort");
	return NULL;
}

vmk_Bool checkForMatchingTarget(struct qfle3fCommand *ioRequest,
                        struct qfle3fFCLun *fclun,
                        struct vmk_ScsiTaskMgmt *taskMgmt) {
        if(ioRequest->target == fclun->target) { return VMK_TRUE; }
        else { return VMK_FALSE; }
}

/**
 * qfle3f_eh_device_reset: Reset a single LUN
 * @sc_cmd:	SCSI command
 *
 * Set from SCSI host template to send task mgmt command to the target
 *	and wait for the response
 */
VMK_ReturnStatus qfle3f_deviceReset(struct qfle3fHBA *hba, struct qfle3fFCLun *fclun,
				struct vmk_ScsiTaskMgmt *taskMgmt)
{
	int rc;
    struct qfle3f_rport *target = fclun->target;
    vmk_uint8 tm_flags;
	VMK_ReturnStatus status = VMK_OK;
    vmk_ScsiCommand sc_cmd;

	qfle3f_log(hba, LOG_IOERR, "Entered");

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED)) {
		/*
		 * Session is not offloaded yet. Let SCSI-ml retry
		 * the command.
		 */
        qfle3f_err(hba, "Target(%d:%d)not offloaded.",
                    target->fcoe_conn_id, target->vmk_targetID);
		return VMK_FAILURE;
	}

    vmk_Memset(&sc_cmd, 0, sizeof(vmk_ScsiCommand));

    sc_cmd.cdbLen = VMK_SCSI_MAX_CDB_LEN;

    tm_flags = FCP_TMF_TGT_RESET;

    rc = qfle3f_initiateTMF(hba, fclun, &sc_cmd, tm_flags);
    if(rc != SUCCESS)
        status = VMK_FAILURE;


    qfle3f_err(hba, "Returning: %s", vmk_StatusToString(status));
	return status;
}

vmk_Bool checkForMatchingLUN(struct qfle3fCommand *ioRequest,
                        struct qfle3fFCLun *fclun,
                        struct vmk_ScsiTaskMgmt *taskMgmt) {
        if(ioRequest->fclun == fclun) { return VMK_TRUE; }
        else { return VMK_FALSE; }
}

/**
 * qfle3f_lunReset: Reset a single LUN
 * @sc_cmd:	SCSI command
 *
 * Set from SCSI host template to send task mgmt command to the target
 *	and wait for the response
 */
VMK_ReturnStatus qfle3f_lunReset(struct qfle3fHBA *hba, struct qfle3fFCLun *fclun,
				struct vmk_ScsiTaskMgmt *taskMgmt)
{
	int rc;
    struct qfle3f_rport *target = fclun->target;
    vmk_uint8 tm_flags;
	VMK_ReturnStatus status = VMK_OK;
    vmk_ScsiCommand sc_cmd;

	qfle3f_notice(hba, "lun: %d  target: %d",
                fclun->vmkLunId, fclun->vmkTargetId);

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_OFFLOADED)) {
		/*
		 * Session is not offloaded yet. Let SCSI-ml retry
		 * the command.
		 */
        qfle3f_err(hba, "Target(%d:%d)not offloaded.",
                    target->fcoe_conn_id, target->vmk_targetID);
		return VMK_FAILURE;
	}

    vmk_Memset(&sc_cmd, 0, sizeof(vmk_ScsiCommand));

    sc_cmd.cdbLen = VMK_SCSI_MAX_CDB_LEN;

    tm_flags = FCP_TMF_LUN_RESET;

    rc = qfle3f_initiateTMF(hba, fclun, &sc_cmd, tm_flags);
    if(rc != SUCCESS)
        status = VMK_FAILURE;

    qfle3f_err(hba, "Returning: %s", vmk_StatusToString(status));
	return status;
}

vmk_Bool checkForMatchingWorldID(struct qfle3fCommand *ioRequest,
                        struct qfle3fFCLun *fclun,
                        struct vmk_ScsiTaskMgmt *taskMgmt) {
        if(ioRequest->sc_cmd->worldId == taskMgmt->worldId) { return VMK_TRUE; }
        else { return VMK_FALSE; }
}


VMK_ReturnStatus qfle3f_virtualReset(struct qfle3fHBA *hba, struct qfle3fFCLun *fclun,
						struct vmk_ScsiTaskMgmt *taskMgmt)
{
	struct qfle3fCommand *ioRequest = NULL;
	struct qfle3f_rport *target = fclun->target;
	VMK_ReturnStatus status = VMK_OK;
	vmk_Bool shouldExit;
	vmk_Bool droppedIoRef;

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
		/*
		 * Session is not offloaded yet. Let SCSI-ml retry
		 * the command.
		 */
		return status;
	}

	qfle3f_notice(hba, "Abort SN = 0x%lx worldID = 0x%x",
            taskMgmt->cmdId.serialNumber, taskMgmt->worldId);
	vmk_SpinlockLock(target->targetLock);

	/* If this call successed, we would be holding a reference for the IO.
	 */
	while(1)
	{
		ioRequest = qfle3f_getCmdSatisfyingACondition(fclun, taskMgmt, checkForMatchingWorldID);
		if(!ioRequest)
			break;

		if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO)) {
			qfle3f_err(hba, "ioRequest (xid = 0x%x) "
					"already in expl logo processing", ioRequest->xid);
			ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
			break;
		}

		status = qfle3f_internalAbortIO(hba, target, ioRequest, &shouldExit, &droppedIoRef);

		/*
		 * Release the reference taken by the qfle3f_getCmdSatisfyingACondition function
		 * Exception: When an explicit logo is issued, the reference is dropped from
		 * within qfle3f_explicitLogout function.
		 *
		 * Note: As the targetLock is held, we need not worry about this getting
		 * set while we are in internalAbortIO
		 */
		if (!droppedIoRef) {
			ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
		}

		if(status != VMK_OK) {
			qfle3f_err(hba, "Returning with status: %s", vmk_StatusToString(status));
			break;
		}

		if(shouldExit == VMK_TRUE) {
			qfle3f_err(hba, "Returning with status: %s", vmk_StatusToString(status));
			break;
		}
	}

	vmk_SpinlockUnlock(target->targetLock);
	qfle3f_err(hba, "Returning: %s", vmk_StatusToString(status));
	return status;
}

VMK_ReturnStatus
qfle3f_scsiTaskMgmt(void *clientData, struct vmk_ScsiTaskMgmt *taskMgmt,
    void *deviceData)
{
    struct qfle3fHBA *hba = (struct qfle3fHBA *)clientData;
    struct qfle3fFCLun *fclun = (struct qfle3fFCLun *)deviceData;
    VMK_ReturnStatus status = VMK_FAILURE;

    VMK_ASSERT(clientData);
    VMK_ASSERT(deviceData);
    VMK_ASSERT(taskMgmt);

    qfle3f_log(hba, LOG_IOERR, "Enter %d:%d:%d",
                fclun->vmkChannel, fclun->vmkTargetId, fclun->vmkLunId);

    switch (taskMgmt->type) {
        case VMK_SCSI_TASKMGMT_ABORT:
            qfle3f_log(hba, LOG_IOERR, "Task Mgmt abort on serial num %lx",
                taskMgmt->cmdId.serialNumber);
            status = qfle3f_ehAbort(hba, fclun, taskMgmt);
            break;
        case VMK_SCSI_TASKMGMT_LUN_RESET:
            qfle3f_log(hba, LOG_IOERR, "Task Mgmt lun reset %d", fclun->vmkLunId);
            status = qfle3f_lunReset(hba, fclun, taskMgmt);
            break;
        case VMK_SCSI_TASKMGMT_DEVICE_RESET:
            qfle3f_log(hba, LOG_IOERR, "Task Mgmt device reset");
            status = qfle3f_deviceReset(hba, fclun, taskMgmt);
            break;
        case VMK_SCSI_TASKMGMT_BUS_RESET:
            qfle3f_log(hba, LOG_IOERR, "Task Mgmt bus reset");
            status = qfle3f_deviceReset(hba, fclun, taskMgmt);
            break;
        case VMK_SCSI_TASKMGMT_VIRT_RESET:
            qfle3f_log(hba, LOG_IOERR, "Task Mgmt virt reset");
            status = qfle3f_virtualReset(hba, fclun, taskMgmt);
            break;
        default:
            qfle3f_warning(hba, "Unknown task management type 0x%x",
                taskMgmt->type);
            return VMK_OK;
    }

    return status;
}


int qfle3f_explicitLogout(struct qfle3fCommand *ioRequest, vmk_Bool *pDroppedIoRef)
{
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba = target->hba;
	int logo_issued;
	int rc = SUCCESS;
	int wait_cnt = 0;
	VMK_ReturnStatus status = VMK_OK;

	qfle3f_log(hba, LOG_ERROR, "expl logo - target = 0x%x xid = 0x%x",
			target->vmk_targetID, ioRequest->xid);
	logo_issued = vmk_BitVectorAtomicTestAndSet(target->flags,
			QFLE3F_FLAG_EXPL_LOGO);

	vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_ABTS_OR_CLNUP_PEN);
	ioRequest->wait_for_comp = 1;
	vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_FALSE);

	qfle3f_initiateCleanup(ioRequest);

	qfle3f_log(hba, LOG_ERROR, "expl logo - logo_issued = 0x%x", logo_issued);
	qfle3f_notice(hba, "waiting for cleanup done = %p", &(ioRequest->tm_done));
	status = qfle3f_sleepWithCondition(hba, ioRequest,
			(vmk_WorldEventID)&ioRequest->tm_done,
			target->targetLock, 2,
			"Waiting from tm_done-5",
			checkIoTmDoneCalled);
	if(status != VMK_OK) {
		qfle3f_err(hba, "Cleanup timed out with status:%s",
				vmk_StatusToString(status));
		vmk_BitVectorClear(ioRequest->req_flags, QFLE3F_FLAG_ABTS_OR_CLNUP_PEN);
		vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
	}

	ioRequest->wait_for_comp = 0;
	/*
	 * release the reference taken in eh_abort
	 * to allow the target to relogin
	 */
	ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
	*pDroppedIoRef = VMK_TRUE;
	/* If flush is in progress, don't issue explicit Logout.
	 */
	if(target->flushInProgress) {
		qfle3f_log(hba, LOG_IOERR, "flush is progress. Do Nothing.");
		return rc;
	}

	vmk_SpinlockUnlock(target->targetLock);

	if (!logo_issued) {
		if(!vmk_BitVectorAtomicTestAndClear(target->flags, QFLE3F_FLAG_SESSION_READY)) {
			vmk_BitVectorClear(target->flags, QFLE3F_FLAG_EXPL_LOGO);
			qfle3f_log(hba, LOG_IOERR, "returning");
			goto exit;
		}
		qfle3f_err(hba, "DoPortLogout: targetPortID=%0x, WWNN=%lx, WWPN=%lx",
                    target->targetPortID, target->nodeName, target->portName);
		DoPortLogout(target->Sess, VMK_TRUE);
		do {
			vmk_WorldSleep(QFLE3F_RELOGIN_WAIT_TIME * VMK_USEC_PER_MSEC);
			if (wait_cnt++ > QFLE3F_RELOGIN_WAIT_CNT) {
				qfle3f_err(hba, "failed to Re-login")
				rc = FAILED;
				break;
			}
		} while (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY));
	}
exit:
	vmk_SpinlockLock(target->targetLock);
	return rc;
}



/**
 * qfle3f_eh_bus_reset: Bus Reset
 * @sc_cmd:	SCSI command
 *
 * Set from SCSI host template to send task mgmt command to 
 * all targets and wait for the response
 */
/*
 * This function is not used any more. From FC/FCoE POV, every
 * target has its own bus, and there is no point in resetting
 * all targets on an adapter when this invocation is made. Hence
 * reset just the target for which the reset is issued.
 */
VMK_ReturnStatus qfle3f_busReset(struct qfle3fHBA *hba, struct qfle3fFCLun *fclun,
						struct vmk_ScsiTaskMgmt *taskMgmt)
{
	struct qfle3f_rport *target = fclun->target;
	int rc = SUCCESS;
	int i;
    VMK_ReturnStatus status = VMK_OK;

	qfle3f_log(hba, LOG_IOERR, "Entered");

	vmk_SemaLock(&hba->hbaMutex);

	for (i = 0; i < QFLE3_FCOE_NUM_CONNECTIONS; i++) {
		target = hba->targetOffloadList[i];
		if (!target) {
			qfle3f_log(hba, LOG_IOERR, "targetOffloadList[%d] Invalid", i);
			continue;
		}
		/* qfle3f_executeTMF is a blocking call */
		rc = qfle3f_deviceReset(hba, fclun, taskMgmt);
		if (rc != SUCCESS) {
			qfle3f_err(hba, "rport[%d] error ",
						target->vmk_targetID);
            status = VMK_FAILURE;
	    }
    }

//bus_reset_err:
	vmk_SemaUnlock(&hba->hbaMutex);

	return status;
}

void qfle3f_processCleanupCompletion(struct qfle3fCommand *ioRequest,
				  struct fcoe_task_ctx_entry *task,
				  vmk_uint8 num_rq)
{
	struct qfle3fHBA *hba = NULL;
	VMK_ReturnStatus status;

	qfle3f_notice(hba, "xid = 0x%x cmd_type = %d", ioRequest->xid, ioRequest->cmd_type);
	qfle3f_scsiDone(ioRequest, VMK_SCSI_HOST_ERROR, 0);
	ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
	if (ioRequest->wait_for_comp) {
		if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
					QFLE3F_FLAG_ABTS_OR_CLNUP_PEN)) {
			qfle3f_notice(hba, "tm_compl - wake up cleanup done = %p", &(ioRequest->tm_done));
		} else {
			qfle3f_notice(hba, "tm_compl - wake up the waiter = %p", &(ioRequest->tm_done));
		}
		vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
        status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
		if(status != VMK_OK)
			qfle3f_log(hba, LOG_IOERR, "World wakeup on "
						"event tm_done: %s", vmk_StatusToString(status));
	}
}

vmk_Bool checkForMatchingCmd(struct qfle3fCommand *ioRequest,
                        struct qfle3fFCLun *fclun,
                        struct vmk_ScsiTaskMgmt *taskMgmt) {
	return ((ioRequest->sc_cmd->cmdId.serialNumber == taskMgmt->cmdId.serialNumber) &&
			(ioRequest->fclun->vmkLunId == fclun->vmkLunId) &&
			(ioRequest->sc_cmd->worldId == taskMgmt->worldId));
}

/**
 * qfle3f_ehAbort - eh_abort_handler api to abort an outstanding
 *			SCSI command
 *
 * @sc_cmd:	SCSI_ML command pointer
 *
 * SCSI abort request handler
 */
VMK_ReturnStatus qfle3f_ehAbort(struct qfle3fHBA *hba, struct qfle3fFCLun *fclun,
						struct vmk_ScsiTaskMgmt *taskMgmt)
{

	struct qfle3fCommand *ioRequest = NULL;
	struct qfle3f_rport *target = fclun->target;
	VMK_ReturnStatus status = VMK_FAILURE;
	vmk_Bool droppedIoRef;

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
		/*
		 * Session is not offloaded yet. Let SCSI-ml retry
		 * the command.
		 */
		return VMK_OK;
	}

	qfle3f_notice(hba, "abort SN = 0x%lx worldID = 0x%x",
            taskMgmt->cmdId.serialNumber, taskMgmt->worldId);

	vmk_SpinlockLock(target->targetLock);

	/* If this call successed, we would be holding a reference for the IO.
	 */
	ioRequest = qfle3f_getCmdSatisfyingACondition(fclun, taskMgmt,
		                    checkForMatchingCmd);
	if(!ioRequest)
	{
		/* Command might have just completed */
		qfle3f_err(hba, "ioRequest is NULL");
		vmk_SpinlockUnlock(target->targetLock);
		return VMK_OK;
	}

	if (vmk_BitVectorTest(target->flags, QFLE3F_FLAG_EXPL_LOGO)) {
		qfle3f_err(hba, "ioRequest (xid = 0x%x) "
				"already in expl logo processing", ioRequest->xid);
		goto drop_ref;
	}

	vmk_Bool shouldExit; // This field is not relevant in this context.

	status = qfle3f_internalAbortIO(hba, target, ioRequest, &shouldExit, &droppedIoRef);

	/* Release the reference taken by the qfle3f_getCmdSatisfyingACondition function
	 * Exception: When an explicit logo is issued, the reference is dropped from
	 * within qfle3f_explicitLogout function.
	 * Note: As the targetLock is held, we need not worry about this getting
	 * set while we are in internalAbortIO
	 */
	if (!droppedIoRef) {
drop_ref:
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
	}
	vmk_SpinlockUnlock(target->targetLock);
	qfle3f_log(hba, LOG_IOERR, "Returning status: %s", vmk_StatusToString(status));

	return status;
}

void qfle3f_processABTSCompletion(struct qfle3fCommand *ioRequest,
			       struct fcoe_task_ctx_entry *task,
			       vmk_uint8 num_rq)
{
	vmk_uint32 r_ctl;
	vmk_uint32 r_a_tov = 10;
	vmk_uint8 issue_rrq = 0;
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba = target->hba;
	VMK_ReturnStatus status;

	qfle3f_notice(hba, "xid = 0x%x cmd_type = %d",
		   ioRequest->xid, ioRequest->cmd_type);

	if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
						QFLE3F_FLAG_ABTS_DONE)) {
		qfle3f_log(hba, LOG_IOERR, "Timer context finished processing"
				" this io - 0x%x", ioRequest->xid);
		return;
	}

	/* Do not issue RRQ as this IO is already cleanedup */
	if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags,
				QFLE3F_FLAG_IO_CLEANUP))
		goto io_compl;

	/* For ABTS issued due to SCSI eh_abort_handler, timeout
	 * values are maintained by scsi-ml itself. Cancel timeout
	 * in case ABTS issued as part of task management function 
	 * or due to FW error.
	 */
	if (vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_ABTS))
		if(ql_vmk_cancel_delayed_work(&ioRequest->timeout_work) == VMK_OK)
			ql_vmk_ref_put(&ioRequest->refcount,
				 qfle3f_commandRelease, ioRequest); /* drop timer hold */

	r_ctl = (vmk_uint8)task->rxwr_only.union_ctx.comp_info.abts_rsp.r_ctl;

	switch (r_ctl) {
	case FC_R_CTL_BASIC_LINK_SERVICE_BA_ACC:
		/*
		 * Dont release this cmd yet. It will be relesed
		 * after we get RRQ response
		 */
		qfle3f_log(hba, LOG_IOERR, "ABTS response - ACC Send RRQ");
		issue_rrq = 1;
		break;

	case FC_R_CTL_BASIC_LINK_SERVICE_BA_RJT:
		qfle3f_log(hba, LOG_IOERR, "ABTS response - RJT");
		break;
	default:
		qfle3f_err(hba, "Unknown ABTS response");
		break;
	}

	if (issue_rrq) {
		qfle3f_log(hba, LOG_IOERR, "Issue RRQ after R_A_TOV");
		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_ISSUE_RRQ);
	}
	vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_RETIRE_OXID);
	qfle3f_commandTimerSet(ioRequest, r_a_tov * VMK_MSEC_PER_SEC);

io_compl:
	if (ioRequest->wait_for_comp) {
		if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
						QFLE3F_FLAG_EH_ABORT)) {
            qfle3f_notice(hba, "tm_compl - wake up the waiter = %p", &(ioRequest->tm_done));
			vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
            status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
			if(status != VMK_OK)
				qfle3f_log(hba, LOG_IOERR, "World wakeup on "
						"event tm_done: %s", vmk_StatusToString(status));
		} else if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
					QFLE3F_FLAG_ABTS_OR_CLNUP_PEN)) {
			/* We received an ABTS completion before the firmware could process
			 * the cleanup request.
			 */
			qfle3f_scsiDone(ioRequest, VMK_SCSI_HOST_ERROR, 0);
			ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
            qfle3f_notice(hba, "tm_compl - wake up cleanup done = %p", &(ioRequest->tm_done));
			vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
            status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
			if(status != VMK_OK)
				qfle3f_log(hba, LOG_IOERR, "World wakeup on "
						"event tm_done: %s", vmk_StatusToString(status));
		} else {
            qfle3f_err(hba, "Silent completion, ioRequest=%p", ioRequest);
		}
	} else {
		/*
		 * We end up here when ABTS is issued as
		 * in asynchronous context, i.e., as part
		 * of task management completion, or
		 * when FW error is received or when the
		 * ABTS is issued when the IO is timed
		 * out.
		 */

		if (ioRequest->on_active_queue) {
			vmk_ListRemove(&ioRequest->link);
			ioRequest->on_active_queue = 0;
			/* Move IO req to retire queue */
			vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&target->ioRetireQueue));
		}
		qfle3f_scsiDone(ioRequest, VMK_SCSI_HOST_ERROR, 0);
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
	}
}

static void qfle3f_lunResetCompletion(struct qfle3fCommand *ioRequest)
{
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba = target->hba;
	vmk_ListLinks *current, *nextPtr;
	struct qfle3fCommand *cmd;
	struct qfle3fFCLun *fclun = ioRequest->fclun;
	int rc = 0;
	int tm_lun = fclun->vmkLunId;
	int lun = -1;

	/* called with targetLock held */
	qfle3f_log(hba, LOG_IOERR, "Entered LUN: 0x%x", tm_lun);
	/*
	 * Walk thru the active_ios queue and ABORT the IO
	 * that matches with the LUN that was reset
	 */
	VMK_LIST_FORALL_SAFE(&target->activeCommands, current, nextPtr) {
		cmd = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);

		lun = cmd->fclun->vmkLunId;

		if (lun == tm_lun) {
		    qfle3f_log(hba, LOG_IOERR, "Found a pending IO: %p:0x%x", cmd, cmd->xid);
			/* Initiate ABTS on this cmd */
			if (!vmk_BitVectorAtomicTestAndSet(cmd->req_flags,
							QFLE3F_FLAG_ISSUE_ABTS)) {
				/* cancel the IO timeout */
				if(ql_vmk_cancel_delayed_work(&cmd->timeout_work) == VMK_OK)
					ql_vmk_ref_put(&cmd->refcount,
						 qfle3f_commandRelease, cmd);
							/* timer hold */

				rc = qfle3f_initiateABTS(cmd);
				/* abts shouldnt fail in this context */
				//WARN_ON(rc != SUCCESS);
			} else
				qfle3f_err(hba, "lun_rst: abts already in "
					   "progress for this IO 0x%x",
					   cmd->xid);
		}
	}
}

static void qfle3f_targetResetCompletion(struct qfle3fCommand *ioRequest)
{
	struct qfle3f_rport *target = ioRequest->target;
	vmk_ListLinks *current, *nextPtr;
	struct qfle3fCommand *cmd;
	struct qfle3fHBA *hba = NULL;
	int rc = 0;

	/* called with targetLock held */
	qfle3f_log(hba, LOG_IOERR, "Entered qfle3f_targetResetCompletion");
	/*
	 * Walk thru the active_ios queue and ABORT the IO
	 * that matches with the LUN that was reset
	 */
	VMK_LIST_FORALL_SAFE(&target->activeCommands, current, nextPtr) {
		qfle3f_log(hba, LOG_IOERR, "TGT RST cmpl: scan for pending IOs");
		cmd = VMK_LIST_ENTRY(current, struct qfle3fCommand, link);
		/* Initiate ABTS */
		if (!vmk_BitVectorAtomicTestAndSet(cmd->req_flags,
							QFLE3F_FLAG_ISSUE_ABTS)) {
			/* cancel the IO timeout */
			if(ql_vmk_cancel_delayed_work(&cmd->timeout_work) == VMK_OK)
				ql_vmk_ref_put(&cmd->refcount,
					 qfle3f_commandRelease, cmd); /* timer hold */
			rc = qfle3f_initiateABTS(cmd);
			/* abts shouldnt fail in this context */
			//WARN_ON(rc != SUCCESS);

		} else
			qfle3f_err(hba, "target_rst: abts already in progress "
				   "for this IO 0x%x", cmd->xid);
	}
}

void qfle3f_processTMCompletion(struct qfle3fCommand *ioRequest,
			     struct fcoe_task_ctx_entry *task, vmk_uint8 num_rq)
{
	struct qfle3f_mp_req *tm_req;
	FC_FRAME_HEADER *fc_hdr;
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba = target->hba;
	vmk_uint64 *hdr;
	vmk_uint64 *temp_hdr;
	void *rsp_buf;
	vmk_ByteCountSmall bytes_xferred = 0;
	VMK_ReturnStatus status;

	/* Called with targetLock held */
	qfle3f_log(hba, LOG_IOERR, "Entered process_tm_compl");

	if (!(vmk_BitVectorTest(ioRequest->req_flags, QFLE3F_FLAG_TM_TIMEOUT)))
		vmk_BitVectorSet(ioRequest->req_flags, QFLE3F_FLAG_TM_COMPL);
	else {
		/* TM has already timed out and we got 
	 	 * delayed completion. Ignore completion
		 * processing.
		 */
		return;
	}

	tm_req = &(ioRequest->mp_req);
	fc_hdr = &(tm_req->resp_fc_hdr);
	hdr = (vmk_uint64 *)fc_hdr;
	temp_hdr = (vmk_uint64 *)
		&task->rxwr_only.union_ctx.comp_info.mp_rsp.fc_hdr;
	//temp_hdr = (vmk_uint64 *)&task->cmn.general.comp_info;
	hdr[0] = vmk_CPUToBE64(temp_hdr[0]);
	hdr[1] = vmk_CPUToBE64(temp_hdr[1]);
	hdr[2] = vmk_CPUToBE64(temp_hdr[2]);

	tm_req->resp_len = task->rxwr_only.union_ctx.comp_info.mp_rsp.mp_payload_len;
	rsp_buf = tm_req->resp_buf;

	if (fc_hdr->r_ctl == FC_R_CTL_FC4_DEVICE_DATA_COMMAND_STATUS) {
		qfle3f_parseFCPResponse(ioRequest,
				     (struct fcoe_fcp_rsp_payload *)
				     rsp_buf, num_rq);
		if (ioRequest->fcp_rsp_code == 0) {
			/* TM successful */
			if (tm_req->tm_flags & FCP_TMF_LUN_RESET)
				qfle3f_lunResetCompletion(ioRequest);
			else if (tm_req->tm_flags & FCP_TMF_TGT_RESET)
				qfle3f_targetResetCompletion(ioRequest);
		}
	} else {
		qfle3f_err(hba, "tmf's fc_hdr r_ctl = 0x%x", fc_hdr->r_ctl);
	}
	switch(ioRequest->fcp_status) {
	case FC_GOOD:
		/* Good IO completion */
		sc_cmd->status.host = VMK_SCSI_HOST_OK;
		if (ioRequest->cdb_status) {
			/* Transport status is good, SCSI status not good */
			sc_cmd->status.device = ioRequest->cdb_status;
		}
		/* Determine bytes transfered for this I/O */
		if (ioRequest->fcp_resid) {
			if (ioRequest->fcp_resid <= vmk_SgGetDataLen(sc_cmd->sgArray)) {
				bytes_xferred = vmk_SgGetDataLen(sc_cmd->sgArray) - ioRequest->fcp_resid;
				if (bytes_xferred < sc_cmd->requiredDataLen) {
					qfle3f_notice(hba, "Data returned %x less then Required data length %x",
					    bytes_xferred, sc_cmd->requiredDataLen);
					sc_cmd->status.host = VMK_SCSI_HOST_ERROR;
				}
			}
			else {
				qfle3f_err(hba, "FCP reinitiatorPortIDual > data len, %x %lx",
							ioRequest->fcp_resid, vmk_SgGetDataLen(sc_cmd->sgArray));
				sc_cmd->status.host = VMK_SCSI_HOST_ERROR;
			}
		}
		else {
            if(sc_cmd->sgArray) {
			    bytes_xferred = vmk_SgGetDataLen(sc_cmd->sgArray);
            }
        }
		break;

	default:
		qfle3f_log(hba, LOG_IOERR, "process_tm_compl: fcp_status = %d",
			   ioRequest->fcp_status);
		break;
	}

	sc_cmd = ioRequest->sc_cmd;
	ioRequest->sc_cmd = NULL;

	/* check if the ioRequest exists in target's tmf_q */
	if (ioRequest->on_tmf_queue) {
		vmk_ListRemove(&ioRequest->link);
		ioRequest->on_tmf_queue = 0;
	} else {
		vmk_WarningMessage("Command not on activeCommands!");
		return;
	}

	ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
	if (ioRequest->wait_for_comp) {
        qfle3f_notice(hba, "tm_compl - wake up the waiter = %p", &(ioRequest->tm_done));
		vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
        status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
		if(status != VMK_OK)
			qfle3f_log(hba, LOG_IOERR, "World wakeup on "
					"event tm_done: %s", vmk_StatusToString(status));
	}
}

static int qfle3f_splitBD(struct qfle3fCommand *ioRequest, vmk_uint64 addr, int sg_len,
			   int bd_index)
{
	struct fcoe_bd_ctx *bd = ioRequest->bd_tbl->bd_tbl;
	int frag_size, sg_frags;

	sg_frags = 0;
	while (sg_len) {
		if (sg_len >= QFLE3F_BD_SPLIT_SZ)
			frag_size = QFLE3F_BD_SPLIT_SZ;
		else
			frag_size = sg_len;
		bd[bd_index + sg_frags].buf_addr_lo = addr & 0xffffffff;
		bd[bd_index + sg_frags].buf_addr_hi  = addr >> 32;
		bd[bd_index + sg_frags].buf_len = (vmk_uint16)frag_size;;
		bd[bd_index + sg_frags].flags = 0;

		addr += (vmk_uint64) frag_size;
		sg_frags++;
		sg_len -= frag_size;
	}
	return sg_frags;

}

static int qfle3f_mapSG(struct qfle3fCommand *ioRequest)
{
	struct qfle3fHBA *hba = ioRequest->vhba;
	vmk_ScsiCommand *sc = ioRequest->sc_cmd;
	struct fcoe_bd_ctx *bd = ioRequest->bd_tbl->bd_tbl;
	int byte_count = 0;
	int sg_count = 0;
	int bd_count = 0;
	int sg_frags;
	vmk_uint32 sg_len;
	vmk_uint64 addr;
	vmk_SgElem *cur_seg = NULL;
	int i;

	sg_count = sc->sgArray->numElems;

    for (i = 0; i < sg_count; i++) {
		cur_seg = &sc->sgArray->elem[i];
		sg_len = cur_seg->length;
		addr = cur_seg->ioAddr;

		if (sg_len > QFLE3F_MAX_BD_LEN) {
			sg_frags = qfle3f_splitBD(ioRequest, addr, sg_len,
						   bd_count);
		} else {

			sg_frags = 1;
			bd[bd_count].buf_addr_lo = U64_LO(addr);
			bd[bd_count].buf_addr_hi  = U64_HI(addr);
			bd[bd_count].buf_len = (vmk_uint16)sg_len;
			bd[bd_count].flags = 0;
		}
		byte_count += sg_len;
		bd_count += sg_frags;
	}

	if (byte_count != vmk_SgGetDataLen(sc->sgArray))
		qfle3f_err(hba, "byte_count = %d != scsi_bufflen = %ld, "
			   "task_id = 0x%x", byte_count, vmk_SgGetDataLen(sc->sgArray),
			   ioRequest->xid);
	return bd_count;
}

static void qfle3f_buildBDListFromSG(struct qfle3fCommand *ioRequest)
{
	vmk_ScsiCommand *sc = ioRequest->sc_cmd;
	struct fcoe_bd_ctx *bd = ioRequest->bd_tbl->bd_tbl;
	int bd_count;

    if(sc->sgArray->numElems) {
		bd_count = qfle3f_mapSG(ioRequest);
//	} else if(scsi_bufflen(sc)) {
//		bd_count = qfle3f_map_single_sg(ioRequest);
	} else {
		bd_count = 0;
		bd[0].buf_addr_lo = bd[0].buf_addr_hi = 0;
		bd[0].buf_len = bd[0].flags = 0;
	}
	ioRequest->bd_tbl->bd_valid = bd_count;
}

static void qfle3f_unmapSGList(struct qfle3fCommand *ioRequest)
{
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
#ifndef __VMKLNX__
	struct qfle3fHBA *hba = ioRequest->vhba;
	struct scatterlist *sg;
#endif

	if (ioRequest->bd_tbl->bd_valid && sc_cmd) {
#ifndef __VMKLNX__
		if (scsi_sg_count(sc_cmd)) {
			sg = scsi_sglist(sc_cmd);
			pci_unmap_sg(hba->pcidev, sg, scsi_sg_count(sc_cmd),
				     sc_cmd->sc_data_direction);
		}
#endif
		ioRequest->bd_tbl->bd_valid = 0;
	}
}

void qfle3f_buildFCPCommand(struct qfle3fCommand *ioRequest,
				  FCP_CMND_PAYLOAD *fcp_cmnd)
{
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
	struct qfle3fFCLun *fclun = ioRequest->fclun;
	//char tag[2];

	vmk_Memset(fcp_cmnd, 0, sizeof(FCP_CMND_PAYLOAD));

#if (VMWARE_ESX_DDK_VERSION >= 65000)
    if(ioRequest->cmd_type == QFLE3F_TASK_MGMT_CMD || ioRequest->ioctl_flags == 1) {
        qfle3f_int_to_scsilun_with_sec_lun_id(fclun->vmkLunId,
		    fcp_cmnd->Lun,
		    VMK_SCSI_INVALID_SECONDLEVEL_ID);
    } else {
        qfle3f_int_to_scsilun_with_sec_lun_id(fclun->vmkLunId,
		    fcp_cmnd->Lun,
		    vmk_ScsiCmdGetSecondLevelLunId(sc_cmd));
    }
#endif

    vmk_uint32 data_xfer_len = vmk_CPUToBE32(ioRequest->data_xfer_len);

	//fcp_cmnd->DataLength = vmk_CPUToBE32(ioRequest->data_xfer_len);
	vmk_Memcpy(fcp_cmnd->DataLength, &data_xfer_len, 4);
	vmk_Memcpy(fcp_cmnd->Cdb, sc_cmd->cdb, sc_cmd->cdbLen);

	fcp_cmnd->CommandReferenceNumber = 0;
	fcp_cmnd->Priority_TaskAttribute = 0;
	fcp_cmnd->TaskManagementFlags = ioRequest->mp_req.tm_flags;
	fcp_cmnd->AdditionalCdbLengthAndFlags = ioRequest->ioRequest_flags;

    /* Update tagged queuing modifier -- default is TSK_SIMPLE (0) */
    if (sc_cmd->flags & VMK_SCSI_COMMAND_FLAGS_ISSUE_WITH_ORDERED_TAG)
        fcp_cmnd->Priority_TaskAttribute = FCP_CMND_PAYLOAD_TASK_ATTR_ORDERED;
    else if (sc_cmd->flags & VMK_SCSI_COMMAND_FLAGS_ISSUE_WITH_HEAD_OF_Q_TAG)
        fcp_cmnd->Priority_TaskAttribute = FCP_CMND_PAYLOAD_TASK_ATTR_HEAD_OF_QUEUE;
    else
        fcp_cmnd->Priority_TaskAttribute = FCP_CMND_PAYLOAD_TASK_ATTR_SIMPLE;
}

void qfle3f_handleSenseData(struct qfle3fCommand *ioRequest, unsigned char *senseData)
{
    vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
    int fcp_sns_len = ioRequest->fcp_sns_len;
    struct qfle3f_rport *target = ioRequest->target;
    struct qfle3fHBA *hba = target->hba;

    if (fcp_sns_len > sizeof(struct vmk_ScsiSenseData)) {
			qfle3f_err(hba, "Truncating sense buffer");
			fcp_sns_len = sizeof(struct vmk_ScsiSenseData);
    }

    vmk_ScsiCmdClearSenseData(sc_cmd);

    if (fcp_sns_len) {
        qfle3f_log(hba, LOG_IOERR, "Setting sensedata of size: %d", fcp_sns_len);
        vmk_ScsiCmdSetSenseData((struct vmk_ScsiSenseData *) senseData,
                                sc_cmd,
                                fcp_sns_len);
    }
}

static void qfle3f_parseFCPResponse(struct qfle3fCommand *ioRequest,
				 struct fcoe_fcp_rsp_payload *fcp_rsp,
				 vmk_uint8 num_rq)
{
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
	struct qfle3f_rport *target = ioRequest->target;
	struct qfle3fHBA *hba = target->hba;
	vmk_uint8 rsp_flags = fcp_rsp->fcp_flags.flags;
	int i;
	unsigned char *rq_data;
	vmk_uint32 rq_buff_len = 0;
	int fcp_sns_len = 0;
	int fcp_rsp_len = 0;

	ioRequest->fcp_resid = 0;
	ioRequest->fcp_status = FC_GOOD;
	if (rsp_flags & (FCOE_FCP_RSP_FLAGS_FCP_RESID_OVER |
			FCOE_FCP_RSP_FLAGS_FCP_RESID_UNDER))
		ioRequest->fcp_resid = fcp_rsp->fcp_resid;

	ioRequest->scsi_comp_flags = rsp_flags;
	//CMD_SCSI_STATUS(sc_cmd) = ioRequest->cdb_status =
	//				fcp_rsp->scsi_status_code;
	sc_cmd->status.device = ioRequest->cdb_status = fcp_rsp->scsi_status_code;

	/* Fetch fcp_rsp_info and fcp_sns_info if available */
	if (num_rq) {

		/*
		 * We do not anticipate num_rq >1, as the linux defined
		 * SCSI_SENSE_BUFFERSIZE is 96 bytes + 8 bytes of FCP_RSP_INFO
		 * 256 bytes of single rq buffer is good enough to hold this.
		 */

		if (rsp_flags &
		    FCOE_FCP_RSP_FLAGS_FCP_RSP_LEN_VALID) {
			fcp_rsp_len = rq_buff_len
					= fcp_rsp->fcp_rsp_len;
		}

		if (rsp_flags &
		    FCOE_FCP_RSP_FLAGS_FCP_SNS_LEN_VALID) {
			fcp_sns_len = fcp_rsp->fcp_sns_len;
			rq_buff_len += fcp_rsp->fcp_sns_len;
		}

		ioRequest->fcp_rsp_len = fcp_rsp_len;
		ioRequest->fcp_sns_len = fcp_sns_len;

		if (rq_buff_len > num_rq * QFLE3F_RQ_BUF_SZ) {
			/* Invalid sense sense length. */
			vmk_LogMessage("invalid sns length %d",
				rq_buff_len);
			/* reset rq_buff_len */
			rq_buff_len =  num_rq * QFLE3F_RQ_BUF_SZ;
		}

		rq_data = qfle3f_getNextReceiveQueueEntry(target, 1);

		if (num_rq > 1) {
			/* We do not need extra sense data */
			for (i = 1; i < num_rq; i++)
				qfle3f_getNextReceiveQueueEntry(target, 1);
		}

		/* fetch fcp_rsp_code */
		if ((fcp_rsp_len == 4) || (fcp_rsp_len == 8)) {
			/* Only for task management function */
			ioRequest->fcp_rsp_code = rq_data[3];
			qfle3f_log(hba, LOG_SCSI_TM, "fcp_rsp_code = %d",
				   ioRequest->fcp_rsp_code);
		}

		/* fetch sense data */
		rq_data += fcp_rsp_len;

        qfle3f_handleSenseData(ioRequest, rq_data);

		/* return RQ entries */
		for (i = 0; i < num_rq; i++)
			qfle3f_returnReceiveQueueEntry(target, 1);
	}
}

/**
 * qfle3f_queuecommand - Queuecommand function of the scsi template
 * @sc_cmd:	vmk_ScsiCommand to be executed
 * @done:	Callback function to be called when sc_cmd is complted
 *
 * This is the IO strategy routine, called by SCSI-ML
 **/
VMK_ReturnStatus
qfle3f_scsiCommand(void *clientData, vmk_ScsiCommand *sc_cmd,
	void *deviceData)
{
	qfle3fHBA_t *hba = (qfle3fHBA_t *) clientData;
	struct qfle3fFCLun *fclun = (struct qfle3fFCLun *) deviceData;
	struct qfle3f_rport *target = fclun->target;
	struct qfle3fCommand *ioRequest;
	struct cnic_dev *dev;

	hba = target->hba;
	dev = hba->cnic;

    if(vmk_BitVectorTest(target->flags, QFLE3F_FLAG_DEV_LOSS)) {
        sc_cmd->status.host = VMK_SCSI_HOST_NO_CONNECT;
        qfle3f_err(hba, "(%d:%d): Returning IO with no connect: 0x%lx", target->vmk_targetID,
                        fclun->vmkLunId, vmk_AtomicRead64(&target->refcount));
        goto exit_qcmd;
    }

    if(vmk_BitVectorTest(hba->flags, QFLE3F_FLAG_UNLOADING)) {
        sc_cmd->status.host = VMK_SCSI_HOST_NO_CONNECT;
        qfle3f_err(hba, "(%d:%d): Returning IO module unloading: 0x%lx", target->vmk_targetID,
                        fclun->vmkLunId, vmk_AtomicRead64(&target->refcount));
        goto exit_qcmd;
    }

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
		/*
		 * Session is not offloaded yet. Let SCSI-ml retry
		 * the command.
		 */
		sc_cmd->status.host = VMK_SCSI_HOST_RETRY;
#ifdef IOERROR_DEBUGGING
		hba->ioRetrySessNotReady++; target->ioRetrySessNotReady++;
#endif
		goto exit_qcmd;
	}

	ioRequest = qfle3f_commandAlloc(target);
	if (!ioRequest) {
		sc_cmd->status.host = VMK_SCSI_HOST_RETRY;
#ifdef IOERROR_DEBUGGING

		hba->ioRetryCmdAllocFailed++; target->ioRetryCmdAllocFailed++;
#endif
		goto exit_qcmd;
	}

	ioRequest->sc_cmd = sc_cmd;
	ioRequest->fclun = fclun;

	if (qfle3f_postIORequest(target, ioRequest)) {
		qfle3f_err(hba, "Unable to post ioRequest");
		sc_cmd->status.host = VMK_SCSI_HOST_RETRY;
#ifdef IOERROR_DEBUGGING
		hba->ioRetrypostIOReqFailed++; target->ioRetrypostIOReqFailed++;
#endif
		goto exit_qcmd;
	}

    return VMK_OK;

exit_qcmd:
	sc_cmd->done(sc_cmd);
	return VMK_OK;
}

void qfle3f_processScsiCommandCompletion(struct qfle3fCommand *ioRequest,
				   struct fcoe_task_ctx_entry *task,
				   vmk_uint8 num_rq)
{
	struct fcoe_fcp_rsp_payload *fcp_rsp;
	struct qfle3f_rport *target = ioRequest->target;
	vmk_ScsiCommand *sc_cmd;
	//struct Scsi_Host *host;
	struct qfle3fHBA *hba = NULL;
	vmk_uint32 bytes_xferred = 0;
	VMK_ReturnStatus status;

	/* scsi_cmd_cmpl is called with target lock held */

	if (vmk_BitVectorAtomicTestAndSet(ioRequest->req_flags, QFLE3F_FLAG_IO_COMPL)) {
		/* we will not receive ABTS response for this IO */
		qfle3f_log(hba, LOG_IOERR, "Timer context finished processing "
			   "this scsi cmd - 0x%x", ioRequest->xid);
	}

	/* Cancel the timeout_work, as we received IO completion */
	status = ql_vmk_cancel_delayed_work(&ioRequest->timeout_work);
    if(status == VMK_OK)
		ql_vmk_ref_put(&ioRequest->refcount,
			 qfle3f_commandRelease, ioRequest); /* drop timer hold */

	sc_cmd = ioRequest->sc_cmd;
	if (sc_cmd == NULL) {
		qfle3f_err(hba, "scsi_cmd_compl - sc_cmd is NULL");
		return;
	}

	/* Fetch fcp_rsp from task context and perform cmd completion */
	fcp_rsp = (struct fcoe_fcp_rsp_payload *)
		   &(task->rxwr_only.union_ctx.comp_info.fcp_rsp.payload);

	/* parse fcp_rsp and obtain sense data from RQ if available */
	qfle3f_parseFCPResponse(ioRequest, fcp_rsp, num_rq);

	if (ioRequest->on_active_queue) {
		vmk_ListRemove(&ioRequest->link);
		ioRequest->on_active_queue = 0;
		/* Move IO req to retire queue */
		vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&target->ioRetireQueue));
	} else {
		/* This should not happen, but could have been pulled
		 * by qfle3f_flushActiveIOs(), or during a race
		 * between command abort and (late) completion.
		 */
		qfle3f_log(hba, LOG_IOERR, "xid not on activeCommands");
		if (ioRequest->wait_for_comp)
			if (vmk_BitVectorAtomicTestAndClear(ioRequest->req_flags,
							QFLE3F_FLAG_EH_ABORT)) {
		        qfle3f_err(hba, "tm_compl - wake up the waiter = %p", &(ioRequest->tm_done));
				vmk_AtomicWrite64(&ioRequest->tm_done_invoked, VMK_TRUE);
                status = vmk_WorldWakeup((vmk_WorldEventID)&(ioRequest->tm_done));
				if(status != VMK_OK)
					qfle3f_log(hba, LOG_IOERR, "World wakeup on "
							"event tm_done: %s",
							vmk_StatusToString(status));
		}
	}

	qfle3f_unmapSGList(ioRequest);
	ioRequest->sc_cmd = NULL;

    sc_cmd->status.host = VMK_SCSI_HOST_OK;

	switch (ioRequest->fcp_status) {
	case FC_GOOD:
		/* Good IO completion */
        if(ioRequest->cdb_status == 0) {
		    sc_cmd->status.device = VMK_SCSI_DEVICE_GOOD;
		} else {
			/* Transport status is good, SCSI status not good */
			qfle3f_log(hba, LOG_IOERR, "scsi_cmpl[0x%x]: cdb_status = %d"
				 " fcp_resid = 0x%x",
				ioRequest->xid, ioRequest->cdb_status,
				ioRequest->fcp_resid);
			sc_cmd->status.device = ioRequest->cdb_status;
		}

		/* Determine bytes transfered for this I/O */
		if (ioRequest->fcp_resid) {
		    if (ioRequest->fcp_resid <= vmk_SgGetDataLen(sc_cmd->sgArray)) {
			    bytes_xferred = vmk_SgGetDataLen(sc_cmd->sgArray) - ioRequest->fcp_resid;
				/* Under-run is expected for Check Condition Status */
			    if ((ioRequest->cdb_status != 2) &&
				    (bytes_xferred < sc_cmd->requiredDataLen)) {
					qfle3f_log(hba, LOG_IOERR, "mid-layer underflow detected xid[0x%x]"
						"lba=0x%lx lbc=0x%x cmd %x:%x:%x:%x:%x"
						" data returned %d required data %d",
						ioRequest->xid, sc_cmd->lba, sc_cmd->lbc,
						sc_cmd->cdb[0], sc_cmd->cdb[1], sc_cmd->cdb[2],
						sc_cmd->cdb[3], sc_cmd->cdb[4],
						bytes_xferred, sc_cmd->requiredDataLen);
	    		}
			} else {
			    qfle3f_err(hba, "FCP resid > data len, %x %lx",
						ioRequest->fcp_resid,
						vmk_SgGetDataLen(sc_cmd->sgArray));
				sc_cmd->status.host = VMK_SCSI_HOST_ERROR;
			}
        } else {
		    bytes_xferred = vmk_SgGetDataLen(sc_cmd->sgArray);
		}
		break;
	default:
		vmk_LogMessage("scsi_cmd_compl: fcp_status = %d",
			ioRequest->fcp_status);
		break;
	}

#ifdef IOERROR_DEBUGGING
	if(ioRequest->printIORelease)
		qfle3f_err(hba, "Calling cmd done for cmdSN: 0x%lx: xid = 0x%x bytes_xferred = 0x%x",
                    sc_cmd->cmdId.serialNumber, ioRequest->xid, bytes_xferred);
#endif
    sc_cmd->bytesXferred = bytes_xferred;
	vmk_ScsiSchedCommandCompletion(sc_cmd);
	ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);
}

int qfle3f_postIORequest(struct qfle3f_rport *target,
			       struct qfle3fCommand *ioRequest)
{
	struct fcoe_task_ctx_entry *task;
	struct fcoe_task_ctx_entry *task_page;
	vmk_ScsiCommand *sc_cmd = ioRequest->sc_cmd;
	//struct qfle3f_port *port = target->port;
	struct qfle3fHBA *hba = target->hba;
	int task_idx, index;
	vmk_uint16 xid;

	/* Initialize rest of ioRequest fields */
	ioRequest->cmd_type = QFLE3F_SCSI_CMD;
	ioRequest->target = target;
	ioRequest->data_xfer_len = vmk_SgGetDataLen(sc_cmd->sgArray);
	//sc_cmd->SCp.ptr = (char *)ioRequest;

	if (sc_cmd->dataDirection == VMK_SCSI_COMMAND_DIRECTION_READ)
		ioRequest->ioRequest_flags = QFLE3F_READ;
	else if (sc_cmd->dataDirection == VMK_SCSI_COMMAND_DIRECTION_WRITE)
		ioRequest->ioRequest_flags = QFLE3F_WRITE;
	else
		ioRequest->ioRequest_flags = 0;

	xid = ioRequest->xid;

	/* Build buffer descriptor list for firmware from sg list */
	qfle3f_buildBDListFromSG(ioRequest);

	task_idx = xid / QFLE3F_TASKS_PER_PAGE;
	index = xid % QFLE3F_TASKS_PER_PAGE;

	/* Initialize task context for this IO request */
	task_page = (struct fcoe_task_ctx_entry *) hba->taskContext[task_idx];
	task = &(task_page[index]);
	qfle3f_initializeTask(ioRequest, task);

	vmk_SpinlockLock(target->targetLock);

	if (target->flushInProgress) {
		qfle3f_err(hba, "Flush in progress..Host Busy");
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest);	/* ID: 001 */
		vmk_SpinlockUnlock(target->targetLock);
		return -1;
	}

	if (!vmk_BitVectorTest(target->flags, QFLE3F_FLAG_SESSION_READY)) {
		qfle3f_err(hba, "Session not ready...post_io");
		ql_vmk_ref_put(&ioRequest->refcount, qfle3f_commandRelease, ioRequest); /* ID: 001 */
		vmk_SpinlockUnlock(target->targetLock);
		return -1;
	}

	/* Time IO req */
	qfle3f_log(hba, LOG_INFO, "ioRequest = %p", ioRequest);
	qfle3f_commandTimerSet(ioRequest, QFLE3F_IO_TIMEOUT);
	/* Obtain free SQ entry */
	qfle3f_addToSQ(target, xid);

	/* Enqueue the ioRequest to activeCommands */
	ioRequest->on_active_queue = 1;
	/* move ioRequest from pending_queue to active_queue */
	vmk_ListInsert(&ioRequest->link, vmk_ListAtRear(&target->activeCommands));

	/* Ring doorbell */
	qfle3f_ringDoorbell(target);
	vmk_SpinlockUnlock(target->targetLock);
	return 0;
}
