/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.asterix.jdbc.core;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.NClob;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLType;
import java.sql.SQLXML;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collections;
import java.util.List;

public class ADBPreparedStatement extends ADBStatement implements PreparedStatement {

    protected final String sql;

    protected final List<Object> args;

    protected final List<ADBColumn> resultColumns;

    public ADBPreparedStatement(ADBConnection connection, String sql) throws SQLException {
        super(connection);
        ADBProtocolBase.SubmitStatementOptions stmtOptions = createSubmitStatementOptions();
        stmtOptions.compileOnly = true;
        stmtOptions.timeoutSeconds = 0; /* TODO:timeout */
        ADBProtocolBase.QueryServiceResponse response = connection.protocol.submitStatement(sql, null, stmtOptions);
        int parameterCount = connection.protocol.getStatementParameterCount(response);
        boolean isQuery = connection.protocol.isStatementCategory(response,
                ADBProtocolBase.QueryServiceResponse.StatementCategory.QUERY);
        List<ADBColumn> columns = isQuery ? connection.protocol.getColumns(response) : Collections.emptyList();
        this.sql = sql;
        this.args = Arrays.asList(new Object[parameterCount]);
        this.resultColumns = columns;
    }

    // Metadata

    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException {
        checkClosed();
        return new ADBParameterMetaData(this, args.size());
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
        checkClosed();
        return new ADBResultSetMetaData(this, resultColumns);
    }

    // Execution

    @Override
    public ResultSet executeQuery() throws SQLException {
        checkClosed();
        return executeQueryImpl(sql, args);
    }

    @Override
    public ADBResultSet executeQuery(String sql) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeQuery");
    }

    @Override
    public long executeLargeUpdate() throws SQLException {
        return executeUpdateImpl(sql, args);
    }

    @Override
    public int executeUpdate() throws SQLException {
        return executeUpdateImpl(sql, args);
    }

    @Override
    public long executeLargeUpdate(String sql) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeLargeUpdate");
    }

    @Override
    public int executeUpdate(String sql) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeUpdate");
    }

    @Override
    public long executeLargeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeLargeUpdate");
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeUpdate");
    }

    @Override
    public long executeLargeUpdate(String sql, int[] columnIndexes) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeLargeUpdate");
    }

    @Override
    public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeUpdate");
    }

    @Override
    public long executeLargeUpdate(String sql, String[] columnNames) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeLargeUpdate");
    }

    @Override
    public int executeUpdate(String sql, String[] columnNames) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "executeUpdate");
    }

    @Override
    public boolean execute() throws SQLException {
        return executeImpl(sql, args);
    }

    @Override
    public boolean execute(String sql) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "execute");
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "execute");
    }

    @Override
    public boolean execute(String sql, int[] columnIndexes) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "execute");
    }

    @Override
    public boolean execute(String sql, String[] columnNames) throws SQLException {
        // Prohibited on PreparedStatement
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "execute");
    }

    @Override
    public void addBatch() throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "addBatch");
    }

    @Override
    public void setEscapeProcessing(boolean enable) throws SQLException {
        checkClosed();
        // ignore as the statement has already been parsed
    }

    // Parameter bindings

    @Override
    public void clearParameters() throws SQLException {
        checkClosed();
        for (int i = 0, n = args.size(); i < n; i++) {
            args.set(i, null);
        }
    }

    private int argIndex(int parameterIndex) throws SQLException {
        boolean ok = 0 < parameterIndex && parameterIndex <= args.size();
        if (!ok) {
            throw getErrorReporter().errorParameterValueNotSupported("parameterIndex");
        }
        return parameterIndex - 1;
    }

    private void setArg(int parameterIndex, Object v) throws SQLException {
        args.set(argIndex(parameterIndex), v);
    }

    // Basic types

    @Override
    public void setNull(int parameterIndex, int sqlType) throws SQLException {
        checkClosed();
        setArg(parameterIndex, null);
    }

    @Override
    public void setNull(int parameterIndex, int sqlType, String typeName) throws SQLException {
        checkClosed();
        setNull(parameterIndex, sqlType);
    }

    @Override
    public void setBoolean(int parameterIndex, boolean v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setByte(int parameterIndex, byte v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setShort(int parameterIndex, short v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setInt(int parameterIndex, int v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setLong(int parameterIndex, long v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setFloat(int parameterIndex, float v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setDouble(int parameterIndex, double v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setBigDecimal(int parameterIndex, BigDecimal v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setString(int parameterIndex, String v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setNString(int parameterIndex, String v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setDate(int parameterIndex, java.sql.Date v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setDate(int parameterIndex, java.sql.Date v, Calendar cal) throws SQLException {
        checkClosed();
        setArg(parameterIndex, cal != null ? new SqlCalendarDate(v, cal.getTimeZone()) : v);
    }

    @Override
    public void setTime(int parameterIndex, java.sql.Time v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setTime(int parameterIndex, java.sql.Time v, Calendar cal) throws SQLException {
        checkClosed();
        setArg(parameterIndex, cal != null ? new SqlCalendarTime(v, cal.getTimeZone()) : v);
    }

    @Override
    public void setTimestamp(int parameterIndex, java.sql.Timestamp v) throws SQLException {
        checkClosed();
        setArg(parameterIndex, v);
    }

    @Override
    public void setTimestamp(int parameterIndex, java.sql.Timestamp v, Calendar cal) throws SQLException {
        checkClosed();
        setArg(parameterIndex, cal != null ? new SqlCalendarTimestamp(v, cal.getTimeZone()) : v);
    }

    // Generic (setObject)

    @Override
    public void setObject(int parameterIndex, Object v) throws SQLException {
        checkClosed();
        if (v == null || isSetObjectCompatible(v.getClass())) {
            setArg(parameterIndex, v);
        } else {
            throw getErrorReporter().errorParameterValueNotSupported("object");
        }
    }

    @Override
    public void setObject(int parameterIndex, Object v, int targetSqlType) throws SQLException {
        setObject(parameterIndex, v);
    }

    @Override
    public void setObject(int parameterIndex, Object v, SQLType targetSqlType) throws SQLException {
        if (targetSqlType == null) {
            setObject(parameterIndex, v);
        } else {
            setObject(parameterIndex, v, targetSqlType.getVendorTypeNumber());
        }
    }

    @Override
    public void setObject(int parameterIndex, Object v, int targetSqlType, int scaleOrLength) throws SQLException {
        setObject(parameterIndex, v, targetSqlType);
    }

    @Override
    public void setObject(int parameterIndex, Object v, SQLType targetSqlType, int scaleOrLength) throws SQLException {
        if (targetSqlType == null) {
            setObject(parameterIndex, v);
        } else {
            setObject(parameterIndex, v, targetSqlType.getVendorTypeNumber());
        }
    }

    // Unsupported

    @Override
    public void setBytes(int parameterIndex, byte[] v) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBytes");
    }

    @Override
    public void setRef(int parameterIndex, Ref x) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setRef");
    }

    @Override
    public void setRowId(int parameterIndex, RowId x) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setRowId");
    }

    @Override
    public void setURL(int parameterIndex, URL v) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setURL");
    }

    // Unsupported - streams

    @Override
    public void setAsciiStream(int parameterIndex, InputStream v) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setAsciiStream");
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setAsciiStream");
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream v, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setAsciiStream");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBinaryStream");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBinaryStream");
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream x, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBinaryStream");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setCharacterStream");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setCharacterStream");
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setCharacterStream");
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader value) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setNCharacterStream");
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader value, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setNCharacterStream");
    }

    @Override
    public void setUnicodeStream(int parameterIndex, InputStream x, int length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setUnicodeStream");
    }

    // Unsupported - LOB, Array, SQLXML

    @Override
    public void setArray(int parameterIndex, Array x) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setArray");
    }

    @Override
    public void setBlob(int parameterIndex, Blob x) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBlob");
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBlob");
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setBlob");
    }

    @Override
    public void setClob(int parameterIndex, Clob x) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setClob");
    }

    @Override
    public void setClob(int parameterIndex, Reader reader) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setClob");
    }

    @Override
    public void setClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setClob");
    }

    @Override
    public void setNClob(int parameterIndex, NClob value) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setNClob");
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setNClob");
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setNClob");
    }

    @Override
    public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException {
        throw getErrorReporter().errorMethodNotSupported(PreparedStatement.class, "setSQLXML");
    }
}
