/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.codec.binary;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Random;
import org.apache.commons.codec.CodecPolicy;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.binary.Base16;
import org.apache.commons.codec.binary.BaseNCodec;
import org.apache.commons.codec.binary.BaseNTestData;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Test;

class Base16Test {
    private static final Charset CHARSET_UTF8 = StandardCharsets.UTF_8;
    private final Random random = new Random();

    Base16Test() {
    }

    public Random getRandom() {
        return this.random;
    }

    @Test
    void testBase16() {
        String content = "Hello World";
        byte[] encodedBytes = new Base16().encode(StringUtils.getBytesUtf8((String)"Hello World"));
        String encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        Assertions.assertEquals((Object)"48656C6C6F20576F726C64", (Object)encodedContent, (String)"encoding hello world");
        byte[] decodedBytes = new Base16().decode(encodedBytes);
        String decodedContent = StringUtils.newStringUtf8((byte[])decodedBytes);
        Assertions.assertEquals((Object)"Hello World", (Object)decodedContent, (String)"decoding hello world");
    }

    @Test
    void testBase16AtBufferEnd() {
        this.testBase16InBuffer(100, 0);
    }

    @Test
    void testBase16AtBufferMiddle() {
        this.testBase16InBuffer(100, 100);
    }

    @Test
    void testBase16AtBufferStart() {
        this.testBase16InBuffer(0, 100);
    }

    private void testBase16InBuffer(int startPasSize, int endPadSize) {
        String content = "Hello World";
        byte[] bytesUtf8 = StringUtils.getBytesUtf8((String)"Hello World");
        byte[] buffer = ArrayUtils.addAll((byte[])bytesUtf8, (byte[])new byte[endPadSize]);
        buffer = ArrayUtils.addAll((byte[])new byte[startPasSize], (byte[])buffer);
        byte[] encodedBytes = new Base16().encode(buffer, startPasSize, bytesUtf8.length);
        String encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        Assertions.assertEquals((Object)"48656C6C6F20576F726C64", (Object)encodedContent, (String)"encoding hello world");
    }

    @Test
    void testByteToStringVariations() {
        Base16 base16 = new Base16();
        byte[] b1 = StringUtils.getBytesUtf8((String)"Hello World");
        byte[] b2 = new byte[]{};
        byte[] b3 = null;
        Assertions.assertEquals((Object)"48656C6C6F20576F726C64", (Object)base16.encodeToString(b1), (String)"byteToString Hello World");
        Assertions.assertEquals((Object)"48656C6C6F20576F726C64", (Object)StringUtils.newStringUtf8((byte[])new Base16().encode(b1)), (String)"byteToString static Hello World");
        Assertions.assertEquals((Object)"", (Object)base16.encodeToString(b2), (String)"byteToString \"\"");
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUtf8((byte[])new Base16().encode(b2)), (String)"byteToString static \"\"");
        Assertions.assertNull((Object)base16.encodeToString(b3), (String)"byteToString null");
        Assertions.assertNull((Object)StringUtils.newStringUtf8((byte[])new Base16().encode(b3)), (String)"byteToString static null");
    }

    @Test
    void testCheckEncodeLengthBounds() {
        Base16 base16 = new Base16();
        Assertions.assertThrows(IllegalArgumentException.class, () -> base16.encode(new byte[10], 0, 0x40000000));
    }

    @Test
    void testCodec68() {
        byte[] x = new byte[]{110, 72, 61, 61, -100};
        Base16 b16 = new Base16();
        Assertions.assertThrows(RuntimeException.class, () -> b16.decode(x));
    }

    @Test
    void testConstructor_LowerCase() {
        Base16 base16 = new Base16(true);
        byte[] encoded = base16.encode(BaseNTestData.DECODED);
        String expectedResult = "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";
        String result = StringUtils.newStringUtf8((byte[])encoded);
        Assertions.assertEquals((Object)"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", (Object)result, (String)"new Base16(true)");
    }

    @Test
    void testConstructor_LowerCase_DecodingPolicy() {
        Base16 base16 = new Base16(false, CodecPolicy.STRICT);
        byte[] encoded = base16.encode(BaseNTestData.DECODED);
        String expectedResult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
        String result = StringUtils.newStringUtf8((byte[])encoded);
        Assertions.assertEquals((Object)result, (Object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tring)"new base16(false, CodecPolicy.STRICT)");
    }

    @Test
    void testConstructors() {
        new Base16();
        new Base16(false);
        new Base16(true);
        new Base16(false, CodecPolicy.LENIENT);
        new Base16(false, CodecPolicy.STRICT);
    }

    @Test
    void testDecodeSingleBytes() {
        String encoded = "556E74696C206E6578742074696D6521";
        BaseNCodec.Context context = new BaseNCodec.Context();
        Base16 b16 = new Base16();
        byte[] encocdedBytes = StringUtils.getBytesUtf8((String)"556E74696C206E6578742074696D6521");
        b16.decode(encocdedBytes, 0, 1, context);
        b16.decode(encocdedBytes, 1, 1, context);
        b16.decode(encocdedBytes, 2, 1, context);
        b16.decode(encocdedBytes, 3, 1, context);
        b16.decode(encocdedBytes, 4, 3, context);
        b16.decode(encocdedBytes, 7, 3, context);
        b16.decode(encocdedBytes, 10, 3, context);
        b16.decode(encocdedBytes, 13, 19, context);
        byte[] decodedBytes = new byte[context.pos];
        System.arraycopy(context.buffer, context.readPos, decodedBytes, 0, decodedBytes.length);
        String decoded = StringUtils.newStringUtf8((byte[])decodedBytes);
        Assertions.assertEquals((Object)"Until next time!", (Object)decoded);
    }

    @Test
    void testDecodeSingleBytesOptimization() {
        BaseNCodec.Context context = new BaseNCodec.Context();
        Assertions.assertEquals((int)0, (int)context.ibitWorkArea);
        Assertions.assertNull((Object)context.buffer);
        byte[] data = new byte[1];
        Base16 b16 = new Base16();
        data[0] = 69;
        b16.decode(data, 0, 1, context);
        Assertions.assertEquals((int)15, (int)context.ibitWorkArea);
        Assertions.assertNull((Object)context.buffer);
        data[0] = 70;
        b16.decode(data, 0, 1, context);
        Assertions.assertEquals((int)0, (int)context.ibitWorkArea);
        Assertions.assertEquals((byte)-17, (byte)context.buffer[0]);
    }

    @Test
    void testEmptyBase16() {
        byte[] empty = new byte[]{};
        byte[] result = new Base16().encode(empty);
        Assertions.assertEquals((int)0, (int)result.length, (String)"empty Base16 encode");
        Assertions.assertNull((Object)new Base16().encode(null), (String)"empty Base16 encode");
        result = new Base16().encode(empty, 0, 1);
        Assertions.assertEquals((int)0, (int)result.length, (String)"empty Base16 encode with offset");
        Assertions.assertNull((Object)new Base16().encode(null), (String)"empty Base16 encode with offset");
        empty = new byte[]{};
        result = new Base16().decode(empty);
        Assertions.assertEquals((int)0, (int)result.length, (String)"empty Base16 decode");
        Assertions.assertNull((Object)new Base16().decode((byte[])null), (String)"empty Base16 encode");
    }

    @Test
    void testEncodeDecodeRandom() {
        for (int i = 1; i < 5; ++i) {
            int len = this.getRandom().nextInt(10000) + 1;
            byte[] data = new byte[len];
            this.getRandom().nextBytes(data);
            byte[] enc = new Base16().encode(data);
            byte[] data2 = new Base16().decode(enc);
            Assertions.assertArrayEquals((byte[])data, (byte[])data2);
        }
    }

    @Test
    void testEncodeDecodeSmall() {
        for (int i = 0; i < 12; ++i) {
            byte[] data = new byte[i];
            this.getRandom().nextBytes(data);
            byte[] enc = new Base16().encode(data);
            byte[] data2 = new Base16().decode(enc);
            Assertions.assertArrayEquals((byte[])data, (byte[])data2, (String)(this.toString(data) + " equals " + this.toString(data2)));
        }
    }

    @Test
    void testIsInAlphabet() {
        int c;
        Base16 b16 = Base16.builder().setLowerCase(true).get();
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)0));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)1));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)-1));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)-15));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)-16));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)-128));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)-1));
        b16 = new Base16(true);
        for (c = 48; c <= 57; c = (int)((char)(c + 1))) {
            Assertions.assertTrue((boolean)b16.isInAlphabet((byte)c));
        }
        for (c = 97; c <= 102; c = (int)((char)(c + 1))) {
            Assertions.assertTrue((boolean)b16.isInAlphabet((byte)c));
        }
        for (c = 65; c <= 70; c = (int)((char)(c + 1))) {
            Assertions.assertFalse((boolean)b16.isInAlphabet((byte)c));
        }
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)47));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)58));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)96));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)103));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)123));
        b16 = new Base16(false);
        for (c = 48; c <= 57; c = (int)((char)(c + 1))) {
            Assertions.assertTrue((boolean)b16.isInAlphabet((byte)c));
        }
        for (c = 97; c <= 102; c = (int)((char)(c + 1))) {
            Assertions.assertFalse((boolean)b16.isInAlphabet((byte)c));
        }
        for (c = 65; c <= 70; c = (int)((char)(c + 1))) {
            Assertions.assertTrue((boolean)b16.isInAlphabet((byte)c));
        }
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)47));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)58));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)64));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)71));
        Assertions.assertFalse((boolean)b16.isInAlphabet((byte)91));
    }

    @Test
    void testKnownDecodings() {
        Assertions.assertEquals((Object)"The quick brown fox jumped over the lazy dogs.", (Object)new String(new Base16(true).decode("54686520717569636b2062726f776e20666f78206a756d706564206f76657220746865206c617a7920646f67732e".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"It was the best of times, it was the worst of times.", (Object)new String(new Base16(true).decode("497420776173207468652062657374206f662074696d65732c206974207761732074686520776f727374206f662074696d65732e".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"http://jakarta.apache.org/commmons", (Object)new String(new Base16(true).decode("687474703a2f2f6a616b617274612e6170616368652e6f72672f636f6d6d6d6f6e73".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz", (Object)new String(new Base16(true).decode("4161426243634464456546664767486849694a6a4b6b4c6c4d6d4e6e4f6f50705171527253735474557556765777587859795a7a".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }", (Object)new String(new Base16(true).decode("7b20302c20312c20322c20332c20342c20352c20362c20372c20382c2039207d".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"xyzzy!", (Object)new String(new Base16(true).decode("78797a7a7921".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testKnownEncodings() {
        Assertions.assertEquals((Object)"54686520717569636b2062726f776e20666f78206a756d706564206f76657220746865206c617a7920646f67732e", (Object)new String(new Base16(true).encode("The quick brown fox jumped over the lazy dogs.".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"497420776173207468652062657374206f662074696d65732c206974207761732074686520776f727374206f662074696d65732e", (Object)new String(new Base16(true).encode("It was the best of times, it was the worst of times.".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"687474703a2f2f6a616b617274612e6170616368652e6f72672f636f6d6d6d6f6e73", (Object)new String(new Base16(true).encode("http://jakarta.apache.org/commmons".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"4161426243634464456546664767486849694a6a4b6b4c6c4d6d4e6e4f6f50705171527253735474557556765777587859795a7a", (Object)new String(new Base16(true).encode("AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"7b20302c20312c20322c20332c20342c20352c20362c20372c20382c2039207d", (Object)new String(new Base16(true).encode("{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"78797a7a7921", (Object)new String(new Base16(true).encode("xyzzy!".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testLenientDecoding() {
        String encoded = "aabbccdde";
        Base16 b16 = new Base16(true, CodecPolicy.LENIENT);
        Assertions.assertEquals((Object)CodecPolicy.LENIENT, (Object)b16.getCodecPolicy());
        byte[] decoded = b16.decode(StringUtils.getBytesUtf8((String)"aabbccdde"));
        Assertions.assertArrayEquals((byte[])new byte[]{-86, -69, -52, -35}, (byte[])decoded);
    }

    @Test
    void testNonBase16Test() {
        byte[] invalidEncodedChars = new byte[]{47, 58, 64, 71, 37, 96, 103};
        byte[] encoded = new byte[1];
        byte[] byArray = invalidEncodedChars;
        int n = byArray.length;
        for (int i = 0; i < n; ++i) {
            byte invalidEncodedChar;
            encoded[0] = invalidEncodedChar = byArray[i];
            Assertions.assertThrows(IllegalArgumentException.class, () -> new Base16().decode(encoded), (String)("Invalid Base16 char: " + (char)invalidEncodedChar));
        }
    }

    @Test
    void testObjectDecodeWithInvalidParameter() {
        Assertions.assertThrows(DecoderException.class, () -> new Base16().decode((Object)5));
    }

    @Test
    void testObjectDecodeWithValidParameter() throws Exception {
        String original = "Hello World!";
        byte[] o = new Base16().encode("Hello World!".getBytes(CHARSET_UTF8));
        Base16 b16 = new Base16();
        Object oDecoded = b16.decode((Object)o);
        byte[] baDecoded = (byte[])oDecoded;
        String dest = new String(baDecoded);
        Assertions.assertEquals((Object)"Hello World!", (Object)dest, (String)"dest string does not equal original");
    }

    @Test
    void testObjectEncode() {
        Base16 b16 = new Base16();
        Assertions.assertEquals((Object)new String(b16.encode("Hello World".getBytes(CHARSET_UTF8))), (Object)"48656C6C6F20576F726C64");
    }

    @Test
    void testObjectEncodeWithInvalidParameter() {
        Assertions.assertThrows(EncoderException.class, () -> new Base16().encode((Object)"Yadayadayada"));
    }

    @Test
    void testObjectEncodeWithValidParameter() throws Exception {
        String original = "Hello World!";
        byte[] origObj = "Hello World!".getBytes(CHARSET_UTF8);
        Object oEncoded = new Base16().encode((Object)origObj);
        byte[] bArray = new Base16().decode((byte[])oEncoded);
        String dest = new String(bArray);
        Assertions.assertEquals((Object)"Hello World!", (Object)dest, (String)"dest string does not equal original");
    }

    @Test
    void testOddEvenDecoding() {
        String encoded = "4142434445";
        BaseNCodec.Context context = new BaseNCodec.Context();
        Base16 base16 = new Base16();
        byte[] encodedBytes = StringUtils.getBytesUtf8((String)"4142434445");
        base16.decode(encodedBytes, 0, 3, context);
        base16.decode(encodedBytes, 3, 4, context);
        base16.decode(encodedBytes, 7, 3, context);
        byte[] decodedBytes = new byte[context.pos];
        System.arraycopy(context.buffer, context.readPos, decodedBytes, 0, decodedBytes.length);
        String decoded = StringUtils.newStringUtf8((byte[])decodedBytes);
        Assertions.assertEquals((Object)"ABCDE", (Object)decoded);
    }

    @Test
    void testPairs() {
        Assertions.assertEquals((Object)"0000", (Object)new String(new Base16().encode(new byte[]{0, 0})));
        Assertions.assertEquals((Object)"0001", (Object)new String(new Base16().encode(new byte[]{0, 1})));
        Assertions.assertEquals((Object)"0002", (Object)new String(new Base16().encode(new byte[]{0, 2})));
        Assertions.assertEquals((Object)"0003", (Object)new String(new Base16().encode(new byte[]{0, 3})));
        Assertions.assertEquals((Object)"0004", (Object)new String(new Base16().encode(new byte[]{0, 4})));
        Assertions.assertEquals((Object)"0005", (Object)new String(new Base16().encode(new byte[]{0, 5})));
        Assertions.assertEquals((Object)"0006", (Object)new String(new Base16().encode(new byte[]{0, 6})));
        Assertions.assertEquals((Object)"0007", (Object)new String(new Base16().encode(new byte[]{0, 7})));
        Assertions.assertEquals((Object)"0008", (Object)new String(new Base16().encode(new byte[]{0, 8})));
        Assertions.assertEquals((Object)"0009", (Object)new String(new Base16().encode(new byte[]{0, 9})));
        Assertions.assertEquals((Object)"000A", (Object)new String(new Base16().encode(new byte[]{0, 10})));
        Assertions.assertEquals((Object)"000B", (Object)new String(new Base16().encode(new byte[]{0, 11})));
        Assertions.assertEquals((Object)"000C", (Object)new String(new Base16().encode(new byte[]{0, 12})));
        Assertions.assertEquals((Object)"000D", (Object)new String(new Base16().encode(new byte[]{0, 13})));
        Assertions.assertEquals((Object)"000E", (Object)new String(new Base16().encode(new byte[]{0, 14})));
        Assertions.assertEquals((Object)"000F", (Object)new String(new Base16().encode(new byte[]{0, 15})));
        Assertions.assertEquals((Object)"0010", (Object)new String(new Base16().encode(new byte[]{0, 16})));
        Assertions.assertEquals((Object)"0011", (Object)new String(new Base16().encode(new byte[]{0, 17})));
        for (int i = -128; i <= 127; ++i) {
            byte[] test = new byte[]{(byte)i, (byte)i};
            Assertions.assertArrayEquals((byte[])test, (byte[])new Base16().decode(new Base16().encode(test)));
        }
    }

    @Test
    void testSingletons() {
        Assertions.assertEquals((Object)"00", (Object)new String(new Base16().encode(new byte[]{0})));
        Assertions.assertEquals((Object)"01", (Object)new String(new Base16().encode(new byte[]{1})));
        Assertions.assertEquals((Object)"02", (Object)new String(new Base16().encode(new byte[]{2})));
        Assertions.assertEquals((Object)"03", (Object)new String(new Base16().encode(new byte[]{3})));
        Assertions.assertEquals((Object)"04", (Object)new String(new Base16().encode(new byte[]{4})));
        Assertions.assertEquals((Object)"05", (Object)new String(new Base16().encode(new byte[]{5})));
        Assertions.assertEquals((Object)"06", (Object)new String(new Base16().encode(new byte[]{6})));
        Assertions.assertEquals((Object)"07", (Object)new String(new Base16().encode(new byte[]{7})));
        Assertions.assertEquals((Object)"08", (Object)new String(new Base16().encode(new byte[]{8})));
        Assertions.assertEquals((Object)"09", (Object)new String(new Base16().encode(new byte[]{9})));
        Assertions.assertEquals((Object)"0A", (Object)new String(new Base16().encode(new byte[]{10})));
        Assertions.assertEquals((Object)"0B", (Object)new String(new Base16().encode(new byte[]{11})));
        Assertions.assertEquals((Object)"0C", (Object)new String(new Base16().encode(new byte[]{12})));
        Assertions.assertEquals((Object)"0D", (Object)new String(new Base16().encode(new byte[]{13})));
        Assertions.assertEquals((Object)"0E", (Object)new String(new Base16().encode(new byte[]{14})));
        Assertions.assertEquals((Object)"0F", (Object)new String(new Base16().encode(new byte[]{15})));
        Assertions.assertEquals((Object)"10", (Object)new String(new Base16().encode(new byte[]{16})));
        Assertions.assertEquals((Object)"11", (Object)new String(new Base16().encode(new byte[]{17})));
        Assertions.assertEquals((Object)"12", (Object)new String(new Base16().encode(new byte[]{18})));
        Assertions.assertEquals((Object)"13", (Object)new String(new Base16().encode(new byte[]{19})));
        Assertions.assertEquals((Object)"14", (Object)new String(new Base16().encode(new byte[]{20})));
        Assertions.assertEquals((Object)"15", (Object)new String(new Base16().encode(new byte[]{21})));
        Assertions.assertEquals((Object)"16", (Object)new String(new Base16().encode(new byte[]{22})));
        Assertions.assertEquals((Object)"17", (Object)new String(new Base16().encode(new byte[]{23})));
        Assertions.assertEquals((Object)"18", (Object)new String(new Base16().encode(new byte[]{24})));
        Assertions.assertEquals((Object)"19", (Object)new String(new Base16().encode(new byte[]{25})));
        Assertions.assertEquals((Object)"1A", (Object)new String(new Base16().encode(new byte[]{26})));
        Assertions.assertEquals((Object)"1B", (Object)new String(new Base16().encode(new byte[]{27})));
        Assertions.assertEquals((Object)"1C", (Object)new String(new Base16().encode(new byte[]{28})));
        Assertions.assertEquals((Object)"1D", (Object)new String(new Base16().encode(new byte[]{29})));
        Assertions.assertEquals((Object)"1E", (Object)new String(new Base16().encode(new byte[]{30})));
        Assertions.assertEquals((Object)"1F", (Object)new String(new Base16().encode(new byte[]{31})));
        Assertions.assertEquals((Object)"20", (Object)new String(new Base16().encode(new byte[]{32})));
        Assertions.assertEquals((Object)"21", (Object)new String(new Base16().encode(new byte[]{33})));
        Assertions.assertEquals((Object)"22", (Object)new String(new Base16().encode(new byte[]{34})));
        Assertions.assertEquals((Object)"23", (Object)new String(new Base16().encode(new byte[]{35})));
        Assertions.assertEquals((Object)"24", (Object)new String(new Base16().encode(new byte[]{36})));
        Assertions.assertEquals((Object)"25", (Object)new String(new Base16().encode(new byte[]{37})));
        Assertions.assertEquals((Object)"26", (Object)new String(new Base16().encode(new byte[]{38})));
        Assertions.assertEquals((Object)"27", (Object)new String(new Base16().encode(new byte[]{39})));
        Assertions.assertEquals((Object)"28", (Object)new String(new Base16().encode(new byte[]{40})));
        Assertions.assertEquals((Object)"29", (Object)new String(new Base16().encode(new byte[]{41})));
        Assertions.assertEquals((Object)"2A", (Object)new String(new Base16().encode(new byte[]{42})));
        Assertions.assertEquals((Object)"2B", (Object)new String(new Base16().encode(new byte[]{43})));
        Assertions.assertEquals((Object)"2C", (Object)new String(new Base16().encode(new byte[]{44})));
        Assertions.assertEquals((Object)"2D", (Object)new String(new Base16().encode(new byte[]{45})));
        Assertions.assertEquals((Object)"2E", (Object)new String(new Base16().encode(new byte[]{46})));
        Assertions.assertEquals((Object)"2F", (Object)new String(new Base16().encode(new byte[]{47})));
        Assertions.assertEquals((Object)"30", (Object)new String(new Base16().encode(new byte[]{48})));
        Assertions.assertEquals((Object)"31", (Object)new String(new Base16().encode(new byte[]{49})));
        Assertions.assertEquals((Object)"32", (Object)new String(new Base16().encode(new byte[]{50})));
        Assertions.assertEquals((Object)"33", (Object)new String(new Base16().encode(new byte[]{51})));
        Assertions.assertEquals((Object)"34", (Object)new String(new Base16().encode(new byte[]{52})));
        Assertions.assertEquals((Object)"35", (Object)new String(new Base16().encode(new byte[]{53})));
        Assertions.assertEquals((Object)"36", (Object)new String(new Base16().encode(new byte[]{54})));
        Assertions.assertEquals((Object)"37", (Object)new String(new Base16().encode(new byte[]{55})));
        Assertions.assertEquals((Object)"38", (Object)new String(new Base16().encode(new byte[]{56})));
        Assertions.assertEquals((Object)"39", (Object)new String(new Base16().encode(new byte[]{57})));
        Assertions.assertEquals((Object)"3A", (Object)new String(new Base16().encode(new byte[]{58})));
        Assertions.assertEquals((Object)"3B", (Object)new String(new Base16().encode(new byte[]{59})));
        Assertions.assertEquals((Object)"3C", (Object)new String(new Base16().encode(new byte[]{60})));
        Assertions.assertEquals((Object)"3D", (Object)new String(new Base16().encode(new byte[]{61})));
        Assertions.assertEquals((Object)"3E", (Object)new String(new Base16().encode(new byte[]{62})));
        Assertions.assertEquals((Object)"3F", (Object)new String(new Base16().encode(new byte[]{63})));
        Assertions.assertEquals((Object)"40", (Object)new String(new Base16().encode(new byte[]{64})));
        Assertions.assertEquals((Object)"41", (Object)new String(new Base16().encode(new byte[]{65})));
        Assertions.assertEquals((Object)"42", (Object)new String(new Base16().encode(new byte[]{66})));
        Assertions.assertEquals((Object)"43", (Object)new String(new Base16().encode(new byte[]{67})));
        Assertions.assertEquals((Object)"44", (Object)new String(new Base16().encode(new byte[]{68})));
        Assertions.assertEquals((Object)"45", (Object)new String(new Base16().encode(new byte[]{69})));
        Assertions.assertEquals((Object)"46", (Object)new String(new Base16().encode(new byte[]{70})));
        Assertions.assertEquals((Object)"47", (Object)new String(new Base16().encode(new byte[]{71})));
        Assertions.assertEquals((Object)"48", (Object)new String(new Base16().encode(new byte[]{72})));
        Assertions.assertEquals((Object)"49", (Object)new String(new Base16().encode(new byte[]{73})));
        Assertions.assertEquals((Object)"4A", (Object)new String(new Base16().encode(new byte[]{74})));
        Assertions.assertEquals((Object)"4B", (Object)new String(new Base16().encode(new byte[]{75})));
        Assertions.assertEquals((Object)"4C", (Object)new String(new Base16().encode(new byte[]{76})));
        Assertions.assertEquals((Object)"4D", (Object)new String(new Base16().encode(new byte[]{77})));
        Assertions.assertEquals((Object)"4E", (Object)new String(new Base16().encode(new byte[]{78})));
        Assertions.assertEquals((Object)"4F", (Object)new String(new Base16().encode(new byte[]{79})));
        Assertions.assertEquals((Object)"50", (Object)new String(new Base16().encode(new byte[]{80})));
        Assertions.assertEquals((Object)"51", (Object)new String(new Base16().encode(new byte[]{81})));
        Assertions.assertEquals((Object)"52", (Object)new String(new Base16().encode(new byte[]{82})));
        Assertions.assertEquals((Object)"53", (Object)new String(new Base16().encode(new byte[]{83})));
        Assertions.assertEquals((Object)"54", (Object)new String(new Base16().encode(new byte[]{84})));
        Assertions.assertEquals((Object)"55", (Object)new String(new Base16().encode(new byte[]{85})));
        Assertions.assertEquals((Object)"56", (Object)new String(new Base16().encode(new byte[]{86})));
        Assertions.assertEquals((Object)"57", (Object)new String(new Base16().encode(new byte[]{87})));
        Assertions.assertEquals((Object)"58", (Object)new String(new Base16().encode(new byte[]{88})));
        Assertions.assertEquals((Object)"59", (Object)new String(new Base16().encode(new byte[]{89})));
        Assertions.assertEquals((Object)"5A", (Object)new String(new Base16().encode(new byte[]{90})));
        Assertions.assertEquals((Object)"5B", (Object)new String(new Base16().encode(new byte[]{91})));
        Assertions.assertEquals((Object)"5C", (Object)new String(new Base16().encode(new byte[]{92})));
        Assertions.assertEquals((Object)"5D", (Object)new String(new Base16().encode(new byte[]{93})));
        Assertions.assertEquals((Object)"5E", (Object)new String(new Base16().encode(new byte[]{94})));
        Assertions.assertEquals((Object)"5F", (Object)new String(new Base16().encode(new byte[]{95})));
        Assertions.assertEquals((Object)"60", (Object)new String(new Base16().encode(new byte[]{96})));
        Assertions.assertEquals((Object)"61", (Object)new String(new Base16().encode(new byte[]{97})));
        Assertions.assertEquals((Object)"62", (Object)new String(new Base16().encode(new byte[]{98})));
        Assertions.assertEquals((Object)"63", (Object)new String(new Base16().encode(new byte[]{99})));
        Assertions.assertEquals((Object)"64", (Object)new String(new Base16().encode(new byte[]{100})));
        Assertions.assertEquals((Object)"65", (Object)new String(new Base16().encode(new byte[]{101})));
        Assertions.assertEquals((Object)"66", (Object)new String(new Base16().encode(new byte[]{102})));
        Assertions.assertEquals((Object)"67", (Object)new String(new Base16().encode(new byte[]{103})));
        Assertions.assertEquals((Object)"68", (Object)new String(new Base16().encode(new byte[]{104})));
        for (int i = -128; i <= 127; ++i) {
            byte[] test = new byte[]{(byte)i};
            Assertions.assertArrayEquals((byte[])test, (byte[])new Base16().decode(new Base16().encode(test)));
        }
    }

    @Test
    void testStrictDecoding() {
        String encoded = "aabbccdde";
        Base16 b16 = new Base16(true, CodecPolicy.STRICT);
        Assertions.assertEquals((Object)CodecPolicy.STRICT, (Object)b16.getCodecPolicy());
        Assertions.assertThrows(IllegalArgumentException.class, () -> b16.decode(StringUtils.getBytesUtf8((String)"aabbccdde")));
    }

    @Test
    void testStringToByteVariations() throws DecoderException {
        Base16 base16 = new Base16();
        String s1 = "48656C6C6F20576F726C64";
        String s2 = "";
        String s3 = null;
        Assertions.assertEquals((Object)"Hello World", (Object)StringUtils.newStringUtf8((byte[])base16.decode("48656C6C6F20576F726C64")), (String)"StringToByte Hello World");
        Assertions.assertEquals((Object)"Hello World", (Object)StringUtils.newStringUtf8((byte[])((byte[])new Base16().decode((Object)"48656C6C6F20576F726C64"))), (String)"StringToByte Hello World");
        Assertions.assertEquals((Object)"Hello World", (Object)StringUtils.newStringUtf8((byte[])new Base16().decode("48656C6C6F20576F726C64")), (String)"StringToByte static Hello World");
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUtf8((byte[])new Base16().decode("")), (String)"StringToByte \"\"");
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUtf8((byte[])new Base16().decode("")), (String)"StringToByte static \"\"");
        Assertions.assertNull((Object)StringUtils.newStringUtf8((byte[])new Base16().decode(s3)), (String)"StringToByte null");
        Assertions.assertNull((Object)StringUtils.newStringUtf8((byte[])new Base16().decode(s3)), (String)"StringToByte static null");
    }

    @Test
    void testTriplets() {
        Assertions.assertEquals((Object)"000000", (Object)new String(new Base16().encode(new byte[]{0, 0, 0})));
        Assertions.assertEquals((Object)"000001", (Object)new String(new Base16().encode(new byte[]{0, 0, 1})));
        Assertions.assertEquals((Object)"000002", (Object)new String(new Base16().encode(new byte[]{0, 0, 2})));
        Assertions.assertEquals((Object)"000003", (Object)new String(new Base16().encode(new byte[]{0, 0, 3})));
        Assertions.assertEquals((Object)"000004", (Object)new String(new Base16().encode(new byte[]{0, 0, 4})));
        Assertions.assertEquals((Object)"000005", (Object)new String(new Base16().encode(new byte[]{0, 0, 5})));
        Assertions.assertEquals((Object)"000006", (Object)new String(new Base16().encode(new byte[]{0, 0, 6})));
        Assertions.assertEquals((Object)"000007", (Object)new String(new Base16().encode(new byte[]{0, 0, 7})));
        Assertions.assertEquals((Object)"000008", (Object)new String(new Base16().encode(new byte[]{0, 0, 8})));
        Assertions.assertEquals((Object)"000009", (Object)new String(new Base16().encode(new byte[]{0, 0, 9})));
        Assertions.assertEquals((Object)"00000A", (Object)new String(new Base16().encode(new byte[]{0, 0, 10})));
        Assertions.assertEquals((Object)"00000B", (Object)new String(new Base16().encode(new byte[]{0, 0, 11})));
        Assertions.assertEquals((Object)"00000C", (Object)new String(new Base16().encode(new byte[]{0, 0, 12})));
        Assertions.assertEquals((Object)"00000D", (Object)new String(new Base16().encode(new byte[]{0, 0, 13})));
        Assertions.assertEquals((Object)"00000E", (Object)new String(new Base16().encode(new byte[]{0, 0, 14})));
        Assertions.assertEquals((Object)"00000F", (Object)new String(new Base16().encode(new byte[]{0, 0, 15})));
    }

    private String toString(byte[] data) {
        StringBuilder buf = new StringBuilder();
        for (int i = 0; i < data.length; ++i) {
            buf.append(data[i]);
            if (i == data.length - 1) continue;
            buf.append(",");
        }
        return buf.toString();
    }
}

