/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.store.file.utils;

import java.io.IOException;
import java.util.UUID;
import java.util.function.BinaryOperator;
import javax.annotation.Nullable;
import org.apache.flink.core.fs.FileSystem;
import org.apache.flink.core.fs.Path;
import org.apache.flink.table.store.file.Snapshot;
import org.apache.flink.table.store.file.utils.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotManager {
    private static final Logger LOG = LoggerFactory.getLogger(SnapshotManager.class);
    private static final String SNAPSHOT_PREFIX = "snapshot-";
    public static final String EARLIEST = "EARLIEST";
    public static final String LATEST = "LATEST";
    private final Path tablePath;

    public SnapshotManager(Path tablePath) {
        this.tablePath = tablePath;
    }

    public Path snapshotDirectory() {
        return new Path(this.tablePath + "/snapshot");
    }

    public Path snapshotPath(long snapshotId) {
        return new Path(this.tablePath + "/snapshot/" + SNAPSHOT_PREFIX + snapshotId);
    }

    public Snapshot snapshot(long snapshotId) {
        return Snapshot.fromPath(this.snapshotPath(snapshotId));
    }

    public boolean snapshotExists(long snapshotId) {
        Path path = this.snapshotPath(snapshotId);
        try {
            return path.getFileSystem().exists(path);
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to determine if snapshot #" + snapshotId + " exists in path " + path, e);
        }
    }

    @Nullable
    public Long latestSnapshotId() {
        try {
            return this.findLatest();
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to find latest snapshot id", e);
        }
    }

    public Long findLatest() throws IOException {
        long nextSnapshot;
        Path snapshotDir = this.snapshotDirectory();
        FileSystem fs = snapshotDir.getFileSystem();
        if (!fs.exists(snapshotDir)) {
            return null;
        }
        Long snapshotId = this.readHint(LATEST);
        if (snapshotId != null && !this.snapshotExists(nextSnapshot = snapshotId + 1L)) {
            return snapshotId;
        }
        return this.findByListFiles(Math::max);
    }

    public Long findEarliest() throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        FileSystem fs = snapshotDir.getFileSystem();
        if (!fs.exists(snapshotDir)) {
            return null;
        }
        Long snapshotId = this.readHint(EARLIEST);
        if (snapshotId != null && this.snapshotExists(snapshotId)) {
            return snapshotId;
        }
        return this.findByListFiles(Math::min);
    }

    public Long readHint(String fileName) {
        Path snapshotDir = this.snapshotDirectory();
        Path path = new Path(snapshotDir, fileName);
        try {
            if (path.getFileSystem().exists(path)) {
                return Long.parseLong(FileUtils.readFileUtf8(path));
            }
        }
        catch (Exception e) {
            LOG.info("Failed to read hint file " + fileName + ". Falling back to listing files.", (Throwable)e);
        }
        return null;
    }

    private Long findByListFiles(BinaryOperator<Long> reducer) throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        return FileUtils.listVersionedFiles(snapshotDir, SNAPSHOT_PREFIX).reduce(reducer).orElse(null);
    }

    public void commitLatestHint(long snapshotId) throws IOException {
        this.commitHint(snapshotId, LATEST);
    }

    public void commitEarliestHint(long snapshotId) throws IOException {
        this.commitHint(snapshotId, EARLIEST);
    }

    private void commitHint(long snapshotId, String fileName) throws IOException {
        Path snapshotDir = this.snapshotDirectory();
        FileSystem fs = snapshotDir.getFileSystem();
        Path hintFile = new Path(snapshotDir, fileName);
        Path tempFile = new Path(snapshotDir, UUID.randomUUID() + "-" + fileName + ".temp");
        FileUtils.writeFileUtf8(tempFile, String.valueOf(snapshotId));
        fs.delete(hintFile, false);
        boolean success = fs.rename(tempFile, hintFile);
        if (!success) {
            fs.delete(tempFile, false);
        }
    }
}

