class Builders {
    constructor(builders) {
        this.builders = builders;
    }
    add(builder) {
        this.builders.push(builder);
    }
}
class BaseBuilder {
    constructor(builders) {
        const clone = [];
        builders.builders.forEach((builder) => clone.push(builder));
        this.builders = new Builders(clone);
    }
    build() {
        const path = this.builders.builders
            .map((builder) => builder.segment())
            .filter((segment) => segment !== "")
            .join(".");
        return { path: path };
    }
}
export const Builder = () => {
    return new PMMLBuilder();
};
class PMMLBuilder extends BaseBuilder {
    constructor() {
        super(new Builders([]));
        this.forHeader = () => {
            return new HeaderBuilder(this.builders);
        };
        this.forDataDictionary = () => {
            return new DataDictionaryBuilder(this.builders);
        };
        this.forModel = (modelIndex) => {
            return new ModelBuilder(this.builders, modelIndex);
        };
        this.builders.add(this);
    }
    segment() {
        return "";
    }
}
class ModelBuilder extends BaseBuilder {
    constructor(builders, modelIndex) {
        super(builders);
        this.forBaselineScore = () => {
            return new BaselineScoreBuilder(this.builders);
        };
        this.forUseReasonCodes = () => {
            return new UseReasonCodesBuilder(this.builders);
        };
        this.forCharacteristics = () => {
            return new CharacteristicsBuilder(this.builders);
        };
        this.forMiningSchema = () => {
            return new MiningSchemaBuilder(this.builders);
        };
        this.forOutput = () => {
            return new OutputBuilder(this.builders);
        };
        this.builders.add(this);
        this.modelIndex = modelIndex;
    }
    segment() {
        return this.modelIndex !== undefined ? `models[${this.modelIndex}]` : `models`;
    }
}
class HeaderBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `Header`;
    }
}
class DataDictionaryBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.forDataField = (dataFieldIndex) => {
            return new DataFieldBuilder(this.builders, dataFieldIndex);
        };
        this.builders.add(this);
    }
    segment() {
        return `DataDictionary`;
    }
}
class DataFieldBuilder extends BaseBuilder {
    constructor(builders, dataFieldIndex) {
        super(builders);
        this.forInterval = (intervalIndex) => {
            return new IntervalBuilder(this.builders, intervalIndex);
        };
        this.forValue = (valueIndex) => {
            return new ValueBuilder(this.builders, valueIndex);
        };
        this.builders.add(this);
        this.dataFieldIndex = dataFieldIndex;
    }
    segment() {
        return this.dataFieldIndex !== undefined ? `DataField[${this.dataFieldIndex}]` : `DataField`;
    }
}
class IntervalBuilder extends BaseBuilder {
    constructor(builders, intervalIndex) {
        super(builders);
        this.builders.add(this);
        this.intervalIndex = intervalIndex;
    }
    segment() {
        return this.intervalIndex !== undefined ? `Interval[${this.intervalIndex}]` : `Interval`;
    }
}
class ValueBuilder extends BaseBuilder {
    constructor(builders, valueIndex) {
        super(builders);
        this.builders.add(this);
        this.valueIndex = valueIndex;
    }
    segment() {
        return this.valueIndex !== undefined ? `Value[${this.valueIndex}]` : `Value`;
    }
}
class CharacteristicsBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.forCharacteristic = (characteristicIndex) => {
            return new CharacteristicBuilder(this.builders, characteristicIndex);
        };
        this.builders.add(this);
    }
    segment() {
        return `Characteristics`;
    }
}
class CharacteristicBuilder extends BaseBuilder {
    constructor(builders, characteristicIndex) {
        super(builders);
        this.forReasonCode = () => {
            return new ReasonCodeBuilder(this.builders);
        };
        this.forBaselineScore = () => {
            return new BaselineScoreBuilder(this.builders);
        };
        this.forAttribute = (attributeIndex) => {
            return new AttributeBuilder(this.builders, attributeIndex);
        };
        this.builders.add(this);
        this.characteristicIndex = characteristicIndex;
    }
    segment() {
        return this.characteristicIndex !== undefined ? `Characteristic[${this.characteristicIndex}]` : `Characteristic`;
    }
}
class ReasonCodeBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `reasonCode`;
    }
}
class BaselineScoreBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `baselineScore`;
    }
}
class UseReasonCodesBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `useReasonCodes`;
    }
}
class AttributeBuilder extends BaseBuilder {
    constructor(builders, attributeIndex) {
        super(builders);
        this.forPredicate = (predicateIndex) => {
            return new PredicateBuilder(this.builders, predicateIndex);
        };
        this.forReasonCode = () => {
            return new ReasonCodeBuilder(this.builders);
        };
        this.forPartialScore = () => {
            return new PartialScoreBuilder(this.builders);
        };
        this.builders.add(this);
        this.attributeIndex = attributeIndex;
    }
    segment() {
        return this.attributeIndex !== undefined ? `Attribute[${this.attributeIndex}]` : `Attribute`;
    }
}
class PartialScoreBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `partialScore`;
    }
}
export class PredicateBuilder extends BaseBuilder {
    constructor(builders, predicateIndex) {
        super(builders);
        this.forFieldName = () => {
            return new FieldNameBuilder(this.builders);
        };
        this.forPredicate = (predicateIndex) => {
            return new PredicateBuilder(this.builders, predicateIndex);
        };
        this.builders.add(this);
        this.predicateIndex = predicateIndex;
    }
    segment() {
        return this.predicateIndex !== undefined ? `predicates[${this.predicateIndex}]` : `predicate`;
    }
}
class FieldNameBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `fieldName`;
    }
}
class MiningSchemaBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.forMiningField = (miningFieldIndex) => {
            return new MiningFieldBuilder(this.builders, miningFieldIndex);
        };
        this.builders.add(this);
    }
    segment() {
        return `MiningSchema`;
    }
}
class MiningFieldBuilder extends BaseBuilder {
    constructor(builders, miningFieldIndex) {
        super(builders);
        this.forImportance = () => {
            return new MiningFieldImportanceBuilder(this.builders);
        };
        this.forLowValue = () => {
            return new MiningFieldLowValueBuilder(this.builders);
        };
        this.forHighValue = () => {
            return new MiningFieldHighValueBuilder(this.builders);
        };
        this.forMissingValueReplacement = () => {
            return new MiningFieldMissingValueReplacementBuilder(this.builders);
        };
        this.forInvalidValueReplacement = () => {
            return new MiningFieldInvalidValueReplacementBuilder(this.builders);
        };
        this.forDataFieldMissing = () => {
            return new MiningFieldDataFieldMissingBuilder(this.builders);
        };
        this.builders.add(this);
        this.miningFieldIndex = miningFieldIndex;
    }
    segment() {
        return this.miningFieldIndex !== undefined ? `MiningField[${this.miningFieldIndex}]` : `MiningField`;
    }
}
class MiningFieldImportanceBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `importance`;
    }
}
class MiningFieldLowValueBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `lowValue`;
    }
}
class MiningFieldHighValueBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `highValue`;
    }
}
class MiningFieldMissingValueReplacementBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `missingValueReplacement`;
    }
}
class MiningFieldInvalidValueReplacementBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `invalidValueReplacement`;
    }
}
class MiningFieldDataFieldMissingBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `dataFieldMissing`;
    }
}
class OutputBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.forOutputField = (outputFieldIndex) => {
            return new OutputFieldBuilder(this.builders, outputFieldIndex);
        };
        this.builders.add(this);
    }
    segment() {
        return `Output`;
    }
}
class OutputFieldBuilder extends BaseBuilder {
    constructor(builders, outputFieldIndex) {
        super(builders);
        this.forTargetField = () => {
            return new OutputFieldTargetFieldBuilder(this.builders);
        };
        this.builders.add(this);
        this.outputFieldIndex = outputFieldIndex;
    }
    segment() {
        return this.outputFieldIndex !== undefined ? `OutputField[${this.outputFieldIndex}]` : `OutputField`;
    }
}
class OutputFieldTargetFieldBuilder extends BaseBuilder {
    constructor(builders) {
        super(builders);
        this.builders.add(this);
    }
    segment() {
        return `targetField`;
    }
}
//# sourceMappingURL=PathBuilders.js.map