import * as vscode from "vscode";
import * as fs from "fs";
import * as path from "path";
import { generateFormCode } from "@kie-tools/form-code-generator/dist/generateFormCode";
import { removeInvalidVarChars } from "@kie-tools/jbpm-form-code-generator-themes/dist/removeInvalidVarChars";
import { jbpmBootstrap4FormCodeGeneratorTheme } from "@kie-tools/jbpm-form-code-generator-themes/dist/jbpmBootstrap4FormCodeGeneratorTheme";
import { jbpmPatternflyFormCodeGeneratorTheme } from "@kie-tools/jbpm-form-code-generator-themes/dist/jbpmPatternflyFormCodeGeneratorTheme";
import { PATTERNFLY_FILE_EXT } from "@kie-tools/form-code-generator-patternfly-theme/dist/theme";
import { BOOTSTRAP4_FILE_EXT } from "@kie-tools/form-code-generator-bootstrap4-theme/dist/theme";
const FORM_CODE_GENERATION_DEST_PATH = "src/main/resources/custom-forms-dev";
const JSON_SCHEMA_PATH = "target/classes/META-INF/jsonSchema";
const BOOTSTRAP_UI_LIBRARY_NAME = "Bootstrap 4";
const PATTERNFLY_UI_LIBRARY_NAME = "PatternFly";
export async function generateFormsCommand() {
    const defaultPath = vscode.workspace.workspaceFolders ? vscode.workspace.workspaceFolders[0].uri.fsPath : undefined;
    const projectUri = await vscode.window.showOpenDialog({
        canSelectFiles: false,
        canSelectFolders: true,
        canSelectMany: false,
        title: "Select project folder",
        defaultUri: defaultPath ? vscode.Uri.file(defaultPath) : undefined,
    });
    if (projectUri === undefined || projectUri[0] === undefined) {
        return;
    }
    const projectPath = projectUri[0].fsPath;
    if (fs.existsSync(`${projectPath}/target`) === false) {
        vscode.window.showErrorMessage(`Couldn't find project's "target" folder. Please install your project before using this command.`);
        return;
    }
    if (fs.existsSync(`${projectPath}/${JSON_SCHEMA_PATH}`) === false) {
        vscode.window.showErrorMessage(`Couldn't find any JSON Schema, did you install your project ("mvn clean install")?`);
        return;
    }
    const uiLibrary = await vscode.window.showQuickPick([BOOTSTRAP_UI_LIBRARY_NAME, PATTERNFLY_UI_LIBRARY_NAME], {
        placeHolder: "Select the UI library for the generated form(s)",
    });
    if (uiLibrary === undefined) {
        return;
    }
    const formSchemas = await getFormSchemas(projectPath);
    const existingFiles = [];
    for (const { name: fileName } of formSchemas) {
        if (fs.existsSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}/${removeInvalidVarChars(path.parse(fileName).name)}.${PATTERNFLY_FILE_EXT}`)) {
            existingFiles.push({ fileName, ext: PATTERNFLY_FILE_EXT });
        }
        if (fs.existsSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}/${removeInvalidVarChars(path.parse(fileName).name)}.${BOOTSTRAP4_FILE_EXT}`)) {
            existingFiles.push({ fileName, ext: BOOTSTRAP4_FILE_EXT });
        }
    }
    if (existingFiles.length === 0) {
        saveFormCode(projectPath, uiLibrary, formSchemas);
        return;
    }
    const shouldOverride = await vscode.window.showQuickPick(["Override", "Cancel"], {
        placeHolder: "You already have custom forms in this project. Do you want to override them?",
    });
    if (shouldOverride === "Override") {
        existingFiles.forEach(({ fileName, ext }) => {
            fs.rmSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}/${removeInvalidVarChars(path.parse(fileName).name)}.${ext}`);
        });
        saveFormCode(projectPath, uiLibrary, formSchemas);
    }
    return;
}
async function getFormSchemas(projectPath) {
    const GENERATE_FOR_ALL_HUMAN_INTERACTIONS = "Generate form code for all User Tasks";
    const GENERATE_FOR_SPECIFIC_HUMAN_INTERACTIONS = "Generate form code for specific User Tasks";
    const generationChoice = await vscode.window.showQuickPick([GENERATE_FOR_ALL_HUMAN_INTERACTIONS, GENERATE_FOR_SPECIFIC_HUMAN_INTERACTIONS], {
        placeHolder: "Select an option",
    });
    const jsonSchemaFilesName = fs.readdirSync(`${projectPath}/${JSON_SCHEMA_PATH}`);
    if (generationChoice === GENERATE_FOR_SPECIFIC_HUMAN_INTERACTIONS) {
        const selectedOptions = await vscode.window.showQuickPick(jsonSchemaFilesName.map((jsonSchemaFile) => ({
            label: path.parse(jsonSchemaFile).name,
        })), {
            canPickMany: true,
            placeHolder: "Choose the User Tasks",
        });
        if (selectedOptions === undefined || selectedOptions.length === 0) {
            return [];
        }
        return readAndParseJsonSchemas(projectPath, selectedOptions.reduce((jsonSchemaFilesName, option) => option.label ? [...jsonSchemaFilesName, `${option.label}.json`] : jsonSchemaFilesName, []));
    }
    if (generationChoice === GENERATE_FOR_ALL_HUMAN_INTERACTIONS) {
        return readAndParseJsonSchemas(projectPath, jsonSchemaFilesName);
    }
    return [];
}
function readAndParseJsonSchemas(projectPath, jsonSchemaFilesName) {
    const formSchemas = [];
    const failedParsingFilesName = [];
    for (const jsonSchemaFileName of jsonSchemaFilesName) {
        try {
            formSchemas.push({
                name: path.parse(jsonSchemaFileName).name,
                schema: JSON.parse(fs.readFileSync(`${projectPath}/${JSON_SCHEMA_PATH}/${jsonSchemaFileName}`, "utf-8")),
            });
        }
        catch (error) {
            console.error(`Error while parsing ${jsonSchemaFileName}:`, error);
            failedParsingFilesName.push(jsonSchemaFileName);
        }
    }
    if (failedParsingFilesName.length > 0) {
        vscode.window.showErrorMessage("JSON Schema parsing failed for the following files:", failedParsingFilesName.join(", "));
    }
    return formSchemas;
}
function saveFormCode(projectPath, uiLibrary, formSchemas) {
    const formCode = uiLibrary.toLowerCase() === BOOTSTRAP_UI_LIBRARY_NAME.toLowerCase()
        ? generateFormCode({ formCodeGeneratorTheme: jbpmBootstrap4FormCodeGeneratorTheme, formSchemas })
        : uiLibrary.toLowerCase() === PATTERNFLY_UI_LIBRARY_NAME.toLowerCase()
            ? generateFormCode({ formCodeGeneratorTheme: jbpmPatternflyFormCodeGeneratorTheme, formSchemas })
            : undefined;
    if (formCode === undefined) {
        vscode.window.showErrorMessage(`The "${uiLibrary}" UI library isn't available.`);
        return undefined;
    }
    const formAssets = formCode.reduce((acc, { formAsset }) => (formAsset !== undefined ? [...acc, formAsset] : acc), []);
    if (formAssets.length > 0) {
        if (fs.existsSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}`) === false) {
            fs.mkdirSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}`);
        }
        formAssets.forEach((formAsset) => {
            fs.writeFileSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}/${formAsset.fileNameWithoutInvalidVarChars}`, formAsset.content);
            fs.writeFileSync(`${projectPath}/${FORM_CODE_GENERATION_DEST_PATH}/${formAsset.nameWithoutInvalidTsVarChars}.config`, JSON.stringify(formAsset.config, null, 2));
        });
        vscode.window.showInformationMessage(`Success generating form code for the following files: ${formAssets.map((formAsset) => formAsset.fileName).join(", ")}`);
    }
    const formErrors = formCode.reduce((acc, { formError }) => ((formError === null || formError === void 0 ? void 0 : formError.error) !== undefined ? [...acc, formError] : acc), []);
    if (formErrors.length > 0) {
        vscode.window.showErrorMessage(`Error generating form code for the following files: ${formErrors.map((formError) => formError.fileName).join(", ")}`);
    }
}
//# sourceMappingURL=generateFormCodeCommand.js.map