/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.plc4x.java.api.model;

import org.apache.plc4x.java.api.types.PlcValueType;

import java.util.Collections;
import java.util.List;

/**
 * Base type for all tag types.
 * Typically, every driver provides an implementation of this interface in order
 * to be able to describe the tags of a resource. As this is completely tied to
 * the implemented protocol, this base interface makes absolutely no assumption to
 * any information it should provide.
 *
 * In order to stay platform and protocol independent every driver connection implementation
 * provides a parseTagAddress(String) method that is able to parse a string representation of
 * a resource into it's individual tag type. Manually constructing PlcTag objects
 * manually makes the solution less independent of the protocol, but might be faster.
 */
public interface PlcTag {

    /**
     * Returns the address string, that this tag would be parsed from.
     *
     * @return Address string representing this Tag
     */
    String getAddressString();

    /**
     * Returns the "datatype" of the response one can expect from this tag.
     * I.e. The mapping between this string and the PlcValue datatype is handled in the ValueHandler class.
     *
     * The contract is to return a String description of the datatype. This doesn't necessarily
     * define the PlcValue type but should be related.
     * i.e. returns "BOOL" -> PlcBOOL, returns "INT16" -> PlcINT
     * returning an empty string results in the default handler for that datatype to be evaluated.
     *
     * @return The data type is generally parsed to the PlcTag class, if not a default datatype is returned.
     */
    default PlcValueType getPlcValueType() {
        return PlcValueType.NULL;
    }

    /**
     * Returns the number of elements to expect of the response one can expect from this field.
     *
     * @return The number of elements to expect.
     */
    default List<ArrayInfo> getArrayInfo() {
        return Collections.emptyList();
    }

}
