/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy;

import java.io.IOException;
import java.io.InputStream;
import java.util.Enumeration;

import javax.servlet.http.Cookie;

import org.apache.portals.applications.webcontent2.rewriter.Sink;

/**
 * HTTP request/response abstraction needed for reverse proxy processing.
 * <p>
 * In normal servlet runtime, this may represent parts of 
 * <code>javax.servlet.http.HttpServletRequest</code> and <code>javax.servlet.http.HttpServletResponse</code>.
 * But, in portlet runtime, this may represent parts of 
 * <code>javax.portlet.PortletRequest</code> and <code>javax.portlet.PortletResponse</code>.
 * </p>
 */
public interface RequestContext
{

    /**
     * Returns a boolean indicating whether this request was made using a secure channel,
     * such as HTTPS.
     * @return
     * @see {@link javax.servlet.ServletRequest#isSecure()}.
     * @see {@link javax.portlet.PortletRequest#isSecure()}.
     */
    public boolean isSecure();

    /**
     * Returns the scheme of the request URI.
     * @return
     * @see {@link javax.servlet.ServletRequest#getScheme()}
     * @see {@link javax.portlet.PortletRequest#getScheme()}.
     */
    public String getScheme();

    /**
     * Returns the host name of the server to which the request was sent.
     * It is the value of the part before ":" in the Host header value, if any, or
     * the resolved server name, or the server IP address.
     * @return
     * @see {@link javax.servlet.ServletRequest#getServerName()}
     * @see {@link javax.portlet.PortletRequest#getServerName()}.
     */
    public String getServerName();

    /**
     * Returns the port number to which the request was sent.
     * It is the value of the part after ":" in the Host header value, if any,
     * or the server port where the client connection was accepted on.
     * @return
     * @see {@link javax.servlet.ServletRequest#getServerPort()}
     * @see {@link javax.portlet.PortletRequest#getServerPort()}.
     */
    public int getServerPort();

    /**
     * Returns the name of the HTTP method with which this request was made,
     * for example, GET, POST, or PUT. Same as the value of the CGI variable REQUEST_METHOD.
     * @return
     * @see {@link javax.servlet.http.HttpServletRequest#getMethod()}
     * @see {@link javax.portlet.ClientDataRequest#getMethod()}
     */
    public String getMethod();

    /**
     * Returns the base relative path of the request before the request path info.
     * <p>
     * Typically, in servlet runtime, it is either the concatenation of
     * the request context path and the servlet path, or the concatenation of
     * the request context path and the servlet filter mapping path prefix.
     * </p>
     * <p>
     * For example, in servlet runtime, if the context path is '/webcontent2',
     * the servlet path is '/rproxy' and the path info is '/a/b/c.html', then
     * this method may return the concatenation of the context path and servlet path,
     * '/webcontent2/rproxy'.
     * <p>
     * <p>
     * Or, if a servlet filter used for reverse proxy in servlet runtime, and
     * if the context path is '/webcontent2', the servlet filter mapping path is '/rproxy/*'
     * and the remaining path info is '/a/b/c.html', then
     * this method may return the concatenation of the context path and filter mapping path prefix,
     * '/webcontent2/rproxy'.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't give a request path info unlike servlet runtime.
     * So, an implementation for that kind of runtime may return an empty string
     * and decide not to use this method.
     * </p>
     * @return
     */
    public String getRequestBasePath();

    /**
     * Returns any extra path information associated with the URL the client sent
     * when it made this request.
     * For example, in servlet runtime, if the relative URL is '/webcontent2/rproxy/a/b/c.html'
     * where the context path is '/webcontent2' and the servlet path is '/rproxy',
     * then this method should return '/a/b/c.html'.
     * @return
     * @see {@link org.apache.portals.applications.webcontent2.portlet.proxy.PortletRequestContext#getRequestBasePath()}
     */
    public String getPathInfo();

    /**
     * Returns the value of the named attribute as an Object, or null if no attribute of the given name exists.
     * @param name
     * @return
     * @see {@link javax.servlet.ServletRequest#getAttribute()}
     * @see {@link javax.portlet.PortletRequest#getAttribute()}
     */
    public Object getAttribute(final String name);

    /**
     * Returns the query string that is contained in the request URL after the path.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't give an HTTP request query string unlike servlet runtime.
     * So, an implementation for that kind of runtime may return null and decide not to use this method.
     * </p>
     * @return
     * @see {@link javax.servlet.http.HttpServletRequest#getQueryString()}
     * @see {@link org.apache.portals.applications.webcontent2.portlet.proxy.PortletRequestContext#getQueryString()}
     */
    public String getQueryString();

    /**
     * Returns an enumeration of all the header names this request contains.
     * If the request has no headers, this method returns an empty enumeration.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't have a method named <code>getHeaderNames()</code> unlike servlet runtime.
     * So, an implementation for portlet runtime may invoke {@link javax.portlet.PortletRequest#getPropertyNames()} internally
     * instead because some portlet container implementations such as Apache Jetspeed-2 may return HTTP header names on the method call.
     * </p>
     * @return
     * @see {@link javax.servlet.http.HttpServletRequest#getHeaderNames()}
     * @see {@link javax.portlet.PortletRequest#getPropertyNames()}
     */
    @SuppressWarnings("rawtypes")
    public Enumeration getHeaderNames();

    /**
     * Returns all the values of the specified request header as an Enumeration of String objects.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't have a method named <code>getHeaders(String)</code> unlike servlet runtime.
     * So, an implementation for portlet runtime may invoke {@link javax.portlet.PortletRequest#getProperties()} internally
     * instead because some portlet container implementations such as Apache Jetspeed-2 may return HTTP headers on the method call.
     * </p>
     * @param name
     * @return
     * @see {@link javax.servlet.http.HttpServletRequest#getHeaders(String)}
     * @see {@link javax.portlet.PortletRequest#getProperties(String)}
     */
    @SuppressWarnings("rawtypes")
    public Enumeration getHeaders(String name);

    /**
     * Returns the value of the specified request header as a String.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't have a method named <code>getHeader(String)</code> unlike servlet runtime.
     * So, an implementation for portlet runtime may invoke {@link javax.portlet.PortletRequest#getProperty(String)} internally
     * instead because some portlet container implementations such as Apache Jetspeed-2 may return HTTP headers on the method call.
     * </p>
     * @param name
     * @return
     * @see {@link javax.servlet.http.HttpServletRequest#getHeader(String)}
     * @see {@link javax.portlet.PortletRequest#getProperty(String)}
     */
    public String getHeader(String name);

    /**
     * Retrieves a <code>java.io.InputStream</code> of the body of the request as binary data.
     * @return
     * @throws IOException
     * @see {@link javax.servlet.ServletRequest#getInputStream()}
     * @see {@link javax.portlet.ClientDataRequest#getPortletInputStream()}
     */
    public InputStream getInputStream() throws IOException;

    /**
     * Sets the status code for this response.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't allow a portlet to set the response status code unlike servlet runtime.
     * So, an implementation for portlet runtime may store it as a member without making any impact by itself.
     * </p>
     * @param sc
     * @see {@link javax.servlet.http.HttpServletResponse#setStatus(int)}
     * @see {@link org.apache.portals.applications.webcontent2.portlet.proxy.PortletRequestContext#setStatus(int)}
     */
    public void setStatus(int sc);

    /**
     * Sends a temporary redirect response to the client using the specified redirect location URL.
     * @param location
     * @throws IOException
     * @see {@link javax.servlet.http.HttpServletResponse#sendRedirect(String)}
     * @see {@link javax.portlet.ActionResponse#sendRedirect(String)}
     */
    public void sendRedirect(String location) throws IOException;

    /**
     * Sets a response header with the given name and value.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't have a method named <code>setHeader(String, String)</code> unlike servlet runtime.
     * So, an implementation for portlet runtime may invoke {@link javax.portlet.PortletRequest#setProperty(String, String)} internally
     * instead because some portlet container implementations such as Apache Jetspeed-2 may return HTTP headers on the method call.
     * </p>
     * @param name
     * @param value
     * @see {@link javax.servlet.http.HttpServletResponse#setHeader(String, String)}
     * @see {@link javax.portlet.PortletResponse#setProperty(String, String)}
     */
    public void setHeader(String name, String value);

    /**
     * Sets a response header with the given name and integer value.
     * <p>
     * <em>Note:</em> Portlet runtime doesn't have a method named <code>setIntHeader(String, int)</code> unlike servlet runtime.
     * So, an implementation for portlet runtime may invoke {@link javax.portlet.PortletRequest#setProperty(String, String)} internally
     * instead because some portlet container implementations such as Apache Jetspeed-2 may return HTTP headers on the method call.
     * </p>
     * @param name
     * @param value
     * @see {@link javax.servlet.http.HttpServletResponse#setIntHeader(String, String)}
     * @see {@link javax.portlet.PortletResponse#setProperty(String, String)}
     */
    public void setIntHeader(String name, int value);

    /**
     * Adds the specified cookie to the response.
     * @param cookie
     * @see {@link javax.servlet.http.HttpServletResponse#addCookie(Cookie)}
     * @see {@link javax.portlet.PortletResponse#addProperty(Cookie)}
     */
    public void addCookie(Cookie cookie);

    /**
     * Returns the {@link Sink} instance used in the current reverse proxy processing.
     * @return
     */
    public Sink getSink();

}
