/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;
import java.net.URI;
import java.util.Date;
import java.util.List;

import javax.servlet.http.Cookie;

import org.apache.commons.chain.Command;
import org.apache.http.HttpResponse;
import org.apache.http.client.CookieStore;
import org.apache.http.client.utils.URIUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;


/**
 * {@link Command} responsible for copying all the original HTTP response cookies
 * from the remote content to the internally created {@link HttpResponse} instance.
 * <p>
 * For example, if it is serving <code>/webcontent2/rproxy/portals/</code>
 * (where <code>/webcontent2/rproxy</code> is the concatenation of the servlet context path and the servlet path)
 * for <code>http://portals.apache.org/</code>
 * and the remote content sets a cookie with cookie path, <code>/</code>, then this method should translate the cookie path (<code>/</code>)
 * to <code>/webcontent2/rproxy/portals/</code> in order not to pollute the cookie paths in the single reverse proxy
 * web application serving multiple remote target web sites.
 * </p>
 */
public class AddCookiesToResponseCommand extends AbstractProxyCommand
{

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        List<org.apache.http.cookie.Cookie> responseCookies = null;

        CookieStore cookieStore = context.getCookieStore();

        if (cookieStore != null)
        {
            responseCookies = cookieStore.getCookies();
        }

        if (responseCookies != null && !responseCookies.isEmpty())
        {
            final boolean isSecureRequest = context.getRequestContext().isSecure();
            String reverseCookiePath = null;

            for (org.apache.http.cookie.Cookie cookie : responseCookies)
            {
                String cookieName = cookie.getName();
                Cookie responseCookie = new Cookie(cookieName, cookie.getValue());
                responseCookie.setSecure(isSecureRequest && cookie.isSecure());
                responseCookie.setVersion(cookie.getVersion());
                responseCookie.setComment(cookie.getComment());
                Date expireDate = cookie.getExpiryDate();

                if (expireDate != null)
                {
                    int maxAgeSeconds = (int) ((expireDate.getTime() - System.currentTimeMillis()) / 1000L);
                    responseCookie.setMaxAge(maxAgeSeconds);
                }

                reverseCookiePath = getReverseCookiePath(context, cookie);

                if (reverseCookiePath != null) {
                    responseCookie.setPath(reverseCookiePath);
                }

                context.getRequestContext().addCookie(responseCookie);
            }
        }

        return false;
    }

    /**
     * Returns the translated cookie path in the reverse proxy side for the original response cookie.
     * <p>
     * For example, if it is serving <code>/webcontent2/rproxy/portals/</code>
     * (where <code>/webcontent2/rproxy</code> is the concatenation of the servlet context path and the servlet path)
     * for <code>http://portals.apache.org/</code>
     * and the remote content sets a cookie with cookie path, <code>/</code>, then this method should translate the cookie path (<code>/</code>)
     * to <code>/webcontent2/rproxy/portals/</code> in order not to pollute the cookie paths in the single reverse proxy
     * web application serving multiple remote target web sites.
     * </p>
     * @param context
     * @param responseCookie
     * @return
     */
    protected String getReverseCookiePath(final ProxyContext context, final org.apache.http.cookie.Cookie responseCookie) 
    {
        String cookiePath = responseCookie.getPath();

        if (cookiePath != null)
        {
            URI remoteURI = context.getRemoteURI();
            URI cookieRemoteURI = URIUtils.resolve(remoteURI, cookiePath);
            cookiePath = context.getResolvedMapping().resolveLocalFromRemote(cookieRemoteURI);
        }

        return cookiePath;
    }
}
