/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.command;

import java.io.IOException;
import java.net.URI;

import javax.servlet.http.HttpServletResponse;

import org.apache.commons.chain.Command;
import org.apache.commons.lang.math.NumberUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpHead;
import org.apache.http.client.methods.HttpOptions;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.apache.http.client.methods.HttpRequestBase;
import org.apache.http.client.methods.HttpTrace;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.protocol.HTTP;
import org.apache.portals.applications.webcontent2.proxy.ProxyContext;
import org.apache.portals.applications.webcontent2.proxy.ReverseProxyException;
import org.apache.portals.applications.webcontent2.proxy.impl.AbstractProxyCommand;


/**
 * {@link Command} responsible for initializing the {@link HttpRequestBase} instance
 * which is used to invoke the resolved remote target URI,
 * based on the original request method name and the resolved remote target URI.
 */
public class InitHttpRequestCommand extends AbstractProxyCommand
{

    /**
     * {@inheritDoc}
     */
    @Override
    protected boolean executeInternal(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpRequestBase httpRequest = createHttpRequest(context);
        context.setHttpRequest(httpRequest);
        return false;
    }

    /**
     * Creates an {@link HttpRequestBase} instance
     * which is used to invoke the resolved remote target URI.
     * @param context
     * @return
     * @throws ReverseProxyException
     * @throws IOException
     */
    protected HttpRequestBase createHttpRequest(final ProxyContext context) throws ReverseProxyException, IOException
    {
        HttpRequestBase httpRequest = null;

        String method = context.getRequestContext().getMethod();
        URI remoteURI = context.getRemoteURI();

        if (HttpGet.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpGet(remoteURI);
        }
        else if (HttpHead.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpHead(remoteURI);
        }
        else if (HttpPost.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpPost(remoteURI);
            long contentLength = NumberUtils.toLong(context.getRequestContext().getHeader(HTTP.CONTENT_LEN));

            if (contentLength > 0L)
            {
                HttpEntity entity = new InputStreamEntity(context.getRequestContext().getInputStream(), contentLength);
                ((HttpPost) httpRequest).setEntity(entity);
            }
        }
        else if (HttpPut.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpPut(remoteURI);

            long contentLength = NumberUtils.toLong(context.getRequestContext().getHeader(HTTP.CONTENT_LEN));

            if (contentLength > 0L)
            {
                HttpEntity entity = new InputStreamEntity(context.getRequestContext().getInputStream(), contentLength);
                ((HttpPut) httpRequest).setEntity(entity);
            }
        }
        else if (HttpDelete.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpDelete(remoteURI);
        }
        else if (HttpOptions.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpOptions(remoteURI);
        }
        else if (HttpTrace.METHOD_NAME.equals(method))
        {
            httpRequest = new HttpHead(remoteURI);
        }
        else
        {
            throw new ReverseProxyException(HttpServletResponse.SC_METHOD_NOT_ALLOWED, "Unsupported method: " + method);
        }

        return httpRequest;
    }
}
