/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.proxy.impl;

import java.net.URI;

import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent2.proxy.ProxyMapping;

/**
 * Simple {@link ProxyMapping} implementation based on path prefix matching.
 * <p>
 * For example, if this represents a mapping between a local path prefix and
 * a remote URI prefix like '/portals/' vs. 'http://portals.apache.org/',
 * then a local request path info, '/portals/a/b/c.html', will be resolved to
 * 'http://portals.apache.org/a/b/c.html' and vice versa.
 * </p>
 */
public class SimpleProxyMapping extends AbstractProxyMapping
{

    /**
     * Local request path info prefix to match/resolve.
     */
    private String localBasePath;

    /**
     * Remote base URI prefix to match/resolve.
     */
    private URI remoteBaseURI;

    /**
     * Zero-argument default constructor.
     */
    public SimpleProxyMapping()
    {
    }

    /**
     * Returns the local request path info prefix.
     * @return
     */
    public String getLocal()
    {
        return localBasePath;
    }

    /**
     * Sets the local request path info prefix.
     * @param localBasePath
     */
    public void setLocal(String localBasePath)
    {
        this.localBasePath = localBasePath;
    }

    /**
     * Returns the remote target URI prefix.
     * @return
     */
    public URI getRemote()
    {
        return remoteBaseURI;
    }

    /**
     * Sets the remote target URI prefix.
     * @param remoteBaseURI
     */
    public void setRemote(URI remoteBaseURI)
    {
        this.remoteBaseURI = remoteBaseURI;
    }

    /**
     * {@inheritDoc}
     */
    public boolean matchesLocal(String localPath)
    {
        return StringUtils.startsWith(localPath, localBasePath);
    }

    /**
     * {@inheritDoc}
     */
    public String resolveRemoteFromLocal(String localPath)
    {
        if (matchesLocal(localPath))
        {
            return remoteBaseURI + localPath.substring(localBasePath.length());
        }

        return null;
    }

    /**
     * {@inheritDoc}
     */
    public boolean matchesRemote(URI remoteURI)
    {
        String remoteURIString = remoteURI.toString();

        if (remoteBaseURI.getScheme() == null && remoteURI.getScheme() != null)
        {
            remoteURIString = remoteURIString.substring(remoteURI.getScheme().length() + 1);
        }

        return StringUtils.startsWith(remoteURIString, remoteBaseURI.toString());
    }

    /**
     * {@inheritDoc}
     */
    public String resolveLocalFromRemote(URI remoteURI)
    {
        if (matchesRemote(remoteURI))
        {
            return localBasePath + StringUtils.substringAfter(remoteURI.toString(), remoteBaseURI.toString());
        }

        return null;
    }

}
