"use strict";
/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphSONReader = exports.GraphSONWriter = exports.GraphSON3Reader = exports.GraphSON2Reader = exports.GraphSON3Writer = exports.GraphSON2Writer = void 0;
/**
 * @author Jorge Bay Gondra
 */
const buffer_1 = require("buffer");
const bytecode_js_1 = __importDefault(require("../../process/bytecode.cjs"));
const type_serializers_js_1 = require("./type-serializers.cjs");
/**
 * GraphSON2 writer.
 */
class GraphSON2Writer {
    options;
    _serializers;
    /**
     * @param {GraphWriterOptions} [options]
     * @param {TypeSerializer} [options.serializers] An object used as an associative array with GraphSON 2 type name as keys and
     * serializer instances as values, ie: { 'g:Int64': longSerializer }.
     * @constructor
     */
    constructor(options = {}) {
        this.options = options;
        // Create instance of the default serializers
        this._serializers = this.getDefaultSerializers().map((serializerConstructor) => {
            const s = new serializerConstructor();
            s.writer = this;
            return s;
        });
        const customSerializers = this.options.serializers || {};
        Object.keys(customSerializers).forEach((key) => {
            const s = customSerializers[key];
            if (!s.serialize) {
                return;
            }
            s.writer = this;
            // Insert custom serializers first
            this._serializers.unshift(s);
        });
    }
    /**
     * Gets the default serializers to be used.
     * @returns {Array}
     */
    getDefaultSerializers() {
        return graphSON2Serializers;
    }
    adaptObject(value) {
        let s;
        for (let i = 0; i < this._serializers.length; i++) {
            const currentSerializer = this._serializers[i];
            if (currentSerializer.canBeUsedFor?.(value)) {
                s = currentSerializer;
                break;
            }
        }
        if (s) {
            return s.serialize(value);
        }
        if (Array.isArray(value)) {
            // We need to handle arrays when there is no serializer
            // for older versions of GraphSON
            return value.map((item) => this.adaptObject(item));
        }
        // Default (strings / objects / ...)
        return value;
    }
    /**
     * Returns the GraphSON representation of the provided object instance.
     * @param {Object} obj
     * @returns {String}
     */
    write(obj) {
        return JSON.stringify(this.adaptObject(obj));
    }
    writeRequest({ requestId, op, processor, args, }) {
        const req = {
            requestId: { '@type': 'g:UUID', '@value': requestId },
            op,
            processor,
            args: this._adaptArgs(args, true),
        };
        if (req.args['gremlin'] instanceof bytecode_js_1.default) {
            req.args['gremlin'] = this.adaptObject(req.args['gremlin']);
        }
        return buffer_1.Buffer.from(JSON.stringify(req));
    }
    /**
     * Takes the given args map and ensures all arguments are passed through to adaptObject
     * @param {Object} args Map of arguments to process.
     * @param {Boolean} protocolLevel Determines whether it's a protocol level binding.
     * @returns {Object}
     * @private
     */
    _adaptArgs(args, protocolLevel) {
        if (args instanceof Object) {
            const newObj = {};
            Object.keys(args).forEach((key) => {
                // bindings key (at the protocol-level needs special handling. without this, it wraps the generated Map
                // in another map for types like EnumValue. Could be a nicer way to do this but for now it's solving the
                // problem with script submission of non JSON native types
                if (protocolLevel && key === 'bindings') {
                    newObj[key] = this._adaptArgs(args[key], false);
                }
                else {
                    newObj[key] = this.adaptObject(args[key]);
                }
            });
            return newObj;
        }
        return args;
    }
}
exports.GraphSON2Writer = GraphSON2Writer;
/**
 * GraphSON3 writer.
 */
class GraphSON3Writer extends GraphSON2Writer {
    getDefaultSerializers() {
        return graphSON3Serializers;
    }
}
exports.GraphSON3Writer = GraphSON3Writer;
/**
 * GraphSON2 reader.
 */
class GraphSON2Reader {
    options;
    _deserializers;
    /**
     * GraphSON Reader
     * @param {GraphReaderOptions} [options]
     * @param {TypeSerializer} [options.serializers] An object used as an associative array with GraphSON 2 type name as keys and
     * deserializer instances as values, ie: { 'g:Int64': longSerializer }.
     * @constructor
     */
    constructor(options = {}) {
        this.options = options;
        this._deserializers = {};
        const defaultDeserializers = this.getDefaultDeserializers();
        Object.keys(defaultDeserializers).forEach((typeName) => {
            const serializerConstructor = defaultDeserializers[typeName];
            const s = new serializerConstructor();
            s.reader = this;
            this._deserializers[typeName] = s;
        });
        if (this.options.serializers) {
            const customSerializers = this.options.serializers || {};
            Object.keys(customSerializers).forEach((key) => {
                const s = customSerializers[key];
                if (!s.deserialize) {
                    return;
                }
                s.reader = this;
                this._deserializers[key] = s;
            });
        }
    }
    /**
     * Gets the default deserializers as an associative array.
     */
    getDefaultDeserializers() {
        return graphSON2Deserializers;
    }
    read(obj) {
        if (obj === undefined) {
            return undefined;
        }
        if (obj === null) {
            return null;
        }
        if (Array.isArray(obj)) {
            return obj.map((item) => this.read(item));
        }
        const type = obj[type_serializers_js_1.typeKey];
        if (type) {
            const d = this._deserializers[type];
            if (d) {
                // Use type serializer
                return d.deserialize(obj);
            }
            return obj[type_serializers_js_1.valueKey];
        }
        if (obj && typeof obj === 'object' && obj.constructor === Object) {
            return this._deserializeObject(obj);
        }
        // Default (for boolean, number and other scalars)
        return obj;
    }
    readResponse(buffer) {
        return this.read(JSON.parse(buffer.toString()));
    }
    _deserializeObject(obj) {
        const keys = Object.keys(obj);
        const result = {};
        for (let i = 0; i < keys.length; i++) {
            // @ts-expect-error
            result[keys[i]] = this.read(obj[keys[i]]);
        }
        return result;
    }
}
exports.GraphSON2Reader = GraphSON2Reader;
/**
 * GraphSON3 reader.
 */
class GraphSON3Reader extends GraphSON2Reader {
    getDefaultDeserializers() {
        return graphSON3Deserializers;
    }
}
exports.GraphSON3Reader = GraphSON3Reader;
const graphSON2Deserializers = {
    'g:Traverser': type_serializers_js_1.TraverserSerializer,
    'g:TraversalStrategy': type_serializers_js_1.TraversalStrategySerializer,
    'g:Int32': type_serializers_js_1.NumberSerializer,
    'g:Int64': type_serializers_js_1.NumberSerializer,
    'g:Float': type_serializers_js_1.NumberSerializer,
    'g:Double': type_serializers_js_1.NumberSerializer,
    'g:Date': type_serializers_js_1.DateSerializer,
    'g:Direction': type_serializers_js_1.DirectionSerializer,
    'g:Vertex': type_serializers_js_1.VertexSerializer,
    'g:Edge': type_serializers_js_1.EdgeSerializer,
    'g:VertexProperty': type_serializers_js_1.VertexPropertySerializer,
    'g:Property': type_serializers_js_1.PropertySerializer,
    'g:Path': type_serializers_js_1.Path3Serializer,
    'g:TextP': type_serializers_js_1.TextPSerializer,
    'g:T': type_serializers_js_1.TSerializer,
    'g:BulkSet': type_serializers_js_1.BulkSetSerializer,
};
const graphSON3Deserializers = Object.assign({}, graphSON2Deserializers, {
    'g:List': type_serializers_js_1.ListSerializer,
    'g:Set': type_serializers_js_1.SetSerializer,
    'g:Map': type_serializers_js_1.MapSerializer,
});
const graphSON2Serializers = [
    type_serializers_js_1.NumberSerializer,
    type_serializers_js_1.DateSerializer,
    type_serializers_js_1.BytecodeSerializer,
    type_serializers_js_1.TraverserSerializer,
    type_serializers_js_1.TraversalStrategySerializer,
    type_serializers_js_1.PSerializer,
    type_serializers_js_1.TextPSerializer,
    type_serializers_js_1.LambdaSerializer,
    type_serializers_js_1.EnumSerializer,
    type_serializers_js_1.VertexSerializer,
    type_serializers_js_1.EdgeSerializer,
    type_serializers_js_1.LongSerializer,
];
// @ts-expect-error
const graphSON3Serializers = graphSON2Serializers.concat([type_serializers_js_1.ListSerializer, type_serializers_js_1.SetSerializer, type_serializers_js_1.MapSerializer]);
exports.GraphSONWriter = GraphSON3Writer;
exports.GraphSONReader = GraphSON3Reader;
