/// <reference types="node" />
/**
 * @author Jorge Bay Gondra
 */
import { Buffer } from 'buffer';
import { BulkSetSerializer, BytecodeSerializer, DateSerializer, DirectionSerializer, EdgeSerializer, EnumSerializer, LambdaSerializer, ListSerializer, LongSerializer, MapSerializer, NumberSerializer, PSerializer, Path3Serializer, PropertySerializer, SetSerializer, TSerializer, TextPSerializer, TraversalStrategySerializer, TraverserSerializer, TypeSerializer, VertexPropertySerializer, VertexSerializer } from './type-serializers.cjs';
export type GraphWriterOptions = {
    serializers?: Record<string, TypeSerializer<any>>;
};
/**
 * GraphSON2 writer.
 */
export declare class GraphSON2Writer {
    private readonly options;
    private readonly _serializers;
    /**
     * @param {GraphWriterOptions} [options]
     * @param {TypeSerializer} [options.serializers] An object used as an associative array with GraphSON 2 type name as keys and
     * serializer instances as values, ie: { 'g:Int64': longSerializer }.
     * @constructor
     */
    constructor(options?: GraphWriterOptions);
    /**
     * Gets the default serializers to be used.
     * @returns {Array}
     */
    getDefaultSerializers(): any[];
    adaptObject<T>(value: T): any;
    /**
     * Returns the GraphSON representation of the provided object instance.
     * @param {Object} obj
     * @returns {String}
     */
    write<T>(obj: T): string;
    writeRequest({ requestId, op, processor, args, }: {
        processor: string | undefined;
        op: string;
        args: any;
        requestId?: string | null;
    }): Buffer;
    /**
     * Takes the given args map and ensures all arguments are passed through to adaptObject
     * @param {Object} args Map of arguments to process.
     * @param {Boolean} protocolLevel Determines whether it's a protocol level binding.
     * @returns {Object}
     * @private
     */
    _adaptArgs<T extends Record<string, any>>(args: T, protocolLevel: boolean): T;
}
/**
 * GraphSON3 writer.
 */
export declare class GraphSON3Writer extends GraphSON2Writer {
    getDefaultSerializers(): (typeof NumberSerializer | typeof DateSerializer | typeof LongSerializer | typeof BytecodeSerializer | typeof PSerializer | typeof LambdaSerializer | typeof EnumSerializer | typeof TraverserSerializer | typeof TraversalStrategySerializer | typeof VertexSerializer | typeof EdgeSerializer)[];
}
export type GraphReaderOptions = {
    serializers?: Record<string, TypeSerializer<any>>;
};
/**
 * GraphSON2 reader.
 */
export declare class GraphSON2Reader {
    private readonly options;
    private readonly _deserializers;
    /**
     * GraphSON Reader
     * @param {GraphReaderOptions} [options]
     * @param {TypeSerializer} [options.serializers] An object used as an associative array with GraphSON 2 type name as keys and
     * deserializer instances as values, ie: { 'g:Int64': longSerializer }.
     * @constructor
     */
    constructor(options?: GraphReaderOptions);
    /**
     * Gets the default deserializers as an associative array.
     */
    getDefaultDeserializers(): {
        'g:Traverser': typeof TraverserSerializer;
        'g:TraversalStrategy': typeof TraversalStrategySerializer;
        'g:Int32': typeof NumberSerializer;
        'g:Int64': typeof NumberSerializer;
        'g:Float': typeof NumberSerializer;
        'g:Double': typeof NumberSerializer;
        'g:Date': typeof DateSerializer;
        'g:Direction': typeof DirectionSerializer;
        'g:Vertex': typeof VertexSerializer;
        'g:Edge': typeof EdgeSerializer;
        'g:VertexProperty': typeof VertexPropertySerializer;
        'g:Property': typeof PropertySerializer;
        'g:Path': typeof Path3Serializer;
        'g:TextP': typeof TextPSerializer;
        'g:T': typeof TSerializer;
        'g:BulkSet': typeof BulkSetSerializer;
    };
    read(obj: any): any;
    readResponse(buffer: Buffer): any;
    _deserializeObject<T extends Record<any, any>>(obj: T): {};
}
/**
 * GraphSON3 reader.
 */
export declare class GraphSON3Reader extends GraphSON2Reader {
    getDefaultDeserializers(): {
        'g:Traverser': typeof TraverserSerializer;
        'g:TraversalStrategy': typeof TraversalStrategySerializer;
        'g:Int32': typeof NumberSerializer;
        'g:Int64': typeof NumberSerializer;
        'g:Float': typeof NumberSerializer;
        'g:Double': typeof NumberSerializer;
        'g:Date': typeof DateSerializer;
        'g:Direction': typeof DirectionSerializer;
        'g:Vertex': typeof VertexSerializer;
        'g:Edge': typeof EdgeSerializer;
        'g:VertexProperty': typeof VertexPropertySerializer;
        'g:Property': typeof PropertySerializer;
        'g:Path': typeof Path3Serializer;
        'g:TextP': typeof TextPSerializer;
        'g:T': typeof TSerializer;
        'g:BulkSet': typeof BulkSetSerializer;
    } & {
        'g:List': typeof ListSerializer;
        'g:Set': typeof SetSerializer;
        'g:Map': typeof MapSerializer;
    };
}
export declare const GraphSONWriter: typeof GraphSON3Writer;
export declare const GraphSONReader: typeof GraphSON3Reader;
