#!BPY

"""
Name: 'Newell Teapot Generator'
Blender: 248
Group: 'Wizards'
Tooltip: 'Adds the famous missing primitive to Blender'
"""

__author__ = "Anthony D'Agostino (Scorpius)"
__url__ = ("blender", "elysiun",
"Author's homepage, http://www.redrival.com/scorpius")
__version__ = "1.0"

__bpydoc__ = """\
This script adds a famous missing primitive: the Newell
Teapot. Most modelers have had this primitive for years, and
now so does Blender. In addition to the teapot, I have taken the liberty
to add the teaspoon and teacup in order to complete the original Newell
teaset.

Usage:<br>
	When the mouse pointer is over the scripts window (in this
case it is a slightly brownish color) the following keyboard shortcuts
are available:

NumPad Plus Key - Increase the resolution in
realtime<br>NumPad Minus Key - Decrease the resolution in realtime

The numbers 1 through 5 on the top row of the keyboard
switch to one of the five predifined bezier patch primitives.

The numbers 1 through 5 on the keypad
switch to one of the five predifined resolutions.

Missing:<br>
	Not much, but other objects in indexed bezier patch format
are easy to add.

Known issues:<br>
	Creates some duplicate vertices. Not by my fault, but because they are
inherent in the ancient patch file format that I found on the net. I
plan to port this to Wings3D when I figure out how to remove duplicate
verts based on a threshold value.

Notes:<br>
	The 'Yoda' model is Copyright (c) 1997 Anto Matkovic (modeled in sPatch).
Used here without permission.
"""

# $Id$
#
# +---------------------------------------------------------+
# | Copyright (c) 2005 Anthony D'Agostino                   |
# | http://www.redrival.com/scorpius                        |
# | scorpius@netzero.com                                    |
# | February 12, 2005                                       |
# | Uses Erlang naming convention, port to Wings3D later    |
# +---------------------------------------------------------+

# ***** BEGIN GPL LICENSE BLOCK *****
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
#
# ***** END GPL LICENCE BLOCK *****

import Blender
import operator, cStringIO

# =================
# === GUI Block ===
# =================
EvtBase,BezBase = 1,Blender.Draw.Create(1)
EvtReso,BezReso = 2,Blender.Draw.Create(0)
EvtText,BezText = 3,Blender.Draw.Create('')

def draw():
	global BezBase, BezReso
	Blender.BGL.glClearColor(0.625, 0.5625, 0.5, 0.0)	# Brown
	Blender.BGL.glClear(Blender.BGL.GL_COLOR_BUFFER_BIT)
	Blender.BGL.glColor3b(0,0,0)
	BezBase = Blender.Draw.Menu("Bezier Object %t | TeaPot %x1 | TeaSpoon %x2 | TeaCup %x3 | OnePatch %x5", EvtBase, 10, 10, 225, 20, BezBase.val, "Select Bezier-Patch Object")
	BezReso = Blender.Draw.Slider("Resolution: ", EvtReso, 10, 35, 225, 20, BezReso.val, 3, 20, 0, "Select Bezier-Patch Resolution")
	Blender.BGL.glRasterPos2i(10, 75)
	Blender.Draw.Text("http://www.redrival.com/scorpius")
	Blender.BGL.glRasterPos2i(10, 90)
	Blender.Draw.Text("Copyright (c) 2005 Anthony D'Agostino")
	Blender.BGL.glRasterPos2i(10, 105)
	Blender.Draw.Text("Newell Teapot Generator")

def key_event(event, value):
	if (event==Blender.Draw.QKEY and not value):
		Blender.Draw.Exit()

	if (event==Blender.Draw.PADPLUSKEY and not value):
		BezReso.val += 1
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PADMINUS and not value):
		BezReso.val -= 1
		make_teapot(BezBase.val, BezReso.val)

	if (event==Blender.Draw.ONEKEY and not value):
		BezBase.val = 1
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.TWOKEY and not value):
		BezBase.val = 2
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.THREEKEY and not value):
		BezBase.val = 3
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.FOURKEY and not value):
		BezBase.val = 4
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.FIVEKEY and not value):
		BezBase.val = 5
		make_teapot(BezBase.val, BezReso.val)

	if (event==Blender.Draw.PAD1 and not value):
		BezReso.val = 3
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD2 and not value):
		BezReso.val = 5
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD3 and not value):
		BezReso.val = 7
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD4 and not value):
		BezReso.val = 9
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD5 and not value):
		BezReso.val = 11
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD6 and not value):
		BezReso.val = 13
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD7 and not value):
		BezReso.val = 15
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD8 and not value):
		BezReso.val = 17
		make_teapot(BezBase.val, BezReso.val)
	if (event==Blender.Draw.PAD9 and not value):
		BezReso.val = 19
		make_teapot(BezBase.val, BezReso.val)

def button_event(eventnumber):
	if eventnumber:
		make_teapot(BezBase.val, BezReso.val)
	else:
		pass
	Blender.Draw.Draw()

Blender.Draw.Register(draw, key_event, button_event)

# ==========================
# === Bezier Patch Block ===
# ==========================
def read_indexed_patch_file(FileName):
	file = cStringIO.StringIO(FileName)
	rawpatches = []
	patches = []
	numpatches = int(file.readline())
	for i in range(numpatches):
		line = file.readline()
		A,B,C,D, E,F,G,H, I,J,K,L, M,N,O,P = map(int, line.split(","))
		patches.append([[A,B,C,D], [E,F,G,H], [I,J,K,L], [M,N,O,P]])
		rawpatches.append([[0,0,0,0], [0,0,0,0], [0,0,0,0], [0,0,0,0]])
	Verts = []
	NumVerts = int(file.readline())
	for i in range(NumVerts):
		line = file.readline()
		V1,V2,V3 = map(float, line.split(","))
		Verts.append((V1,V2,V3))
	for i in range(len(patches)):
		for j in range(4):	#len(patches[i])):
			for k in range(4):	#len(patches[i][j])):
				index = patches[i][j][k] - 1
				rawpatches[i][j][k] = Verts[index]
	return rawpatches

def patches_to_raw(Patches, Resolution):
	Raw = []
	for Patch in Patches:
		Verts = make_verts(Patch, Resolution)
		Faces = make_faces(Resolution)
		RawQuads = indexed_to_rawquads(Verts, Faces)
		Raw.append(RawQuads)
	Raw = reduce(operator.add, Raw)  # flatten the list
	return Raw

def make_bezier(CtrlPnts, Resolution):
	B1 = lambda T: T*T*T
	B2 = lambda T: 3*T * T * (1-T)
	B3 = lambda T: 3*T * (1-T) * (1-T)
	B4 = lambda T: (1-T) * (1-T) * (1-T)
	MakeVec = lambda V: Blender.Mathutils.Vector(V)
	P1,P2,P3,P4 = map(MakeVec, CtrlPnts)
	CurveVerts = []
	for i in range(Resolution+1):
		T = float(i)/Resolution
		X,Y,Z = B1(T)*P1 + B2(T)*P2 + B3(T)*P3 + B4(T)*P4
		CurveVerts.append((X,Y,Z))
	return CurveVerts

def make_bezier(CtrlPnts, Resolution):
	B1 = lambda T: T*T*T
	B2 = lambda T: 3*T * T * (1-T)
	B3 = lambda T: 3*T * (1-T) * (1-T)
	B4 = lambda T: (1-T) * (1-T) * (1-T)
	P1,P2,P3,P4 = map(Blender.Mathutils.Vector, CtrlPnts)
	def MakeVert(T):
		X,Y,Z = B1(T)*P1 + B2(T)*P2 + B3(T)*P3 + B4(T)*P4
		return (X,Y,Z)
	CurveVerts = [MakeVert(float(i)/Resolution) for i in range(Resolution+1)]
	return CurveVerts

def make_verts(A, Resolution):
	S = []
	for i in A:
		C = make_bezier(i, Resolution)
		S.append(C)
	B = transpose(S)
	S = []
	for i in B:
		C = make_bezier(i, Resolution)
		S.append(C)
	Verts = S
	Verts = reduce(operator.add, Verts)  # flatten the list
	return Verts

def make_faces(Resolution):
	n = Resolution+1
	Faces = []
	for i in range(n-1):
		for j in range(n-1):
			v1 = (i+1)*n+j
			v2 = (i+1)*n+j+1
			v3 = i*n+j+1
			v4 = i*n+j
			Faces.append([v1,v2,v3,v4])
	return Faces

def indexed_to_rawquads(Verts, Faces):
	Rows = len(Faces)
	Cols = len(Faces[0])	# or 4
	RawQuads = [[None]*Cols for i in range(Rows)]
	for i in range(Rows):
		for j in range(Cols):
			Index = Faces[i][j]
			RawQuads[i][j] = Verts[Index]
	return RawQuads

def raw_to_indexed(RawFaces): # Generate verts and faces lists, without dups
	Verts = []
	Coords = {}
	Index = 0
	for i in range(len(RawFaces)):
		for j in range(len(RawFaces[i])):
			vertex = RawFaces[i][j]
			if not Coords.has_key(vertex):
				Coords[vertex] = Index
				Index += 1
				Verts.append(vertex)
			RawFaces[i][j] = Coords[vertex]
	return Verts, RawFaces

def transpose(RowsByCols):
	Rows = len(RowsByCols)
	Cols = len(RowsByCols[0])
	ColsByRows = [[None]*Rows for i in range(Cols)]
	for i in range(Cols):
		for j in range(Rows):
			ColsByRows[i][j] = RowsByCols[j][i]
	return ColsByRows

def make_teapot(FileName, Resolution):
	FileNames = [None, teapot, teaspoon, teacup, yoda, onepatch]
	FileName = FileNames[FileName]
	Patches = read_indexed_patch_file(FileName)
	Raw = patches_to_raw(Patches, Resolution)
	Verts, Faces = raw_to_indexed(Raw)
	create_mesh(Verts, Faces, "Teapot")
	Blender.Window.RedrawAll()

# ===========================
# === Create Blender Mesh ===
# ===========================
def create_mesh(verts, faces, objname, norms=[], uvs=[]):
	me = Blender.Mesh.New(objname)
	me.verts.extend(verts)
	me.faces.extend(faces)
	#	# if norms:
	#	  for i in range(len(norms)):
	#		  me.verts[i].no = Blender.Mathutils.Vector(norms[i])
	#		  me.vertexUV = 1
	#		  me.verts[i].uvco = Blender.Mathutils.Vector(uvs[i])
	#	  for face in me.faces:
	#		  face.smooth = 1
	scn = Blender.Scene.GetCurrent()
	for ob in scn.objects:
		if ob.name.find(objname) == 1:
			scn.objects.unlink(ob)
	ob = scn.objects.new(me, objname)

# =================================
# === Indexed Bezier Data Block ===
# =================================
onepatch='''1
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
16
-1.5,-1.5,2
-0.5,-1.5,2
0.5,-1.5,2
1.5,-1.5,2
-1.5,-0.5,2
-0.5,-0.5,0
0.5,-0.5,0
1.5,-0.5,2
-1.5,0.5,2
-0.5,0.5,0
0.5,0.5,0
1.5,0.5,2
-1.5,1.5,2
-0.5,1.5,2
0.5,1.5,2
1.5,1.5,2
'''

teacup='''26
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
4,17,18,19,8,20,21,22,12,23,24,25,16,26,27,28
19,29,30,31,22,32,33,34,25,35,36,37,28,38,39,40
31,41,42,1,34,43,44,5,37,45,46,9,40,47,48,13
13,14,15,16,49,50,51,52,53,54,55,56,57,58,59,60
16,26,27,28,52,61,62,63,56,64,65,66,60,67,68,69
28,38,39,40,63,70,71,72,66,73,74,75,69,76,77,78
40,47,48,13,72,79,80,49,75,81,82,53,78,83,84,57
193,194,195,196,197,198,199,200,201,202,203,204,1,2,3,4
196,205,206,207,200,208,209,210,204,211,212,213,4,17,18,19
207,214,215,216,210,217,218,219,213,220,221,222,19,29,30,31
216,223,224,193,219,225,226,197,222,227,228,201,31,41,42,1
229,230,231,28,232,233,234,235,236,237,238,239,240,241,242,243
28,244,245,229,235,246,247,232,239,248,249,236,243,250,251,240
57,58,59,60,85,86,87,88,89,90,91,92,93,94,95,96
60,67,68,69,88,97,98,99,92,100,101,102,96,103,104,105
69,76,77,78,99,106,107,108,102,109,110,111,105,112,113,114
78,83,84,57,108,115,116,85,111,117,118,89,114,119,120,93
93,94,95,96,121,122,123,124,125,126,127,128,129,130,131,132
96,103,104,105,124,133,134,135,128,136,137,138,132,139,140,141
105,112,113,114,135,142,143,144,138,145,146,147,141,148,149,150
114,119,120,93,144,151,152,121,147,153,154,125,150,155,156,129
129,130,131,132,157,158,159,160,161,162,163,164,165,166,167,168
132,139,140,141,160,169,170,171,164,172,173,174,168,175,176,177
141,148,149,150,171,178,179,180,174,181,182,183,177,184,185,186
150,155,156,129,180,187,188,157,183,189,190,161,186,191,192,165
251
0.409091,0.772727,0.0
0.409091,0.772727,-0.229091
0.229091,0.772727,-0.409091
0.0,0.772727,-0.409091
0.409091,0.886364,0.0
0.409091,0.886364,-0.229091
0.229091,0.886364,-0.409091
0.0,0.886364,-0.409091
0.454545,0.886364,0.0
0.454545,0.886364,-0.254545
0.254545,0.886364,-0.454545
0.0,0.886364,-0.454545
0.454545,0.772727,0.0
0.454545,0.772727,-0.254545
0.254545,0.772727,-0.454545
0.0,0.772727,-0.454545
-0.229091,0.772727,-0.409091
-0.409091,0.772727,-0.229091
-0.409091,0.772727,0.0
-0.229091,0.886364,-0.409091
-0.409091,0.886364,-0.229091
-0.409091,0.886364,0.0
-0.254545,0.886364,-0.454545
-0.454545,0.886364,-0.254545
-0.454545,0.886364,0.0
-0.254545,0.772727,-0.454545
-0.454545,0.772727,-0.254545
-0.454545,0.772727,0.0
-0.409091,0.772727,0.229091
-0.229091,0.772727,0.409091
0.0,0.772727,0.409091
-0.409091,0.886364,0.229091
-0.229091,0.886364,0.409091
0.0,0.886364,0.409091
-0.454545,0.886364,0.254545
-0.254545,0.886364,0.454545
0.0,0.886364,0.454545
-0.454545,0.772727,0.254545
-0.254545,0.772727,0.454545
0.0,0.772727,0.454545
0.229091,0.772727,0.409091
0.409091,0.772727,0.229091
0.229091,0.886364,0.409091
0.409091,0.886364,0.229091
0.254545,0.886364,0.454545
0.454545,0.886364,0.254545
0.254545,0.772727,0.454545
0.454545,0.772727,0.254545
0.454545,0.545455,0.0
0.454545,0.545455,-0.254545
0.254545,0.545455,-0.454545
0.0,0.545455,-0.454545
0.454545,0.272727,0.0
0.454545,0.272727,-0.254545
0.254545,0.272727,-0.454545
0.0,0.272727,-0.454545
0.318182,0.0454545,0.0
0.318182,0.0454545,-0.178182
0.178182,0.0454545,-0.318182
0.0,0.0454545,-0.318182
-0.254545,0.545455,-0.454545
-0.454545,0.545455,-0.254545
-0.454545,0.545455,0.0
-0.254545,0.272727,-0.454545
-0.454545,0.272727,-0.254545
-0.454545,0.272727,0.0
-0.178182,0.0454545,-0.318182
-0.318182,0.0454545,-0.178182
-0.318182,0.0454545,0.0
-0.454545,0.545455,0.254545
-0.254545,0.545455,0.454545
0.0,0.545455,0.454545
-0.454545,0.272727,0.254545
-0.254545,0.272727,0.454545
0.0,0.272727,0.454545
-0.318182,0.0454545,0.178182
-0.178182,0.0454545,0.318182
0.0,0.0454545,0.318182
0.254545,0.545455,0.454545
0.454545,0.545455,0.254545
0.254545,0.272727,0.454545
0.454545,0.272727,0.254545
0.178182,0.0454545,0.318182
0.318182,0.0454545,0.178182
0.545455,0.0454545,0.0
0.545455,0.0454545,-0.305455
0.305455,0.0454545,-0.545455
0.0,0.0454545,-0.545455
0.727273,0.136364,0.0
0.727273,0.136364,-0.407273
0.407273,0.136364,-0.727273
0.0,0.136364,-0.727273
0.909091,0.136364,0.0
0.909091,0.136364,-0.509091
0.509091,0.136364,-0.909091
0.0,0.136364,-0.909091
-0.305455,0.0454545,-0.545455
-0.545455,0.0454545,-0.305455
-0.545455,0.0454545,0.0
-0.407273,0.136364,-0.727273
-0.727273,0.136364,-0.407273
-0.727273,0.136364,0.0
-0.509091,0.136364,-0.909091
-0.909091,0.136364,-0.509091
-0.909091,0.136364,0.0
-0.545455,0.0454545,0.305455
-0.305455,0.0454545,0.545455
0.0,0.0454545,0.545455
-0.727273,0.136364,0.407273
-0.407273,0.136364,0.727273
0.0,0.136364,0.727273
-0.909091,0.136364,0.509091
-0.509091,0.136364,0.909091
0.0,0.136364,0.909091
0.305455,0.0454545,0.545455
0.545455,0.0454545,0.305455
0.407273,0.136364,0.727273
0.727273,0.136364,0.407273
0.509091,0.136364,0.909091
0.909091,0.136364,0.509091
1.0,0.136364,0.0
1.0,0.136364,-0.56
0.56,0.136364,-1.0
0.0,0.136364,-1.0
1.0,0.0909091,0.0
1.0,0.0909091,-0.56
0.56,0.0909091,-1.0
0.0,0.0909091,-1.0
0.909091,0.0909091,0.0
0.909091,0.0909091,-0.509091
0.509091,0.0909091,-0.909091
0.0,0.0909091,-0.909091
-0.56,0.136364,-1.0
-1.0,0.136364,-0.56
-1.0,0.136364,0.0
-0.56,0.0909091,-1.0
-1.0,0.0909091,-0.56
-1.0,0.0909091,0.0
-0.509091,0.0909091,-0.909091
-0.909091,0.0909091,-0.509091
-0.909091,0.0909091,0.0
-1.0,0.136364,0.56
-0.56,0.136364,1.0
0.0,0.136364,1.0
-1.0,0.0909091,0.56
-0.56,0.0909091,1.0
0.0,0.0909091,1.0
-0.909091,0.0909091,0.509091
-0.509091,0.0909091,0.909091
0.0,0.0909091,0.909091
0.56,0.136364,1.0
1.0,0.136364,0.56
0.56,0.0909091,1.0
1.0,0.0909091,0.56
0.509091,0.0909091,0.909091
0.909091,0.0909091,0.509091
0.727273,0.0909091,0.0
0.727273,0.0909091,-0.407273
0.407273,0.0909091,-0.727273
0.0,0.0909091,-0.727273
0.545455,0.0,0.0
0.545455,0.0,-0.305455
0.305455,0.0,-0.545455
0.0,0.0,-0.545455
0.318182,0.0,0.0
0.318182,0.0,-0.178182
0.178182,0.0,-0.318182
0.0,0.0,-0.318182
-0.407273,0.0909091,-0.727273
-0.727273,0.0909091,-0.407273
-0.727273,0.0909091,0.0
-0.305455,0.0,-0.545455
-0.545455,0.0,-0.305455
-0.545455,0.0,0.0
-0.178182,0.0,-0.318182
-0.318182,0.0,-0.178182
-0.318182,0.0,0.0
-0.727273,0.0909091,0.407273
-0.407273,0.0909091,0.727273
0.0,0.0909091,0.727273
-0.545455,0.0,0.305455
-0.305455,0.0,0.545455
0.0,0.0,0.545455
-0.318182,0.0,0.178182
-0.178182,0.0,0.318182
0.0,0.0,0.318182
0.407273,0.0909091,0.727273
0.727273,0.0909091,0.407273
0.305455,0.0,0.545455
0.545455,0.0,0.305455
0.178182,0.0,0.318182
0.318182,0.0,0.178182
0.272727,0.0454545,0.0
0.272727,0.0454545,-0.152727
0.152727,0.0454545,-0.272727
0.0,0.0454545,-0.272727
0.409091,0.272727,0.0
0.409091,0.272727,-0.229091
0.229091,0.272727,-0.409091
0.0,0.272727,-0.409091
0.409091,0.545455,0.0
0.409091,0.545455,-0.229091
0.229091,0.545455,-0.409091
0.0,0.545455,-0.409091
-0.152727,0.0454545,-0.272727
-0.272727,0.0454545,-0.152727
-0.272727,0.0454545,0.0
-0.229091,0.272727,-0.409091
-0.409091,0.272727,-0.229091
-0.409091,0.272727,0.0
-0.229091,0.545455,-0.409091
-0.409091,0.545455,-0.229091
-0.409091,0.545455,0.0
-0.272727,0.0454545,0.152727
-0.152727,0.0454545,0.272727
0.0,0.0454545,0.272727
-0.409091,0.272727,0.229091
-0.229091,0.272727,0.409091
0.0,0.272727,0.409091
-0.409091,0.545455,0.229091
-0.229091,0.545455,0.409091
0.0,0.545455,0.409091
0.152727,0.0454545,0.272727
0.272727,0.0454545,0.152727
0.229091,0.272727,0.409091
0.409091,0.272727,0.229091
0.229091,0.545455,0.409091
0.409091,0.545455,0.229091
-0.454545,0.704545,0.0
-0.454545,0.704545,-0.0454545
-0.454545,0.772727,-0.0454545
-0.772727,0.863636,0.0
-0.772727,0.863636,-0.0454545
-0.818182,0.954545,-0.0454545
-0.818182,0.954545,0.0
-0.772727,0.522727,0.0
-0.772727,0.522727,-0.0454545
-0.909091,0.477273,-0.0454545
-0.909091,0.477273,0.0
-0.409091,0.363636,0.0
-0.409091,0.363636,-0.0454545
-0.409091,0.295455,-0.0454545
-0.409091,0.295455,0.0
-0.454545,0.772727,0.0454545
-0.454545,0.704545,0.0454545
-0.818182,0.954545,0.0454545
-0.772727,0.863636,0.0454545
-0.909091,0.477273,0.0454545
-0.772727,0.522727,0.0454545
-0.409091,0.295455,0.0454545
-0.409091,0.363636,0.0454545
'''

teapot='''32
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
4,17,18,19,8,20,21,22,12,23,24,25,16,26,27,28
19,29,30,31,22,32,33,34,25,35,36,37,28,38,39,40
31,41,42,1,34,43,44,5,37,45,46,9,40,47,48,13
13,14,15,16,49,50,51,52,53,54,55,56,57,58,59,60
16,26,27,28,52,61,62,63,56,64,65,66,60,67,68,69
28,38,39,40,63,70,71,72,66,73,74,75,69,76,77,78
40,47,48,13,72,79,80,49,75,81,82,53,78,83,84,57
57,58,59,60,85,86,87,88,89,90,91,92,93,94,95,96
60,67,68,69,88,97,98,99,92,100,101,102,96,103,104,105
69,76,77,78,99,106,107,108,102,109,110,111,105,112,113,114
78,83,84,57,108,115,116,85,111,117,118,89,114,119,120,93
121,122,123,124,125,126,127,128,129,130,131,132,133,134,135,136
124,137,138,121,128,139,140,125,132,141,142,129,136,143,144,133
133,134,135,136,145,146,147,148,149,150,151,152,69,153,154,155
136,143,144,133,148,156,157,145,152,158,159,149,155,160,161,69
162,163,164,165,166,167,168,169,170,171,172,173,174,175,176,177
165,178,179,162,169,180,181,166,173,182,183,170,177,184,185,174
174,175,176,177,186,187,188,189,190,191,192,193,194,195,196,197
177,184,185,174,189,198,199,186,193,200,201,190,197,202,203,194
204,204,204,204,207,208,209,210,211,211,211,211,212,213,214,215
204,204,204,204,210,217,218,219,211,211,211,211,215,220,221,222
204,204,204,204,219,224,225,226,211,211,211,211,222,227,228,229
204,204,204,204,226,230,231,207,211,211,211,211,229,232,233,212
212,213,214,215,234,235,236,237,238,239,240,241,242,243,244,245
215,220,221,222,237,246,247,248,241,249,250,251,245,252,253,254
222,227,228,229,248,255,256,257,251,258,259,260,254,261,262,263
229,232,233,212,257,264,265,234,260,266,267,238,263,268,269,242
270,270,270,270,279,280,281,282,275,276,277,278,271,272,273,274
270,270,270,270,282,289,290,291,278,286,287,288,274,283,284,285
270,270,270,270,291,298,299,300,288,295,296,297,285,292,293,294
270,270,270,270,300,305,306,279,297,303,304,275,294,301,302,271
306
1.4,0.0,2.4
1.4,-0.784,2.4
0.784,-1.4,2.4
0.0,-1.4,2.4
1.3375,0.0,2.53125
1.3375,-0.749,2.53125
0.749,-1.3375,2.53125
0.0,-1.3375,2.53125
1.4375,0.0,2.53125
1.4375,-0.805,2.53125
0.805,-1.4375,2.53125
0.0,-1.4375,2.53125
1.5,0.0,2.4
1.5,-0.84,2.4
0.84,-1.5,2.4
0.0,-1.5,2.4
-0.784,-1.4,2.4
-1.4,-0.784,2.4
-1.4,0.0,2.4
-0.749,-1.3375,2.53125
-1.3375,-0.749,2.53125
-1.3375,0.0,2.53125
-0.805,-1.4375,2.53125
-1.4375,-0.805,2.53125
-1.4375,0.0,2.53125
-0.84,-1.5,2.4
-1.5,-0.84,2.4
-1.5,0.0,2.4
-1.4,0.784,2.4
-0.784,1.4,2.4
0.0,1.4,2.4
-1.3375,0.749,2.53125
-0.749,1.3375,2.53125
0.0,1.3375,2.53125
-1.4375,0.805,2.53125
-0.805,1.4375,2.53125
0.0,1.4375,2.53125
-1.5,0.84,2.4
-0.84,1.5,2.4
0.0,1.5,2.4
0.784,1.4,2.4
1.4,0.784,2.4
0.749,1.3375,2.53125
1.3375,0.749,2.53125
0.805,1.4375,2.53125
1.4375,0.805,2.53125
0.84,1.5,2.4
1.5,0.84,2.4
1.75,0.0,1.875
1.75,-0.98,1.875
0.98,-1.75,1.875
0.0,-1.75,1.875
2.0,0.0,1.35
2.0,-1.12,1.35
1.12,-2.0,1.35
0.0,-2.0,1.35
2.0,0.0,0.9
2.0,-1.12,0.9
1.12,-2.0,0.9
0.0,-2.0,0.9
-0.98,-1.75,1.875
-1.75,-0.98,1.875
-1.75,0.0,1.875
-1.12,-2.0,1.35
-2.0,-1.12,1.35
-2.0,0.0,1.35
-1.12,-2.0,0.9
-2.0,-1.12,0.9
-2.0,0.0,0.9
-1.75,0.98,1.875
-0.98,1.75,1.875
0.0,1.75,1.875
-2.0,1.12,1.35
-1.12,2.0,1.35
0.0,2.0,1.35
-2.0,1.12,0.9
-1.12,2.0,0.9
0.0,2.0,0.9
0.98,1.75,1.875
1.75,0.98,1.875
1.12,2.0,1.35
2.0,1.12,1.35
1.12,2.0,0.9
2.0,1.12,0.9
2.0,0.0,0.45
2.0,-1.12,0.45
1.12,-2.0,0.45
0.0,-2.0,0.45
1.5,0.0,0.225
1.5,-0.84,0.225
0.84,-1.5,0.225
0.0,-1.5,0.225
1.5,0.0,0.15
1.5,-0.84,0.15
0.84,-1.5,0.15
0.0,-1.5,0.15
-1.12,-2.0,0.45
-2.0,-1.12,0.45
-2.0,0.0,0.45
-0.84,-1.5,0.225
-1.5,-0.84,0.225
-1.5,0.0,0.225
-0.84,-1.5,0.15
-1.5,-0.84,0.15
-1.5,0.0,0.15
-2.0,1.12,0.45
-1.12,2.0,0.45
0.0,2.0,0.45
-1.5,0.84,0.225
-0.84,1.5,0.225
0.0,1.5,0.225
-1.5,0.84,0.15
-0.84,1.5,0.15
0.0,1.5,0.15
1.12,2.0,0.45
2.0,1.12,0.45
0.84,1.5,0.225
1.5,0.84,0.225
0.84,1.5,0.15
1.5,0.84,0.15
-1.6,0.0,2.025
-1.6,-0.3,2.025
-1.5,-0.3,2.25
-1.5,0.0,2.25
-2.3,0.0,2.025
-2.3,-0.3,2.025
-2.5,-0.3,2.25
-2.5,0.0,2.25
-2.7,0.0,2.025
-2.7,-0.3,2.025
-3.0,-0.3,2.25
-3.0,0.0,2.25
-2.7,0.0,1.8
-2.7,-0.3,1.8
-3.0,-0.3,1.8
-3.0,0.0,1.8
-1.5,0.3,2.25
-1.6,0.3,2.025
-2.5,0.3,2.25
-2.3,0.3,2.025
-3.0,0.3,2.25
-2.7,0.3,2.025
-3.0,0.3,1.8
-2.7,0.3,1.8
-2.7,0.0,1.575
-2.7,-0.3,1.575
-3.0,-0.3,1.35
-3.0,0.0,1.35
-2.5,0.0,1.125
-2.5,-0.3,1.125
-2.65,-0.3,0.9375
-2.65,0.0,0.9375
-2.0,-0.3,0.9
-1.9,-0.3,0.6
-1.9,0.0,0.6
-3.0,0.3,1.35
-2.7,0.3,1.575
-2.65,0.3,0.9375
-2.5,0.3,1.125
-1.9,0.3,0.6
-2.0,0.3,0.9
1.7,0.0,1.425
1.7,-0.66,1.425
1.7,-0.66,0.6
1.7,0.0,0.6
2.6,0.0,1.425
2.6,-0.66,1.425
3.1,-0.66,0.825
3.1,0.0,0.825
2.3,0.0,2.1
2.3,-0.25,2.1
2.4,-0.25,2.025
2.4,0.0,2.025
2.7,0.0,2.4
2.7,-0.25,2.4
3.3,-0.25,2.4
3.3,0.0,2.4
1.7,0.66,0.6
1.7,0.66,1.425
3.1,0.66,0.825
2.6,0.66,1.425
2.4,0.25,2.025
2.3,0.25,2.1
3.3,0.25,2.4
2.7,0.25,2.4
2.8,0.0,2.475
2.8,-0.25,2.475
3.525,-0.25,2.49375
3.525,0.0,2.49375
2.9,0.0,2.475
2.9,-0.15,2.475
3.45,-0.15,2.5125
3.45,0.0,2.5125
2.8,0.0,2.4
2.8,-0.15,2.4
3.2,-0.15,2.4
3.2,0.0,2.4
3.525,0.25,2.49375
2.8,0.25,2.475
3.45,0.15,2.5125
2.9,0.15,2.475
3.2,0.15,2.4
2.8,0.15,2.4
0.0,0.0,3.15
0.0,-0.002,3.15
0.002,0.0,3.15
0.8,0.0,3.15
0.8,-0.45,3.15
0.45,-0.8,3.15
0.0,-0.8,3.15
0.0,0.0,2.85
0.2,0.0,2.7
0.2,-0.112,2.7
0.112,-0.2,2.7
0.0,-0.2,2.7
-0.002,0.0,3.15
-0.45,-0.8,3.15
-0.8,-0.45,3.15
-0.8,0.0,3.15
-0.112,-0.2,2.7
-0.2,-0.112,2.7
-0.2,0.0,2.7
0.0,0.002,3.15
-0.8,0.45,3.15
-0.45,0.8,3.15
0.0,0.8,3.15
-0.2,0.112,2.7
-0.112,0.2,2.7
0.0,0.2,2.7
0.45,0.8,3.15
0.8,0.45,3.15
0.112,0.2,2.7
0.2,0.112,2.7
0.4,0.0,2.55
0.4,-0.224,2.55
0.224,-0.4,2.55
0.0,-0.4,2.55
1.3,0.0,2.55
1.3,-0.728,2.55
0.728,-1.3,2.55
0.0,-1.3,2.55
1.3,0.0,2.4
1.3,-0.728,2.4
0.728,-1.3,2.4
0.0,-1.3,2.4
-0.224,-0.4,2.55
-0.4,-0.224,2.55
-0.4,0.0,2.55
-0.728,-1.3,2.55
-1.3,-0.728,2.55
-1.3,0.0,2.55
-0.728,-1.3,2.4
-1.3,-0.728,2.4
-1.3,0.0,2.4
-0.4,0.224,2.55
-0.224,0.4,2.55
0.0,0.4,2.55
-1.3,0.728,2.55
-0.728,1.3,2.55
0.0,1.3,2.55
-1.3,0.728,2.4
-0.728,1.3,2.4
0.0,1.3,2.4
0.224,0.4,2.55
0.4,0.224,2.55
0.728,1.3,2.55
1.3,0.728,2.55
0.728,1.3,2.4
1.3,0.728,2.4
0.0,0.0,0.0
1.5,0.0,0.15
1.5,0.84,0.15
0.84,1.5,0.15
0.0,1.5,0.15
1.5,0.0,0.075
1.5,0.84,0.075
0.84,1.5,0.075
0.0,1.5,0.075
1.425,0.0,0.0
1.425,0.798,0.0
0.798,1.425,0.0
0.0,1.425,0.0
-0.84,1.5,0.15
-1.5,0.84,0.15
-1.5,0.0,0.15
-0.84,1.5,0.075
-1.5,0.84,0.075
-1.5,0.0,0.075
-0.798,1.425,0.0
-1.425,0.798,0.0
-1.425,0.0,0.0
-1.5,-0.84,0.15
-0.84,-1.5,0.15
0.0,-1.5,0.15
-1.5,-0.84,0.075
-0.84,-1.5,0.075
0.0,-1.5,0.075
-1.425,-0.798,0.0
-0.798,-1.425,0.0
0.0,-1.425,0.0
0.84,-1.5,0.15
1.5,-0.84,0.15
0.84,-1.5,0.075
1.5,-0.84,0.075
0.798,-1.425,0.0
1.425,-0.798,0.0
'''

teaspoon='''16
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
17,18,19,20,21,22,23,24,25,26,27,28,29,30,31,32
33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48
49,50,51,52,53,54,55,56,57,58,59,60,61,62,63,64
65,66,67,68,69,70,71,72,73,74,75,76,77,78,79,80
81,82,83,84,85,86,87,88,89,90,91,92,93,94,95,96
97,98,99,100,101,102,103,104,105,106,107,108,109,110,111,112
113,114,115,116,117,118,119,120,121,122,123,124,125,126,127,128
129,130,131,132,133,134,135,136,137,138,139,140,141,142,143,144
145,146,147,148,149,150,151,152,153,154,155,156,157,158,159,160
161,162,163,164,165,166,167,168,169,170,171,172,173,174,175,176
177,178,179,180,181,182,183,184,185,186,187,188,189,190,191,192
193,194,195,196,197,198,199,200,201,202,203,204,205,206,207,208
209,210,211,212,213,214,215,216,217,218,219,220,221,222,223,224
225,226,227,228,229,230,231,232,233,234,235,236,237,238,239,240
241,242,243,244,245,246,247,248,249,250,251,252,253,254,255,256
256
-0.000107143,0.205357,0.0
0.0,0.196429,-0.0178571
0.0,0.196429,-0.0178571
0.000107143,0.205357,0.0
-0.0535714,0.205357,0.0
-0.0222714,0.178571,-0.0534286
0.0222714,0.178571,-0.0534286
0.0535714,0.205357,0.0
-0.107143,0.0952429,-0.0178571
-0.0446429,0.0952429,-0.0892857
0.0446429,0.0952429,-0.0892857
0.107143,0.0952429,-0.0178571
-0.107143,0.0,-0.0178571
-0.0446429,0.0,-0.0892857
0.0446429,0.0,-0.0892857
0.107143,0.0,-0.0178571
0.000107143,0.205357,0.0
0.000135714,0.207589,0.00446429
0.000157143,0.216518,0.00446429
0.000125,0.214286,0.0
0.0535714,0.205357,0.0
0.0613964,0.212054,0.0133571
0.0714286,0.220982,0.015625
0.0625,0.214286,0.0
0.107143,0.0952429,-0.0178571
0.122768,0.0952429,0.0
0.142857,0.0952429,0.00446429
0.125,0.0952429,-0.0178571
0.107143,0.0,-0.0178571
0.122768,0.0,0.0
0.142857,0.0,0.00446429
0.125,0.0,-0.0178571
0.000125,0.214286,0.0
0.0,0.205357,-0.0178571
0.0,0.205357,-0.0178571
-0.000125,0.214286,0.0
0.0625,0.214286,0.0
0.0267857,0.1875,-0.0625
-0.0267857,0.1875,-0.0625
-0.0625,0.214286,0.0
0.125,0.0952429,-0.0178571
0.0535714,0.0952429,-0.107143
-0.0535714,0.0952429,-0.107143
-0.125,0.0952429,-0.0178571
0.125,0.0,-0.0178571
0.0535714,0.0,-0.107143
-0.0535714,0.0,-0.107143
-0.125,0.0,-0.0178571
-0.000125,0.214286,0.0
-0.000157143,0.216518,0.00446429
-0.000135714,0.207589,0.00446429
-0.000107143,0.205357,0.0
-0.0625,0.214286,0.0
-0.0714286,0.220982,0.015625
-0.0613964,0.212054,0.0133571
-0.0535714,0.205357,0.0
-0.125,0.0952429,-0.0178571
-0.142857,0.0952429,0.00446429
-0.122768,0.0952429,0.0
-0.107143,0.0952429,-0.0178571
-0.125,0.0,-0.0178571
-0.142857,0.0,0.00446429
-0.122768,0.0,0.0
-0.107143,0.0,-0.0178571
-0.107143,0.0,-0.0178571
-0.0446429,0.0,-0.0892857
0.0446429,0.0,-0.0892857
0.107143,0.0,-0.0178571
-0.107143,-0.142857,-0.0178571
-0.0446429,-0.142857,-0.0892857
0.0446429,-0.142857,-0.0892857
0.107143,-0.142857,-0.0178571
-0.0133929,-0.160714,0.0386893
-0.00557857,-0.160714,0.0386893
0.00557857,-0.160714,0.0386893
0.0133929,-0.160714,0.0386893
-0.0133929,-0.25,0.0535714
-0.00557857,-0.25,0.0535714
0.00557857,-0.25,0.0535714
0.0133929,-0.25,0.0535714
0.107143,0.0,-0.0178571
0.122768,0.0,0.0
0.142857,0.0,0.00446429
0.125,0.0,-0.0178571
0.107143,-0.142857,-0.0178571
0.122768,-0.142857,0.0
0.142857,-0.142857,0.00446429
0.125,-0.142857,-0.0178571
0.0133929,-0.160714,0.0386893
0.0153464,-0.160714,0.0386893
0.0178571,-0.160714,0.0314357
0.015625,-0.160714,0.0297607
0.0133929,-0.25,0.0535714
0.0153464,-0.25,0.0535714
0.0178571,-0.25,0.0463179
0.015625,-0.25,0.0446429
0.125,0.0,-0.0178571
0.0535714,0.0,-0.107143
-0.0535714,0.0,-0.107143
-0.125,0.0,-0.0178571
0.125,-0.142857,-0.0178571
0.0535714,-0.142857,-0.107143
-0.0535714,-0.142857,-0.107143
-0.125,-0.142857,-0.0178571
0.015625,-0.160714,0.0297607
0.00669643,-0.160714,0.0230643
-0.00781071,-0.160714,0.0208321
-0.015625,-0.160714,0.0297607
0.015625,-0.25,0.0446429
0.00669643,-0.25,0.0379464
-0.00781071,-0.25,0.0357143
-0.015625,-0.25,0.0446429
-0.125,0.0,-0.0178571
-0.142857,0.0,0.00446429
-0.122768,0.0,0.0
-0.107143,0.0,-0.0178571
-0.125,-0.142857,-0.0178571
-0.142857,-0.142857,0.00446429
-0.122768,-0.142857,0.0
-0.107143,-0.142857,-0.0178571
-0.015625,-0.160714,0.0297607
-0.0175786,-0.160714,0.0319929
-0.0153464,-0.160714,0.0386893
-0.0133929,-0.160714,0.0386893
-0.015625,-0.25,0.0446429
-0.0175786,-0.25,0.046875
-0.0153464,-0.25,0.0535714
-0.0133929,-0.25,0.0535714
-0.0133929,-0.25,0.0535714
-0.00557857,-0.25,0.0535714
0.00557857,-0.25,0.0535714
0.0133929,-0.25,0.0535714
-0.0133929,-0.46425,0.0892857
-0.00557857,-0.46425,0.0892857
0.00557857,-0.46425,0.0892857
0.0133929,-0.46425,0.0892857
-0.0446429,-0.678571,0.0535714
-0.00892857,-0.678571,0.0625
0.00892857,-0.678571,0.0625
0.0446429,-0.678571,0.0535714
-0.0446429,-0.857143,0.0357143
-0.00892857,-0.857143,0.0446429
0.00892857,-0.857143,0.0446429
0.0446429,-0.857143,0.0357143
0.0133929,-0.25,0.0535714
0.0153464,-0.25,0.0535714
0.0178571,-0.25,0.0463179
0.015625,-0.25,0.0446429
0.0133929,-0.46425,0.0892857
0.0153464,-0.464286,0.0892857
0.0178571,-0.46425,0.0820321
0.015625,-0.46425,0.0803571
0.0446429,-0.678571,0.0535714
0.0535714,-0.678571,0.0513393
0.0535714,-0.678571,0.0334821
0.0446429,-0.678571,0.0357143
0.0446429,-0.857143,0.0357143
0.0535714,-0.857143,0.0334821
0.0535714,-0.857143,0.015625
0.0446429,-0.857143,0.0178571
0.015625,-0.25,0.0446429
0.00669643,-0.25,0.0379464
-0.00781071,-0.25,0.0357143
-0.015625,-0.25,0.0446429
0.015625,-0.46425,0.0803571
0.00669643,-0.464286,0.0736607
-0.00781071,-0.46425,0.0714286
-0.015625,-0.46425,0.0803571
0.0446429,-0.678571,0.0357143
0.00892857,-0.678571,0.0446429
-0.00892857,-0.678571,0.0446429
-0.0446429,-0.678571,0.0357143
0.0446429,-0.857143,0.0178571
0.00892857,-0.857143,0.0267857
-0.00892857,-0.857143,0.0267857
-0.0446429,-0.857143,0.0178571
-0.015625,-0.25,0.0446429
-0.0175786,-0.25,0.046875
-0.0153464,-0.25,0.0535714
-0.0133929,-0.25,0.0535714
-0.015625,-0.46425,0.0803571
-0.0175786,-0.464286,0.0825893
-0.0153464,-0.464286,0.0892857
-0.0133929,-0.46425,0.0892857
-0.0446429,-0.678571,0.0357143
-0.0535714,-0.678571,0.0334821
-0.0535714,-0.678571,0.0513393
-0.0446429,-0.678571,0.0535714
-0.0446429,-0.857143,0.0178571
-0.0535714,-0.857143,0.015625
-0.0535714,-0.857143,0.0334821
-0.0446429,-0.857143,0.0357143
-0.0446429,-0.857143,0.0357143
-0.00892857,-0.857143,0.0446429
0.00892857,-0.857143,0.0446429
0.0446429,-0.857143,0.0357143
-0.0446429,-0.928571,0.0285714
-0.00892857,-0.928571,0.0375
0.00892857,-0.928571,0.0375
0.0446429,-0.928571,0.0285714
-0.0539286,-0.999643,0.0178571
0.000357143,-0.999643,0.0178571
0.0,-0.999643,0.0178571
0.0535714,-0.999643,0.0178571
-0.000357143,-1,0.0178571
0.000357143,-1,0.0178571
0.0,-1,0.0178571
0.0,-1,0.0178571
0.0446429,-0.857143,0.0357143
0.0535714,-0.857143,0.0334821
0.0535714,-0.857143,0.015625
0.0446429,-0.857143,0.0178571
0.0446429,-0.928571,0.0285714
0.0535714,-0.928571,0.0263393
0.0535714,-0.928571,0.00848214
0.0446429,-0.928571,0.0107143
0.0535714,-0.999643,0.0178571
0.0669643,-0.999643,0.0178571
0.0673214,-0.999643,0.0
0.0539286,-0.999643,0.0
0.0,-1,0.0178571
0.0,-1,0.0178571
0.000357143,-1,0.0
0.000357143,-1,0.0
0.0446429,-0.857143,0.0178571
0.00892857,-0.857143,0.0267857
-0.00892857,-0.857143,0.0267857
-0.0446429,-0.857143,0.0178571
0.0446429,-0.928571,0.0107143
0.00892857,-0.928571,0.0196429
-0.00892857,-0.928571,0.0196429
-0.0446429,-0.928571,0.0107143
0.0539286,-0.999643,0.0
0.000357143,-0.999643,0.0
-0.000357143,-0.999643,0.0
-0.0539286,-0.999643,0.0
0.000357143,-1,0.0
0.000357143,-1,0.0
-0.000357143,-1,0.0
-0.000357143,-1,0.0
-0.0446429,-0.857143,0.0178571
-0.0535714,-0.857143,0.015625
-0.0535714,-0.857143,0.0334821
-0.0446429,-0.857143,0.0357143
-0.0446429,-0.928571,0.0107143
-0.0535714,-0.928571,0.00848214
-0.0535714,-0.928571,0.0263393
-0.0446429,-0.928571,0.0285714
-0.0539286,-0.999643,0.0
-0.0673214,-0.999643,0.0
-0.0675,-0.999643,0.0178571
-0.0539286,-0.999643,0.0178571
-0.000357143,-1,0.0
-0.000357143,-1,0.0
-0.000535714,-1,0.0178571
-0.000357143,-1,0.0178571
'''

yoda='''666
1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,16
1,5,9,13,17,18,19,20,21,22,23,24,25,26,27,28
1,29,30,31,2,32,33,34,3,35,36,37,4,38,39,40
1,17,21,25,29,41,42,43,30,44,45,46,31,47,48,49
50,51,52,53,54,55,56,57,58,59,60,61,62,63,64,65
50,54,58,62,66,67,68,69,70,71,72,73,74,75,76,77
50,78,79,80,51,81,82,83,52,84,85,86,53,87,88,89
50,66,70,74,78,90,91,92,79,93,94,95,80,96,97,98
74,99,100,101,92,102,103,104,95,105,106,107,98,108,109,110
74,75,76,77,99,111,112,113,100,114,115,116,101,117,118,119
120,121,122,101,123,124,125,117,126,127,128,118,129,130,131,119
120,123,126,129,132,133,134,135,136,137,138,139,140,141,142,143
120,144,145,146,121,147,148,149,122,150,151,152,101,104,107,110
120,132,136,140,144,153,154,155,145,156,157,158,146,159,160,161
140,141,142,143,162,163,164,165,166,167,168,169,170,171,172,173
140,162,166,170,155,174,175,176,158,177,178,179,161,180,181,182
62,63,64,65,183,184,185,186,187,188,189,190,191,192,193,194
62,183,187,191,69,195,196,197,73,198,199,200,77,201,202,203
77,201,202,203,113,204,205,206,116,207,208,209,119,210,211,212
129,130,131,119,213,214,215,210,216,217,218,211,219,220,221,212
129,213,216,219,135,222,223,224,139,225,226,227,143,228,229,230
143,228,229,230,165,231,232,233,169,234,235,236,173,237,238,239
240,241,242,243,244,245,246,247,248,249,250,251,31,34,37,40
240,244,248,31,252,253,254,47,255,256,257,48,258,259,260,49
240,252,255,258,261,262,263,264,265,266,267,268,269,270,271,272
240,261,265,269,241,273,274,275,242,276,277,278,243,279,280,281
243,282,283,284,247,285,286,287,251,288,289,290,40,291,292,293
243,279,280,281,282,294,295,296,283,297,298,299,284,300,301,302
303,304,305,306,307,308,309,310,311,312,313,314,315,316,317,318
303,307,311,315,319,320,321,322,323,324,325,326,284,287,290,293
303,319,323,284,327,328,329,300,330,331,332,301,333,334,335,302
303,327,330,333,304,336,337,338,305,339,340,341,306,342,343,344
191,192,193,194,345,346,347,348,349,350,351,352,353,354,355,356
191,345,349,353,197,357,358,359,200,360,361,362,203,363,364,365
203,363,364,365,206,366,367,368,209,369,370,371,212,372,373,374
219,220,221,212,375,376,377,372,378,379,380,373,381,382,383,374
219,375,378,381,224,384,385,386,227,387,388,389,230,390,391,392
230,390,391,392,233,393,394,395,236,396,397,398,239,399,400,401
25,26,27,28,402,403,404,405,406,407,408,409,410,411,412,413
25,402,406,410,43,414,415,416,46,417,418,419,49,420,421,422
258,259,260,49,423,424,425,420,426,427,428,421,429,430,431,422
258,423,426,429,264,432,433,434,268,435,436,437,272,438,439,440
441,442,443,444,445,446,447,448,449,450,451,452,453,454,455,456
441,445,449,453,457,458,459,460,461,462,463,464,465,466,467,468
441,469,470,471,442,472,473,474,443,475,476,477,444,478,479,480
441,457,461,465,469,481,482,483,470,484,485,486,471,487,488,489
272,438,439,440,490,491,492,493,494,495,496,497,498,499,500,501
272,490,494,498,271,502,503,504,270,505,506,507,269,508,509,510
269,508,509,510,275,511,512,513,278,514,515,516,281,517,518,519
281,517,518,519,296,520,521,522,299,523,524,525,302,526,527,528
302,526,527,528,335,529,530,531,334,532,533,534,333,535,536,537
333,535,536,537,338,538,539,540,341,541,542,543,344,544,545,546
547,548,549,550,551,552,553,554,555,556,557,558,559,560,561,562
547,551,555,559,563,564,565,566,567,568,569,570,571,572,573,574
547,575,576,353,548,577,578,354,549,579,580,355,550,581,582,356
547,563,567,571,575,583,584,585,576,586,587,588,353,359,362,365
571,589,590,591,585,592,593,594,588,595,596,597,365,368,371,374
571,572,573,574,589,598,599,600,590,601,602,603,591,604,605,480
606,607,608,591,609,610,611,604,612,613,614,605,471,474,477,480
606,609,612,471,615,616,617,487,618,619,620,488,621,622,623,489
606,624,625,381,607,626,627,382,608,628,629,383,591,594,597,374
606,615,618,621,624,630,631,632,625,633,634,635,381,386,389,392
621,622,623,489,636,637,638,639,640,641,642,643,644,645,646,647
621,636,640,644,632,648,649,650,635,651,652,653,392,395,398,401
465,466,467,468,654,655,656,657,658,659,660,661,662,663,664,665
465,654,658,662,483,666,667,668,486,669,670,671,489,639,643,647
672,673,674,675,676,677,678,679,680,681,682,683,684,685,686,687
672,676,680,684,688,689,690,691,692,693,694,695,696,697,698,699
672,700,701,468,688,702,703,464,692,704,705,460,696,706,707,453
672,700,701,468,673,708,709,657,674,710,711,661,675,712,713,665
714,715,716,717,718,719,720,721,722,723,724,725,726,727,728,729
714,718,722,726,730,731,732,733,734,735,736,737,194,348,352,356
714,738,739,740,715,741,742,743,716,744,745,746,717,747,748,749
714,730,734,194,738,750,751,190,739,752,753,186,740,754,755,65
756,757,758,759,760,761,762,763,764,765,766,767,740,743,746,749
756,760,764,740,768,769,770,754,771,772,773,755,53,57,61,65
756,774,775,776,757,777,778,779,758,780,781,782,759,783,784,785
756,768,771,53,774,786,787,87,775,788,789,88,776,790,791,89
759,783,784,785,792,793,794,795,796,797,798,799,800,801,802,803
759,792,796,800,763,804,805,806,767,807,808,809,749,810,811,812
344,813,814,815,343,816,817,818,342,819,820,821,306,822,823,824
344,544,545,546,813,825,826,827,814,828,829,830,815,831,832,833
574,570,566,559,834,835,836,837,838,839,840,841,842,843,844,845
574,834,838,842,600,846,847,848,603,849,850,851,480,479,478,444
852,853,854,833,855,856,857,858,859,860,861,862,863,864,865,866
852,855,859,863,867,868,869,870,871,872,873,874,562,875,876,877
852,878,879,880,853,881,882,883,854,884,885,886,833,832,831,815
852,867,871,562,878,887,888,558,879,889,890,554,880,891,892,550
833,830,827,546,858,893,894,895,862,896,897,898,866,899,900,901
546,543,540,537,895,902,903,904,898,905,906,907,901,908,909,910
537,534,531,528,904,911,912,913,907,914,915,916,910,917,918,919
528,525,522,519,913,920,921,922,916,923,924,925,919,926,927,928
510,929,930,699,507,931,932,695,504,933,934,691,498,935,936,684
510,929,930,699,513,937,938,939,516,940,941,942,519,922,925,928
498,935,936,684,499,943,944,685,500,945,946,686,501,947,948,687
880,949,950,726,883,951,952,727,886,953,954,728,815,955,956,729
880,891,892,550,949,957,958,581,950,959,960,582,726,733,737,356
306,822,823,824,310,961,962,963,314,964,965,966,318,967,968,969
970,971,972,749,973,974,975,976,977,978,979,980,981,982,983,969
970,984,985,986,971,987,988,989,972,990,991,992,749,748,747,717
986,993,994,995,989,996,997,998,992,999,1000,1001,717,721,725,729
1002,1003,1004,815,1005,1006,1007,955,1008,1009,1010,956,995,998,1001,729
1002,1011,1012,1013,1003,1014,1015,1016,1004,1017,1018,1019,815,818,821,824
1013,1020,1021,981,1016,1022,1023,982,1019,1024,1025,983,824,963,966,969
749,810,811,812,976,1026,1027,1028,980,1029,1030,1031,969,968,967,318
785,1032,1033,1034,795,1035,1036,1037,799,1038,1039,1040,803,1041,1042,1043
785,782,779,776,1032,1044,1045,1046,1033,1047,1048,1049,1034,1050,1051,1052
776,790,791,89,1046,1053,1054,1055,1049,1056,1057,1058,1052,1059,1060,1061
80,1062,1063,1064,83,1065,1066,1067,86,1068,1069,1070,89,1055,1058,1061
80,96,97,98,1062,1071,1072,1073,1063,1074,1075,1076,1064,1077,1078,1079
98,108,109,110,1073,1080,1081,1082,1076,1083,1084,1085,1079,1086,1087,1088
146,1089,1090,1091,149,1092,1093,1094,152,1095,1096,1097,110,1082,1085,1088
146,159,160,161,1089,1098,1099,1100,1090,1101,1102,1103,1091,1104,1105,1106
161,180,181,182,1100,1107,1108,1109,1103,1110,1111,1112,1106,1113,1114,1115
315,316,317,318,1116,1117,1118,1031,1119,1120,1121,1028,1122,1123,1124,812
315,1116,1119,1122,322,1125,1126,1127,326,1128,1129,1130,293,1131,1132,1133
803,1041,1042,1043,1134,1135,1136,1137,1138,1139,1140,1141,1142,1143,1144,1145
803,1134,1138,1142,802,1146,1147,1148,801,1149,1150,1151,800,1152,1153,1154
1142,1143,1144,1145,1155,1156,1157,1158,1159,1160,1161,1162,1163,1164,1165,1166
1142,1155,1159,1163,1148,1167,1168,1169,1151,1170,1171,1172,1154,1173,1174,1175
1175,1172,1169,1163,1176,1177,1178,1179,1180,1181,1182,1183,1184,1185,1186,1187
1175,1175,1175,1175,1188,1189,1189,1176,1190,1191,1192,1180,1193,1194,1195,1184
1175,1188,1190,1193,1174,1196,1197,1198,1173,1199,1200,1201,1154,1202,1203,1122
1163,1164,1165,1166,1179,1204,1205,1206,1183,1207,1208,1209,1187,1210,1211,1212
1187,1210,1211,1212,1213,1214,1215,1216,1217,1218,1219,1220,1221,1222,1223,1224
1187,1213,1217,1221,1186,1225,1226,1227,1185,1228,1229,1230,1184,1231,1232,1233
1184,1231,1232,1233,1195,1234,1235,1236,1194,1237,1238,1239,1193,1240,1241,1242
1193,1240,1241,1242,1198,1243,1244,1245,1201,1246,1247,1248,1122,1127,1130,1133
293,1131,1132,1133,292,1249,1250,1251,291,1252,1253,1254,40,1255,1256,1257
1242,1239,1236,1233,1258,1259,1260,1261,1262,1263,1264,1265,1266,1267,1268,1269
1242,1258,1262,1266,1245,1270,1271,1272,1248,1273,1274,1275,1133,1251,1254,1257
1233,1230,1227,1221,1261,1276,1277,1278,1265,1279,1280,1281,1269,1282,1283,1284
1221,1222,1223,1224,1278,1285,1286,1287,1281,1288,1289,1290,1284,1291,1292,1293
812,809,806,800,1124,1294,1295,1152,1123,1296,1297,1153,1122,1203,1202,1154
562,875,876,877,561,1298,1299,1300,560,1301,1302,1303,559,837,841,845
1304,1305,1306,1307,1308,1309,1310,1311,1312,1313,1314,1315,1316,1317,1318,1319
1304,1308,1312,1316,1320,1321,1322,1323,1324,1325,1326,1327,1328,1329,1330,1331
1304,1320,1324,1328,1332,1333,1334,1335,1336,1337,1338,1339,1340,1341,1342,1343
1304,1332,1336,1340,1305,1344,1345,1346,1306,1347,1348,1349,1307,1350,1351,1352
1340,1353,1354,1355,1346,1356,1357,1358,1349,1359,1360,1361,1352,1362,1363,1364
1340,1341,1342,1343,1353,1365,1366,1367,1354,1368,1369,1370,1355,1371,1372,1373
1355,1371,1372,1373,1374,1375,1376,1377,1378,1379,1380,1381,1382,1383,1384,1385
1355,1374,1378,1382,1358,1386,1387,1388,1361,1389,1390,1391,1364,1392,1393,1394
1382,1383,1384,1385,1395,1396,1397,1398,1399,1400,1401,1402,1403,1404,1405,1406
1382,1395,1399,1403,1388,1407,1408,1409,1391,1410,1411,1412,1394,1413,1414,1415
1403,1404,1405,1406,1416,1417,1418,1419,1420,1421,1422,1423,1424,1425,1426,1427
1403,1416,1420,1424,1409,1428,1429,1430,1412,1431,1432,1433,1415,1434,1435,1436
1424,1437,1438,1439,1430,1440,1441,1442,1433,1443,1444,1445,1436,1446,1447,1448
1424,1425,1426,1427,1437,1449,1450,1451,1438,1452,1453,1454,1439,1455,1456,1457
1439,1458,1459,1460,1442,1461,1462,1463,1445,1464,1465,1466,1448,1467,1468,1469
1439,1455,1456,1457,1458,1470,1471,1472,1459,1473,1474,1475,1460,1476,1477,1478
1460,1479,1480,1481,1463,1482,1483,1484,1466,1485,1486,1487,1469,1488,1489,1490
1460,1476,1477,1478,1479,1491,1492,1493,1480,1494,1495,1496,1481,1497,1498,1499
1481,1500,1501,1316,1484,1502,1503,1317,1487,1504,1505,1318,1490,1506,1507,1319
1481,1497,1498,1499,1500,1508,1509,1510,1501,1511,1512,1513,1316,1323,1327,1331
1307,1514,1515,1516,1311,1517,1518,1519,1315,1520,1521,1522,1319,1523,1524,1525
1307,1350,1351,1352,1514,1526,1527,1528,1515,1529,1530,1531,1516,1532,1533,1534
1352,1362,1363,1364,1528,1535,1536,1537,1531,1538,1539,1540,1534,1541,1542,1543
1364,1392,1393,1394,1537,1544,1545,1546,1540,1547,1548,1549,1543,1550,1551,1552
1394,1413,1414,1415,1546,1553,1554,1555,1549,1556,1557,1558,1552,1559,1560,1561
1415,1434,1435,1436,1555,1562,1563,1564,1558,1565,1566,1567,1561,1568,1569,1570
1436,1446,1447,1448,1564,1571,1572,1573,1567,1574,1575,1576,1570,1577,1578,1579
1448,1467,1468,1469,1573,1580,1581,1582,1576,1583,1584,1585,1579,1586,1587,1588
1469,1488,1489,1490,1582,1589,1590,1591,1585,1592,1593,1594,1588,1595,1596,1597
1490,1506,1507,1319,1591,1598,1599,1523,1594,1600,1601,1524,1597,1602,1603,1525
1516,1604,1605,1606,1519,1607,1608,1609,1522,1610,1611,1612,1525,1613,1614,1615
1516,1532,1533,1534,1604,1616,1617,1541,1605,1618,1619,1542,1606,1620,1621,1543
1516,1516,1516,1516,1604,1616,1616,1532,1605,1622,1623,1533,1606,1624,1625,1534
1534,1534,1534,1534,1625,1626,1626,1541,1624,1627,1619,1542,1606,1620,1621,1543
1579,1586,1587,1588,1578,1628,1629,1630,1577,1631,1632,1633,1570,1634,1635,1636
1579,1579,1579,1579,1637,1638,1638,1578,1639,1640,1631,1577,1636,1635,1634,1570
1579,1579,1579,1579,1637,1641,1641,1586,1639,1642,1629,1587,1636,1633,1630,1588
1588,1595,1596,1597,1630,1643,1644,1645,1633,1646,1647,1648,1636,1649,1650,1651
1525,1613,1614,1615,1603,1652,1653,1654,1602,1655,1656,1657,1597,1645,1648,1651
1561,1658,1659,1651,1560,1660,1661,1657,1559,1662,1663,1654,1552,1664,1665,1615
1561,1568,1569,1570,1658,1666,1667,1634,1659,1668,1669,1635,1651,1650,1649,1636
1552,1664,1665,1615,1551,1670,1671,1612,1550,1672,1673,1609,1543,1621,1620,1606
1674,1675,1676,1677,1678,1679,1680,1681,1682,1683,1684,1685,1686,1687,1688,1689
1674,1678,1682,1686,1690,1691,1692,1693,1694,1695,1696,1697,1698,1699,1700,1701
1674,1690,1694,1698,1702,1703,1704,1705,1706,1707,1708,1709,1710,1711,1712,1713
1674,1702,1706,1710,1675,1714,1715,1716,1676,1717,1718,1719,1677,1720,1721,1722
1710,1723,1724,1725,1716,1726,1727,1728,1719,1729,1730,1731,1722,1732,1733,1734
1710,1711,1712,1713,1723,1735,1736,1737,1724,1738,1739,1740,1725,1741,1742,1743
1725,1741,1742,1743,1744,1745,1746,1747,1748,1749,1750,1751,1752,1753,1754,1755
1725,1744,1748,1752,1728,1756,1757,1758,1731,1759,1760,1761,1734,1762,1763,1764
1752,1753,1754,1755,1765,1766,1767,1768,1769,1770,1771,1772,1773,1774,1775,1776
1752,1765,1769,1773,1758,1777,1778,1779,1761,1780,1781,1782,1764,1783,1784,1785
1773,1774,1775,1776,1786,1787,1788,1789,1790,1791,1792,1793,1794,1795,1796,1797
1773,1786,1790,1794,1779,1798,1799,1800,1782,1801,1802,1803,1785,1804,1805,1806
1794,1807,1808,1809,1800,1810,1811,1812,1803,1813,1814,1815,1806,1816,1817,1818
1794,1795,1796,1797,1807,1819,1820,1821,1808,1822,1823,1824,1809,1825,1826,1827
1809,1828,1829,1830,1812,1831,1832,1833,1815,1834,1835,1836,1818,1837,1838,1839
1809,1825,1826,1827,1828,1840,1841,1842,1829,1843,1844,1845,1830,1846,1847,1848
1830,1849,1850,1851,1833,1852,1853,1854,1836,1855,1856,1857,1839,1858,1859,1860
1830,1846,1847,1848,1849,1861,1862,1863,1850,1864,1865,1866,1851,1867,1868,1869
1851,1870,1871,1686,1854,1872,1873,1687,1857,1874,1875,1688,1860,1876,1877,1689
1851,1867,1868,1869,1870,1878,1879,1880,1871,1881,1882,1883,1686,1693,1697,1701
1698,1699,1700,1701,1884,1885,1886,1887,1888,1889,1890,1891,1892,1893,1894,1895
1698,1884,1888,1892,1705,1896,1897,1898,1709,1899,1900,1901,1713,1902,1903,456
1713,1902,1903,456,1737,1904,1905,1906,1740,1907,1908,1909,1743,1910,1911,1912
1743,1910,1911,1912,1747,1913,1914,1915,1751,1916,1917,1918,1755,1919,1920,1921
1755,1919,1920,1921,1768,1922,1923,1924,1772,1925,1926,1927,1776,1928,1929,1930
1776,1928,1929,1930,1789,1931,1932,1933,1793,1934,1935,1936,1797,1937,1938,1939
1797,1937,1938,1939,1821,1940,1941,1942,1824,1943,1944,1945,1827,1946,1947,1948
1827,1946,1947,1948,1842,1949,1950,1951,1845,1952,1953,1954,1848,1955,1956,1957
1848,1955,1956,1957,1863,1958,1959,1960,1866,1961,1962,1963,1869,1964,1965,1966
1869,1964,1965,1966,1880,1967,1968,1969,1883,1970,1971,1972,1701,1887,1891,1895
919,1973,1974,1921,926,1975,1976,1918,927,1977,1978,1915,928,1979,1980,1912
919,1973,1974,1921,918,1981,1982,1924,917,1983,1984,1927,910,1985,1986,1930
910,1985,1986,1930,909,1987,1988,1933,908,1989,1990,1936,901,1991,1992,1939
863,1993,1994,1957,864,1995,1996,1954,865,1997,1998,1951,866,1999,2000,1948
863,1993,1994,1957,870,2001,2002,1960,874,2003,2004,1963,877,2005,2006,1966
1939,1942,1945,1948,1992,2007,2008,2000,1991,2009,2010,1999,901,900,899,866
1895,1972,1969,1966,2011,2012,2013,2006,2014,2015,2016,2005,845,1303,1300,877
1895,1894,1893,1892,2011,2017,2018,2019,2014,2020,2021,2022,845,844,843,842
928,942,939,699,1979,2023,2024,698,1980,2025,2026,697,1912,2027,2028,696
842,848,851,444,2022,2029,2030,448,2019,2031,2032,452,1892,1898,1901,456
696,706,707,453,2028,2033,2034,454,2027,2035,2036,455,1912,1909,1906,456
1818,2037,2038,2039,1817,2040,2041,2042,1816,2043,2044,2045,1806,2046,2047,2048
1818,1837,1838,1839,2037,2049,2050,2051,2038,2052,2053,2054,2039,2055,2056,2057
1839,1858,1859,1860,2051,2058,2059,2060,2054,2061,2062,2063,2057,2064,2065,2066
1860,1876,1877,1689,2060,2067,2068,2069,2063,2070,2071,2072,2066,2073,2074,2075
1689,1685,1681,1677,2069,2076,2077,2078,2072,2079,2080,2081,2075,2082,2083,2084
1677,1720,1721,1722,2078,2085,2086,2087,2081,2088,2089,2090,2084,2091,2092,2093
1722,1732,1733,1734,2087,2094,2095,2096,2090,2097,2098,2099,2093,2100,2101,2102
1734,1762,1763,1764,2096,2103,2104,2105,2099,2106,2107,2108,2102,2109,2110,2111
1764,1783,1784,1785,2105,2112,2113,2114,2108,2115,2116,2117,2111,2118,2119,2120
1785,1804,1805,1806,2114,2121,2122,2046,2117,2123,2124,2047,2120,2125,2126,2048
2039,2127,2128,1457,2042,2129,2130,1454,2045,2131,2132,1451,2048,2133,2134,1427
2039,2055,2056,2057,2127,2135,2136,2137,2128,2138,2139,2140,1457,1472,1475,1478
2057,2064,2065,2066,2137,2141,2142,2143,2140,2144,2145,2146,1478,1493,1496,1499
2066,2073,2074,2075,2143,2147,2148,2149,2146,2150,2151,2152,1499,1510,1513,1331
2075,2082,2083,2084,2149,2153,2154,2155,2152,2156,2157,2158,1331,1330,1329,1328
2084,2091,2092,2093,2155,2159,2160,2161,2158,2162,2163,2164,1328,1335,1339,1343
2093,2100,2101,2102,2161,2165,2166,2167,2164,2168,2169,2170,1343,1367,1370,1373
2102,2109,2110,2111,2167,2171,2172,2173,2170,2174,2175,2176,1373,1377,1381,1385
2111,2118,2119,2120,2173,2177,2178,2179,2176,2180,2181,2182,1385,1398,1402,1406
2120,2125,2126,2048,2179,2183,2184,2133,2182,2185,2186,2134,1406,1419,1423,1427
1257,1275,1272,1266,2187,2188,2189,2190,2191,2192,2193,2194,2195,2196,2197,2198
1257,2187,2191,2195,1256,2199,2200,2201,1255,2202,2203,2204,40,2205,2206,2207
1266,1267,1268,1269,2190,2208,2209,2210,2194,2211,2212,2213,2198,2214,2215,2216
1269,1282,1283,1284,2210,2217,2218,2219,2213,2220,2221,2222,2216,2223,2224,2225
1284,1291,1292,1293,2219,2226,2227,2228,2222,2229,2230,2231,2225,2232,2233,2234
4,38,39,40,2235,2236,2237,2205,2238,2239,2240,2206,2241,2242,2243,2207
4,2235,2238,2241,8,2244,2245,2246,12,2247,2248,2249,16,2250,2251,2252
182,2253,2254,2255,1109,2256,2257,2258,1112,2259,2260,2261,1115,2262,2263,2264
182,179,176,170,2253,2265,2266,2267,2254,2268,2269,2270,2255,2271,2272,2273
170,171,172,173,2267,2274,2275,2276,2270,2277,2278,2279,2273,2280,2281,2282
173,237,238,239,2276,2283,2284,2285,2279,2286,2287,2288,2282,2289,2290,2291
239,399,400,401,2285,2292,2293,2294,2288,2295,2296,2297,2291,2298,2299,2300
644,2301,2302,2303,650,2304,2305,2306,653,2307,2308,2309,401,2294,2297,2300
644,645,646,647,2301,2310,2311,2312,2302,2313,2314,2315,2303,2316,2317,2318
647,671,668,662,2312,2319,2320,2321,2315,2322,2323,2324,2318,2325,2326,2327
662,663,664,665,2321,2328,2329,2330,2324,2331,2332,2333,2327,2334,2335,2336
665,713,712,675,2330,2337,2338,2339,2333,2340,2341,2342,2336,2343,2344,2345
675,679,683,687,2339,2346,2347,2348,2342,2349,2350,2351,2345,2352,2353,2354
687,948,947,501,2348,2355,2356,2357,2351,2358,2359,2360,2354,2361,2362,2363
501,497,493,440,2357,2364,2365,2366,2360,2367,2368,2369,2363,2370,2371,2372
440,437,434,429,2366,2373,2374,2375,2369,2376,2377,2378,2372,2379,2380,2381
429,430,431,422,2375,2382,2383,2384,2378,2385,2386,2387,2381,2388,2389,2390
422,419,416,410,2384,2391,2392,2393,2387,2394,2395,2396,2390,2397,2398,2399
410,411,412,413,2393,2400,2401,2402,2396,2403,2404,2405,2399,2406,2407,2408
2409,2410,2411,2412,2413,2414,2415,2416,2417,2418,2419,2420,2421,2422,2423,2424
2409,2413,2417,2421,2425,2426,2427,2428,2429,2430,2431,2432,2433,2434,2435,2436
2409,2437,2438,13,2410,2439,2440,14,2411,2441,2442,15,2412,2443,2444,16
2409,2425,2429,2433,2437,2445,2446,2447,2438,2448,2449,2450,13,20,24,28
2451,2452,2453,2454,2455,2456,2457,2458,2459,2460,2461,2462,2463,2464,2465,2466
2451,2455,2459,2463,2467,2468,2469,2470,2471,2472,2473,2474,2475,2476,2477,2478
2451,2467,2471,2475,2479,2480,2481,2482,2483,2484,2485,2486,2487,2488,2489,2490
2451,2479,2483,2487,2452,2491,2492,2493,2453,2494,2495,2496,2454,2497,2498,2499
2475,2500,2501,2502,2482,2503,2504,2505,2486,2506,2507,2508,2490,2509,2510,2511
2475,2476,2477,2478,2500,2512,2513,2514,2501,2515,2516,2517,2502,2518,2519,2520
2521,2522,2523,2502,2524,2525,2526,2518,2527,2528,2529,2519,2530,2531,2532,2520
2521,2524,2527,2530,2533,2534,2535,2536,2537,2538,2539,2540,2541,2542,2543,2544
2521,2533,2537,2541,2545,2546,2547,2548,2549,2550,2551,2552,2553,2554,2555,2556
2521,2545,2549,2553,2522,2557,2558,2559,2523,2560,2561,2562,2502,2505,2508,2511
2541,2542,2543,2544,2563,2564,2565,2566,2567,2568,2569,2570,2571,2572,2573,2574
2541,2563,2567,2571,2548,2575,2576,2577,2552,2578,2579,2580,2556,2581,2582,2583
2487,2584,2585,2586,2493,2587,2588,2589,2496,2590,2591,2592,2499,2593,2594,2595
2487,2488,2489,2490,2584,2596,2597,2598,2585,2599,2600,2601,2586,2602,2603,2604
2490,2509,2510,2511,2598,2605,2606,2607,2601,2608,2609,2610,2604,2611,2612,2613
2553,2614,2615,2616,2559,2617,2618,2619,2562,2620,2621,2622,2511,2607,2610,2613
2553,2554,2555,2556,2614,2623,2624,2625,2615,2626,2627,2628,2616,2629,2630,2631
2556,2581,2582,2583,2625,2632,2633,2634,2628,2635,2636,2637,2631,2638,2639,2640
2641,2642,2643,2644,2645,2646,2647,2648,2649,2650,2651,2652,2653,2654,2655,2656
2641,2645,2649,2653,2657,2658,2659,2660,2661,2662,2663,2664,2665,2666,2667,2668
2641,2669,2670,2421,2657,2671,2672,2422,2661,2673,2674,2423,2665,2675,2676,2424
2641,2669,2670,2421,2642,2677,2678,2428,2643,2679,2680,2432,2644,2681,2682,2436
2665,2666,2667,2668,2683,2684,2685,2686,2687,2688,2689,2690,2691,2692,2693,2694
2665,2683,2687,2691,2675,2695,2696,2697,2676,2698,2699,2700,2424,2701,2702,2703
2704,2705,2706,2691,2707,2708,2709,2692,2710,2711,2712,2693,2713,2714,2715,2694
2704,2707,2710,2713,2716,2717,2718,2719,2720,2721,2722,2723,2724,2725,2726,2727
2704,2716,2720,2724,2728,2729,2730,2731,2732,2733,2734,2735,2736,2737,2738,2739
2704,2728,2732,2736,2705,2740,2741,2742,2706,2743,2744,2745,2691,2697,2700,2703
2586,2746,2747,2748,2589,2749,2750,2751,2592,2752,2753,2754,2595,2755,2756,2757
2586,2602,2603,2604,2746,2758,2759,2760,2747,2761,2762,2763,2748,2764,2765,2766
2604,2611,2612,2613,2760,2767,2768,2769,2763,2770,2771,2772,2766,2773,2774,2775
2616,2776,2777,2778,2619,2779,2780,2781,2622,2782,2783,2784,2613,2769,2772,2775
2616,2629,2630,2631,2776,2785,2786,2787,2777,2788,2789,2790,2778,2791,2792,2793
2631,2638,2639,2640,2787,2794,2795,2796,2790,2797,2798,2799,2793,2800,2801,2802
2433,2434,2435,2436,2803,2804,2805,2806,2807,2808,2809,2810,2811,2812,2813,2814
2433,2803,2807,2811,2447,2815,2816,2817,2450,2818,2819,2820,28,405,409,413
2644,2821,2822,2823,2648,2824,2825,2826,2652,2827,2828,2829,2656,2830,2831,2832
2644,2681,2682,2436,2821,2833,2834,2806,2822,2835,2836,2810,2823,2837,2838,2814
2839,2840,2841,2842,2843,2844,2845,2846,2847,2848,2849,2850,2851,2852,2853,2854
2839,2843,2847,2851,2855,2856,2857,2858,2859,2860,2861,2862,2863,2864,2865,2866
2839,2867,2868,2869,2840,2870,2871,2872,2841,2873,2874,2875,2842,2876,2877,2878
2839,2855,2859,2863,2867,2879,2880,2881,2868,2882,2883,2884,2869,2885,2886,2887
2656,2830,2831,2832,2888,2889,2890,2891,2892,2893,2894,2895,2896,2897,2898,2899
2656,2888,2892,2896,2655,2900,2901,2902,2654,2903,2904,2905,2653,2906,2907,2908
2653,2906,2907,2908,2660,2909,2910,2911,2664,2912,2913,2914,2668,2915,2916,2917
2668,2915,2916,2917,2686,2918,2919,2920,2690,2921,2922,2923,2694,2924,2925,2926
2694,2924,2925,2926,2715,2927,2928,2929,2714,2930,2931,2932,2713,2933,2934,2935
2713,2933,2934,2935,2719,2936,2937,2938,2723,2939,2940,2941,2727,2942,2943,2944
2945,2946,2947,2948,2949,2950,2951,2952,2953,2954,2955,2956,2748,2751,2754,2757
2945,2949,2953,2748,2957,2958,2959,2764,2960,2961,2962,2765,2963,2964,2965,2766
2945,2966,2967,2968,2946,2969,2970,2971,2947,2972,2973,2974,2948,2975,2976,2977
2945,2957,2960,2963,2966,2978,2979,2980,2967,2981,2982,2983,2968,2984,2985,2986
2963,2987,2988,2989,2980,2990,2991,2992,2983,2993,2994,2995,2986,2996,2997,2854
2963,2964,2965,2766,2987,2998,2999,2773,2988,3000,3001,2774,2989,3002,3003,2775
3004,3005,3006,2989,3007,3008,3009,3002,3010,3011,3012,3003,2778,2781,2784,2775
3004,3007,3010,2778,3013,3014,3015,2791,3016,3017,3018,2792,3019,3020,3021,2793
3004,3022,3023,2851,3005,3024,3025,2852,3006,3026,3027,2853,2989,2992,2995,2854
3004,3013,3016,3019,3022,3028,3029,3030,3023,3031,3032,3033,2851,2858,2862,2866
3019,3020,3021,2793,3034,3035,3036,2800,3037,3038,3039,2801,3040,3041,3042,2802
3019,3034,3037,3040,3030,3043,3044,3045,3033,3046,3047,3048,2866,3049,3050,3051
2863,2864,2865,2866,3052,3053,3054,3049,3055,3056,3057,3050,3058,3059,3060,3051
2863,3052,3055,3058,2881,3061,3062,3063,2884,3064,3065,3066,2887,3067,3068,3069
3070,3071,3072,3073,3074,3075,3076,3077,3078,3079,3080,3081,2887,2886,2885,2869
3070,3074,3078,2887,3082,3083,3084,3067,3085,3086,3087,3068,3088,3089,3090,3069
3070,3082,3085,3088,3091,3092,3093,3094,3095,3096,3097,3098,3099,3100,3101,3102
3070,3091,3095,3099,3071,3103,3104,3105,3072,3106,3107,3108,3073,3109,3110,3111
3112,3113,3114,3115,3116,3117,3118,3119,3120,3121,3122,3123,2595,2594,2593,2499
3112,3124,3125,3126,3113,3127,3128,3129,3114,3130,3131,3132,3115,3133,3134,3135
3112,3136,3137,3138,3124,3139,3140,3141,3125,3142,3143,3144,3126,3145,3146,3147
3112,3116,3120,2595,3136,3148,3149,2755,3137,3150,3151,2756,3138,3152,3153,2757
3154,3155,3156,3157,3158,3159,3160,3161,3162,3163,3164,3165,3166,3167,3168,3169
3154,3158,3162,3166,3170,3171,3172,3173,3174,3175,3176,3177,2454,2458,2462,2466
3154,3170,3174,2454,3178,3179,3180,2497,3181,3182,3183,2498,3115,3119,3123,2499
3154,3178,3181,3115,3155,3184,3185,3133,3156,3186,3187,3134,3157,3188,3189,3135
3157,3188,3189,3135,3190,3191,3192,3193,3194,3195,3196,3197,3198,3199,3200,3201
3157,3190,3194,3198,3161,3202,3203,3204,3165,3205,3206,3207,3169,3208,3209,3210
2727,2942,2943,2944,3211,3212,3213,3214,3215,3216,3217,3218,3219,3220,3221,3222
2727,3211,3215,3219,2726,3223,3224,3225,2725,3226,3227,3228,2724,3229,3230,3231
2986,3232,3233,3234,2985,3235,3236,3237,2984,3238,3239,3240,2968,3241,3242,3243
2986,2996,2997,2854,3232,3244,3245,2850,3233,3246,3247,2846,3234,3248,3249,2842
3250,3251,3252,3253,3254,3255,3256,3257,3258,3259,3260,3261,3222,3221,3220,3219
3250,3251,3252,3253,3262,3263,3264,3265,3266,3267,3268,3269,2977,2976,2975,2948
3250,3254,3258,3222,3270,3271,3272,3273,3274,3275,3276,3277,3278,3279,3280,3281
3250,3270,3274,3278,3262,3282,3283,3284,3266,3285,3286,3287,2977,3288,3289,3290
3222,3218,3214,2944,3273,3291,3292,3293,3277,3294,3295,3296,3281,3297,3298,3299
2944,2941,2938,2935,3293,3300,3301,3302,3296,3303,3304,3305,3299,3306,3307,3308
2935,2932,2929,2926,3302,3309,3310,3311,3305,3312,3313,3314,3308,3315,3316,3317
2926,2923,2920,2917,3311,3318,3319,3320,3314,3321,3322,3323,3317,3324,3325,3326
2908,3327,3328,3111,2905,3329,3330,3108,2902,3331,3332,3105,2896,3333,3334,3099
2908,3327,3328,3111,2911,3335,3336,3337,2914,3338,3339,3340,2917,3320,3323,3326
2896,3333,3334,3099,2897,3341,3342,3100,2898,3343,3344,3101,2899,3345,3346,3102
3253,3347,3348,3138,3257,3349,3350,3141,3261,3351,3352,3144,3219,3353,3354,3147
3253,3347,3348,3138,3265,3355,3356,3152,3269,3357,3358,3153,2948,2952,2956,2757
2724,3229,3230,3231,2731,3359,3360,3361,2735,3362,3363,3364,2739,3365,3366,3367
3368,3369,3370,3135,3371,3372,3373,3374,3375,3376,3377,3378,3379,3380,3381,3367
3368,3382,3383,3384,3369,3385,3386,3387,3370,3388,3389,3390,3135,3132,3129,3126
3384,3391,3392,3393,3387,3394,3395,3396,3390,3397,3398,3399,3126,3145,3146,3147
3400,3401,3402,3219,3403,3404,3405,3353,3406,3407,3408,3354,3393,3396,3399,3147
3400,3409,3410,3411,3401,3412,3413,3414,3402,3415,3416,3417,3219,3225,3228,3231
3411,3418,3419,3379,3414,3420,3421,3380,3417,3422,3423,3381,3231,3361,3364,3367
3135,3374,3378,3367,3193,3424,3425,3366,3197,3426,3427,3365,3201,3428,3429,2739
3169,3208,3209,3210,3430,3431,3432,3433,3434,3435,3436,3437,1034,1037,1040,1043
3169,3430,3434,1034,3168,3438,3439,1050,3167,3440,3441,1051,3166,3442,3443,1052
3166,3442,3443,1052,3173,3444,3445,1059,3177,3446,3447,1060,2466,3448,3449,1061
2463,2464,2465,2466,3450,3451,3452,3448,3453,3454,3455,3449,1064,1067,1070,1061
2463,3450,3453,1064,2470,3456,3457,1077,2474,3458,3459,1078,2478,3460,3461,1079
2478,3460,3461,1079,2514,3462,3463,1086,2517,3464,3465,1087,2520,3466,3467,1088
2530,2531,2532,2520,3468,3469,3470,3466,3471,3472,3473,3467,1091,1094,1097,1088
2530,3468,3471,1091,2536,3474,3475,1104,2540,3476,3477,1105,2544,3478,3479,1106
2544,3478,3479,1106,2566,3480,3481,1113,2570,3482,3483,1114,2574,3484,3485,1115
2736,3486,3487,3488,2737,3489,3490,3491,2738,3492,3493,3494,2739,3429,3428,3201
2736,3486,3487,3488,2742,3495,3496,3497,2745,3498,3499,3500,2703,3501,3502,3503
3210,3207,3204,3198,3504,3505,3506,3507,3508,3509,3510,3511,3512,3513,3514,3515
3210,3504,3508,3512,3433,3516,3517,3518,3437,3519,3520,3521,1043,1137,1141,1145
3512,3513,3514,3515,3522,3523,3524,3525,3526,3527,3528,3529,3530,3531,3532,3533
3512,3522,3526,3530,3518,3534,3535,3536,3521,3537,3538,3539,1145,1158,1162,1166
3533,3529,3525,3515,3540,3541,3542,3543,3544,3545,3546,3547,3548,3549,3550,3488
3533,3533,3533,3533,3551,3552,3552,3540,3553,3554,3555,3544,3556,3557,3558,3548
3533,3551,3553,3556,3532,3559,3560,3561,3531,3562,3563,3564,3530,3565,3566,3567
3530,3565,3566,3567,3536,3568,3569,3570,3539,3571,3572,3573,1166,1206,1209,1212
3567,3564,3561,3556,3574,3575,3576,3577,3578,3579,3580,3581,3582,3583,3584,3585
3567,3574,3578,3582,3570,3586,3587,3588,3573,3589,3590,3591,1212,1216,1220,1224
3556,3557,3558,3548,3577,3592,3593,3594,3581,3595,3596,3597,3585,3598,3599,3600
3548,3549,3550,3488,3594,3601,3602,3497,3597,3603,3604,3500,3600,3605,3606,3503
2703,3501,3502,3503,2702,3607,3608,3609,2701,3610,3611,3612,2424,3613,3614,3615
3600,3605,3606,3503,3616,3617,3618,3609,3619,3620,3621,3612,3622,3623,3624,3615
3600,3616,3619,3622,3599,3625,3626,3627,3598,3628,3629,3630,3585,3631,3632,3633
3585,3631,3632,3633,3584,3634,3635,3636,3583,3637,3638,3639,3582,3640,3641,3642
3582,3640,3641,3642,3588,3643,3644,3645,3591,3646,3647,3648,1224,1287,1290,1293
3201,3494,3491,3488,3200,3649,3650,3547,3199,3651,3652,3543,3198,3507,3511,3515
2977,3288,3289,3290,2974,3653,3654,3655,2971,3656,3657,3658,2968,3241,3242,3243
3659,3660,3661,3662,3663,3664,3665,3666,3667,3668,3669,3670,3671,3672,3673,3674
3659,3663,3667,3671,3675,3676,3677,3678,3679,3680,3681,3682,3683,3684,3685,3686
3659,3675,3679,3683,3687,3688,3689,3690,3691,3692,3693,3694,3695,3696,3697,3698
3659,3687,3691,3695,3660,3699,3700,3701,3661,3702,3703,3704,3662,3705,3706,3707
3695,3708,3709,3710,3701,3711,3712,3713,3704,3714,3715,3716,3707,3717,3718,3719
3695,3696,3697,3698,3708,3720,3721,3722,3709,3723,3724,3725,3710,3726,3727,3728
3710,3726,3727,3728,3729,3730,3731,3732,3733,3734,3735,3736,3737,3738,3739,3740
3710,3729,3733,3737,3713,3741,3742,3743,3716,3744,3745,3746,3719,3747,3748,3749
3737,3738,3739,3740,3750,3751,3752,3753,3754,3755,3756,3757,3758,3759,3760,3761
3737,3750,3754,3758,3743,3762,3763,3764,3746,3765,3766,3767,3749,3768,3769,3770
3758,3759,3760,3761,3771,3772,3773,3774,3775,3776,3777,3778,3779,3780,3781,3782
3758,3771,3775,3779,3764,3783,3784,3785,3767,3786,3787,3788,3770,3789,3790,3791
3779,3792,3793,3794,3785,3795,3796,3797,3788,3798,3799,3800,3791,3801,3802,3803
3779,3780,3781,3782,3792,3804,3805,3806,3793,3807,3808,3809,3794,3810,3811,3812
3794,3813,3814,3815,3797,3816,3817,3818,3800,3819,3820,3821,3803,3822,3823,3824
3794,3810,3811,3812,3813,3825,3826,3827,3814,3828,3829,3830,3815,3831,3832,3833
3815,3834,3835,3836,3818,3837,3838,3839,3821,3840,3841,3842,3824,3843,3844,3845
3815,3831,3832,3833,3834,3846,3847,3848,3835,3849,3850,3851,3836,3852,3853,3854
3836,3855,3856,3671,3839,3857,3858,3672,3842,3859,3860,3673,3845,3861,3862,3674
3836,3852,3853,3854,3855,3863,3864,3865,3856,3866,3867,3868,3671,3678,3682,3686
3683,3684,3685,3686,3869,3870,3871,3872,3873,3874,3875,3876,3877,3878,3879,3880
3683,3869,3873,3877,3690,3881,3882,3883,3694,3884,3885,3886,3698,3887,3888,3889
3698,3887,3888,3889,3722,3890,3891,3892,3725,3893,3894,3895,3728,3896,3897,3898
3728,3896,3897,3898,3732,3899,3900,3901,3736,3902,3903,3904,3740,3905,3906,3907
3740,3905,3906,3907,3753,3908,3909,3910,3757,3911,3912,3913,3761,3914,3915,3916
3761,3914,3915,3916,3774,3917,3918,3919,3778,3920,3921,3922,3782,3923,3924,3925
3782,3923,3924,3925,3806,3926,3927,3928,3809,3929,3930,3931,3812,3932,3933,3934
3812,3932,3933,3934,3827,3935,3936,3937,3830,3938,3939,3940,3833,3941,3942,3943
3833,3941,3942,3943,3848,3944,3945,3946,3851,3947,3948,3949,3854,3950,3951,3952
3854,3950,3951,3952,3865,3953,3954,3955,3868,3956,3957,3958,3686,3872,3876,3880
3877,3878,3879,3880,3959,3960,3961,3962,3963,3964,3965,3966,3967,3968,3969,3970
3877,3959,3963,3967,3883,3971,3972,3973,3886,3974,3975,3976,3889,3892,3895,3898
3877,3877,3877,3877,3883,3971,3971,3959,3886,3977,3978,3963,3889,3979,3980,3967
3889,3889,3889,3889,3979,3981,3981,3892,3980,3982,3975,3895,3967,3973,3976,3898
3934,3931,3928,3925,3937,3983,3984,3985,3940,3986,3987,3988,3943,3989,3990,3991
3934,3934,3934,3934,3992,3993,3993,3931,3994,3995,3984,3928,3991,3988,3985,3925
3934,3934,3934,3934,3992,3996,3996,3937,3994,3997,3986,3940,3991,3990,3989,3943
3943,3989,3990,3991,3946,3998,3999,4000,3949,4001,4002,4003,3952,4004,4005,4006
3880,3958,3955,3952,3962,4007,4008,4004,3966,4009,4010,4005,3970,4011,4012,4006
3916,3913,3910,3907,4013,4014,4015,4016,4017,4018,4019,4020,4006,4012,4011,3970
3916,4013,4017,4006,3919,4021,4022,4003,3922,4023,4024,4000,3925,3985,3988,3991
3907,3904,3901,3898,4016,4025,4026,3976,4020,4027,4028,3973,3970,3969,3968,3967
4029,4030,4031,4032,4033,4034,4035,4036,4037,4038,4039,4040,4041,4042,4043,4044
4029,4033,4037,4041,4045,4046,4047,4048,4049,4050,4051,4052,4053,4054,4055,4056
4029,4045,4049,4053,4057,4058,4059,4060,4061,4062,4063,4064,4065,4066,4067,4068
4029,4057,4061,4065,4030,4069,4070,4071,4031,4072,4073,4074,4032,4075,4076,4077
4065,4078,4079,4080,4071,4081,4082,4083,4074,4084,4085,4086,4077,4087,4088,4089
4065,4066,4067,4068,4078,4090,4091,4092,4079,4093,4094,4095,4080,4096,4097,4098
4080,4096,4097,4098,4099,4100,4101,4102,4103,4104,4105,4106,4107,4108,4109,4110
4080,4099,4103,4107,4083,4111,4112,4113,4086,4114,4115,4116,4089,4117,4118,4119
4107,4108,4109,4110,4120,4121,4122,4123,4124,4125,4126,4127,4128,4129,4130,4131
4107,4120,4124,4128,4113,4132,4133,4134,4116,4135,4136,4137,4119,4138,4139,4140
4128,4129,4130,4131,4141,4142,4143,4144,4145,4146,4147,4148,4149,4150,4151,4152
4128,4141,4145,4149,4134,4153,4154,4155,4137,4156,4157,4158,4140,4159,4160,4161
4149,4162,4163,4164,4155,4165,4166,4167,4158,4168,4169,4170,4161,4171,4172,4173
4149,4150,4151,4152,4162,4174,4175,4176,4163,4177,4178,4179,4164,4180,4181,4182
4164,4183,4184,4185,4167,4186,4187,4188,4170,4189,4190,4191,4173,4192,4193,4194
4164,4180,4181,4182,4183,4195,4196,4197,4184,4198,4199,4200,4185,4201,4202,4203
4185,4204,4205,4206,4188,4207,4208,4209,4191,4210,4211,4212,4194,4213,4214,4215
4185,4201,4202,4203,4204,4216,4217,4218,4205,4219,4220,4221,4206,4222,4223,4224
4206,4225,4226,4041,4209,4227,4228,4042,4212,4229,4230,4043,4215,4231,4232,4044
4206,4222,4223,4224,4225,4233,4234,4235,4226,4236,4237,4238,4041,4048,4052,4056
4032,4239,4240,4241,4036,4242,4243,4244,4040,4245,4246,4247,4044,4248,4249,4250
4032,4075,4076,4077,4239,4251,4252,4253,4240,4254,4255,4256,4241,4257,4258,2878
4077,4087,4088,4089,4253,4259,4260,4261,4256,4262,4263,4264,2878,4265,4266,4267
4089,4117,4118,4119,4261,4268,4269,4270,4264,4271,4272,4273,4267,4274,4275,4276
4119,4138,4139,4140,4270,4277,4278,4279,4273,4280,4281,4282,4276,4283,4284,4285
4140,4159,4160,4161,4279,4286,4287,4288,4282,4289,4290,4291,4285,4292,4293,4294
4161,4171,4172,4173,4288,4295,4296,4297,4291,4298,4299,4300,4294,4301,4302,4303
4173,4192,4193,4194,4297,4304,4305,4306,4300,4307,4308,4309,4303,4310,4311,4312
4194,4213,4214,4215,4306,4313,4314,4315,4309,4316,4317,4318,4312,4319,4320,4321
4215,4231,4232,4044,4315,4322,4323,4248,4318,4324,4325,4249,4321,4326,4327,4250
3317,3324,3325,3326,4328,4329,4330,4331,4332,4333,4334,4335,4276,4275,4274,4267
3317,4328,4332,4276,3316,4336,4337,4283,3315,4338,4339,4284,3308,4340,4341,4285
3308,4340,4341,4285,3307,4342,4343,4292,3306,4344,4345,4293,3299,4346,4347,4294
3278,3279,3280,3281,4348,4349,4350,4351,4352,4353,4354,4355,4312,4311,4310,4303
3278,4348,4352,4312,3284,4356,4357,4319,3287,4358,4359,4320,3290,4360,4361,4321
4294,4301,4302,4303,4347,4362,4363,4355,4346,4364,4365,4351,3299,3298,3297,3281
4250,4366,4367,3243,4327,4368,4369,3658,4326,4370,4371,3655,4321,4361,4360,3290
4250,4247,4244,4241,4366,4372,4373,4374,4367,4375,4376,4377,3243,3240,3237,3234
3326,3340,3337,3111,4331,4378,4379,3110,4335,4380,4381,3109,4267,4382,4383,3073
3234,3248,3249,2842,4377,4384,4385,2876,4374,4386,4387,2877,4241,4257,4258,2878
3073,3077,3081,2869,4383,4388,4389,2872,4382,4390,4391,2875,4267,4266,4265,2878
4182,4179,4176,4152,4392,4393,4394,4395,4396,4397,4398,4399,4400,4401,4402,4403
4182,4392,4396,4400,4197,4404,4405,4406,4200,4407,4408,4409,4203,4410,4411,4412
4203,4410,4411,4412,4218,4413,4414,4415,4221,4416,4417,4418,4224,4419,4420,4421
4224,4419,4420,4421,4235,4422,4423,4424,4238,4425,4426,4427,4056,4428,4429,4430
4056,4428,4429,4430,4055,4431,4432,4433,4054,4434,4435,4436,4053,4437,4438,4439
4053,4437,4438,4439,4060,4440,4441,4442,4064,4443,4444,4445,4068,4446,4447,4448
4068,4446,4447,4448,4092,4449,4450,4451,4095,4452,4453,4454,4098,4455,4456,4457
4098,4455,4456,4457,4102,4458,4459,4460,4106,4461,4462,4463,4110,4464,4465,4466
4110,4464,4465,4466,4123,4467,4468,4469,4127,4470,4471,4472,4131,4473,4474,4475
4131,4473,4474,4475,4144,4476,4477,4478,4148,4479,4480,4481,4152,4395,4399,4403
4400,4401,4402,4403,4482,4483,4484,4485,4486,4487,4488,4489,3803,3802,3801,3791
4400,4482,4486,3803,4406,4490,4491,3822,4409,4492,4493,3823,4412,4494,4495,3824
4412,4494,4495,3824,4415,4496,4497,3843,4418,4498,4499,3844,4421,4500,4501,3845
4421,4500,4501,3845,4424,4502,4503,3861,4427,4504,4505,3862,4430,4506,4507,3674
4430,4506,4507,3674,4433,4508,4509,3670,4436,4510,4511,3666,4439,4512,4513,3662
4439,4512,4513,3662,4442,4514,4515,3705,4445,4516,4517,3706,4448,4518,4519,3707
4448,4518,4519,3707,4451,4520,4521,3717,4454,4522,4523,3718,4457,4524,4525,3719
4457,4524,4525,3719,4460,4526,4527,3747,4463,4528,4529,3748,4466,4530,4531,3749
4466,4530,4531,3749,4469,4532,4533,3768,4472,4534,4535,3769,4475,4536,4537,3770
4475,4536,4537,3770,4478,4538,4539,3789,4481,4540,4541,3790,4403,4485,4489,3791
3615,4542,4543,4544,3614,4545,4546,4547,3613,4548,4549,4550,2424,4551,4552,4553
3615,3624,3623,3622,4542,4554,4555,4556,4543,4557,4558,4559,4544,4560,4561,4562
3622,3627,3630,3633,4556,4563,4564,4565,4559,4566,4567,4568,4562,4569,4570,4571
3633,3636,3639,3642,4565,4572,4573,4574,4568,4575,4576,4577,4571,4578,4579,4580
3642,3645,3648,1293,4574,4581,4582,2228,4577,4583,4584,2231,4580,4585,4586,2234
16,2250,2251,2252,2444,4587,4588,4589,2443,4590,4591,4592,2412,4593,4594,4595
2412,2416,2420,2424,4593,4596,4597,4551,4594,4598,4599,4552,4595,4600,4601,4553
1115,2262,2263,2264,3485,4602,4603,4604,3484,4605,4606,4607,2574,4608,4609,4610
2574,4608,4609,4610,2573,4611,4612,4613,2572,4614,4615,4616,2571,4617,4618,4619
2571,4617,4618,4619,2577,4620,4621,4622,2580,4623,4624,4625,2583,4626,4627,4628
2583,4626,4627,4628,2634,4629,4630,4631,2637,4632,4633,4634,2640,4635,4636,4637
2640,4635,4636,4637,2796,4638,4639,4640,2799,4641,4642,4643,2802,4644,4645,4646
3040,4647,4648,4649,3045,4650,4651,4652,3048,4653,4654,4655,3051,4656,4657,4658
3040,3041,3042,2802,4647,4659,4660,4644,4648,4661,4662,4645,4649,4663,4664,4646
3051,4656,4657,4658,3060,4665,4666,4667,3059,4668,4669,4670,3058,4671,4672,4673
3058,4671,4672,4673,3063,4674,4675,4676,3066,4677,4678,4679,3069,4680,4681,4682
3069,4680,4681,4682,3090,4683,4684,4685,3089,4686,4687,4688,3088,4689,4690,4691
3088,4689,4690,4691,3094,4692,4693,4694,3098,4695,4696,4697,3102,4698,4699,4700
3102,4698,4699,4700,3346,4701,4702,4703,3345,4704,4705,4706,2899,4707,4708,4709
2899,2895,2891,2832,4707,4710,4711,4712,4708,4713,4714,4715,4709,4716,4717,4718
2832,2829,2826,2823,4712,4719,4720,4721,4715,4722,4723,4724,4718,4725,4726,4727
2823,2837,2838,2814,4721,4728,4729,4730,4724,4731,4732,4733,4727,4734,4735,4736
2814,2813,2812,2811,4730,4737,4738,4739,4733,4740,4741,4742,4736,4743,4744,4745
2811,2817,2820,413,4739,4746,4747,2402,4742,4748,4749,2405,4745,4750,4751,2408
2225,2232,2233,2234,4752,4753,4754,4755,4756,4757,4758,4759,4760,4761,4762,4763
2225,4752,4756,4760,2224,4764,4765,4766,2223,4767,4768,4769,2216,4770,4771,4772
2216,4770,4771,4772,2215,4773,4774,4775,2214,4776,4777,4778,2198,4779,4780,4781
2198,4779,4780,4781,2197,4782,4783,4784,2196,4785,4786,4787,2195,4788,4789,4790
2195,4788,4789,4790,2201,4791,4792,4793,2204,4794,4795,4796,2207,4797,4798,4799
2207,4797,4798,4799,2243,4800,4801,4802,2242,4803,4804,4805,2241,4806,4807,4808
2241,4806,4807,4808,2246,4809,4810,4811,2249,4812,4813,4814,2252,4815,4816,4817
2252,4815,4816,4817,4589,4818,4819,4820,4592,4821,4822,4823,4595,4824,4825,4826
4595,4600,4601,4553,4824,4827,4828,4829,4825,4830,4831,4832,4826,4833,4834,4835
4553,4550,4547,4544,4829,4836,4837,4838,4832,4839,4840,4841,4835,4842,4843,4844
4544,4560,4561,4562,4838,4845,4846,4847,4841,4848,4849,4850,4844,4851,4852,4853
4562,4569,4570,4571,4847,4854,4855,4856,4850,4857,4858,4859,4853,4860,4861,4862
4571,4578,4579,4580,4856,4863,4864,4865,4859,4866,4867,4868,4862,4869,4870,4871
4580,4585,4586,2234,4865,4872,4873,4755,4868,4874,4875,4759,4871,4876,4877,4763
4878,4879,4880,4881,4882,4883,4884,4885,4886,4887,4888,4889,2399,2398,2397,2390
4878,4882,4886,2399,4890,4891,4892,2406,4893,4894,4895,2407,4896,4897,4898,2408
4896,4899,4900,4901,4897,4902,4903,4904,4898,4905,4906,4907,2408,4751,4750,4745
4901,4908,4909,4910,4904,4911,4912,4913,4907,4914,4915,4916,4745,4744,4743,4736
4910,4917,4918,4919,4913,4920,4921,4922,4916,4923,4924,4925,4736,4735,4734,4727
4919,4926,4927,4928,4922,4929,4930,4931,4925,4932,4933,4934,4727,4726,4725,4718
4928,4935,4936,4937,4931,4938,4939,4940,4934,4941,4942,4943,4718,4717,4716,4709
4937,4944,4945,4946,4940,4947,4948,4949,4943,4950,4951,4952,4709,4706,4703,4700
4946,4953,4954,4955,4949,4956,4957,4958,4952,4959,4960,4961,4700,4697,4694,4691
4955,4962,4963,4964,4958,4965,4966,4967,4961,4968,4969,4970,4691,4688,4685,4682
4964,4971,4972,4973,4967,4974,4975,4976,4970,4977,4978,4979,4682,4679,4676,4673
4973,4980,4981,4982,4976,4983,4984,4985,4979,4986,4987,4988,4673,4670,4667,4658
4982,4989,4990,4991,4985,4992,4993,4994,4988,4995,4996,4997,4658,4655,4652,4649
4991,4998,4999,5000,4994,5001,5002,5003,4997,5004,5005,5006,4649,4663,4664,4646
5000,5007,5008,5009,5003,5010,5011,5012,5006,5013,5014,5015,4646,4643,4640,4637
5009,5016,5017,5018,5012,5019,5020,5021,5015,5022,5023,5024,4637,4634,4631,4628
5018,5025,5026,5027,5021,5028,5029,5030,5024,5031,5032,5033,4628,4625,4622,4619
5027,5034,5035,5036,5030,5037,5038,5039,5033,5040,5041,5042,4619,4616,4613,4610
5036,5043,5044,5045,5039,5046,5047,5048,5042,5049,5050,5051,4610,4607,4604,2264
5045,5052,5053,5054,5048,5055,5056,5057,5051,5058,5059,5060,2264,2261,2258,2255
5054,5061,5062,5063,5057,5064,5065,5066,5060,5067,5068,5069,2255,2271,2272,2273
5063,5070,5071,5072,5066,5073,5074,5075,5069,5076,5077,5078,2273,2280,2281,2282
5072,5079,5080,5081,5075,5082,5083,5084,5078,5085,5086,5087,2282,2289,2290,2291
5081,5088,5089,5090,5084,5091,5092,5093,5087,5094,5095,5096,2291,2298,2299,2300
5090,5097,5098,5099,5093,5100,5101,5102,5096,5103,5104,5105,2300,2309,2306,2303
5099,5106,5107,5108,5102,5109,5110,5111,5105,5112,5113,5114,2303,2316,2317,2318
5108,5115,5116,5117,5111,5118,5119,5120,5114,5121,5122,5123,2318,2325,2326,2327
5117,5124,5125,5126,5120,5127,5128,5129,5123,5130,5131,5132,2327,2334,2335,2336
5126,5133,5134,5135,5129,5136,5137,5138,5132,5139,5140,5141,2336,2343,2344,2345
5135,5142,5143,5144,5138,5145,5146,5147,5141,5148,5149,5150,2345,2352,2353,2354
5144,5151,5152,5153,5147,5154,5155,5156,5150,5157,5158,5159,2354,2361,2362,2363
5153,5160,5161,5162,5156,5163,5164,5165,5159,5166,5167,5168,2363,2370,2371,2372
5162,5169,5170,5171,5165,5172,5173,5174,5168,5175,5176,5177,2372,2379,2380,2381
5171,5178,5179,4881,5174,5180,5181,4885,5177,5182,5183,4889,2381,2388,2389,2390
5184,5185,5186,4760,5187,5188,5189,4761,5190,5191,5192,4762,5193,5194,5195,4763
5184,5196,5197,5198,5185,5199,5200,5201,5186,5202,5203,5204,4760,4766,4769,4772
5198,5205,5206,5207,5201,5208,5209,5210,5204,5211,5212,5213,4772,4775,4778,4781
5207,5214,5215,5216,5210,5217,5218,5219,5213,5220,5221,5222,4781,4784,4787,4790
5216,5223,5224,5225,5219,5226,5227,5228,5222,5229,5230,5231,4790,4793,4796,4799
5225,5232,5233,5234,5228,5235,5236,5237,5231,5238,5239,5240,4799,4802,4805,4808
5234,5241,5242,5243,5237,5244,5245,5246,5240,5247,5248,5249,4808,4811,4814,4817
5243,5250,5251,5252,5246,5253,5254,5255,5249,5256,5257,5258,4817,4820,4823,4826
5252,5259,5260,5261,5255,5262,5263,5264,5258,5265,5266,5267,4826,4833,4834,4835
5261,5268,5269,5270,5264,5271,5272,5273,5267,5274,5275,5276,4835,4842,4843,4844
5270,5277,5278,5279,5273,5280,5281,5282,5276,5283,5284,5285,4844,4851,4852,4853
5279,5286,5287,5288,5282,5289,5290,5291,5285,5292,5293,5294,4853,4860,4861,4862
5288,5295,5296,5297,5291,5298,5299,5300,5294,5301,5302,5303,4862,4869,4870,4871
5297,5304,5305,5193,5300,5306,5307,5194,5303,5308,5309,5195,4871,4876,4877,4763
5310,5311,5312,5313,5314,5310,5315,5316,5317,5318,5319,5320,5321,5322,5323,5319
5310,5310,5310,5310,5324,5325,5325,5314,5326,5327,5318,5317,5319,5323,5322,5321
5310,5314,5317,5321,5311,5310,5328,5329,5312,5330,5331,5332,5313,5333,5334,5331
5310,5310,5310,5310,5335,5336,5336,5314,5337,5338,5328,5317,5331,5332,5329,5321
5310,5314,5317,5321,5335,5336,5321,5339,5337,5331,5340,5341,5331,5342,5343,5344
5310,5314,5317,5321,5324,5325,5321,5339,5326,5319,5345,5341,5319,5346,5347,5344
5310,5310,5310,5310,5335,5348,5348,5311,5337,5349,5330,5312,5331,5334,5333,5313
5310,5310,5310,5310,5311,5350,5350,5324,5312,5315,5351,5326,5313,5316,5320,5319
5310,5335,5337,5331,5311,5348,5331,5342,5312,5313,5352,5343,5313,5353,5354,5344
5310,5311,5312,5313,5324,5350,5313,5353,5326,5319,5355,5354,5319,5346,5347,5344
5310,5324,5326,5319,5335,5310,5351,5320,5337,5349,5313,5316,5331,5334,5333,5313
5310,5324,5326,5319,5335,5310,5327,5323,5337,5338,5321,5322,5331,5332,5329,5321
5344,5354,5353,5313,5341,5344,5356,5333,5339,5357,5331,5334,5321,5329,5332,5331
5344,5344,5344,5344,5343,5352,5352,5354,5342,5358,5356,5353,5331,5334,5333,5313
5344,5341,5339,5321,5354,5344,5359,5322,5353,5360,5319,5323,5313,5316,5320,5319
5344,5344,5344,5344,5347,5355,5355,5354,5346,5361,5360,5353,5319,5320,5316,5313
5344,5344,5344,5344,5347,5345,5345,5341,5346,5362,5359,5339,5319,5323,5322,5321
5344,5344,5344,5344,5343,5340,5340,5341,5342,5363,5357,5339,5331,5332,5329,5321
5344,5343,5342,5331,5347,5344,5363,5332,5346,5362,5321,5329,5319,5323,5322,5321
5344,5343,5342,5331,5347,5344,5358,5334,5346,5361,5313,5333,5319,5320,5316,5313
5364,5365,5366,5367,5368,5364,5369,5370,5371,5372,5373,5374,5375,5376,5377,5373
5364,5364,5364,5364,5378,5379,5379,5368,5380,5381,5372,5371,5373,5377,5376,5375
5364,5368,5371,5375,5365,5364,5382,5383,5366,5384,5385,5386,5367,5387,5388,5385
5364,5364,5364,5364,5389,5390,5390,5368,5391,5392,5382,5371,5385,5386,5383,5375
5364,5368,5371,5375,5389,5390,5375,5393,5391,5385,5394,5395,5385,5396,5397,5398
5364,5368,5371,5375,5378,5379,5375,5393,5380,5373,5399,5395,5373,5400,5401,5398
5364,5364,5364,5364,5389,5402,5402,5365,5391,5403,5384,5366,5385,5388,5387,5367
5364,5364,5364,5364,5365,5404,5404,5378,5366,5369,5405,5380,5367,5370,5374,5373
5364,5389,5391,5385,5365,5402,5385,5396,5366,5367,5406,5397,5367,5407,5408,5398
5364,5365,5366,5367,5378,5404,5367,5407,5380,5373,5409,5408,5373,5400,5401,5398
5364,5378,5380,5373,5389,5364,5405,5374,5391,5403,5367,5370,5385,5388,5387,5367
5364,5378,5380,5373,5389,5364,5381,5377,5391,5392,5375,5376,5385,5386,5383,5375
5398,5408,5407,5367,5395,5398,5410,5387,5393,5411,5385,5388,5375,5383,5386,5385
5398,5398,5398,5398,5397,5406,5406,5408,5396,5412,5410,5407,5385,5388,5387,5367
5398,5395,5393,5375,5408,5398,5413,5376,5407,5414,5373,5377,5367,5370,5374,5373
5398,5398,5398,5398,5401,5409,5409,5408,5400,5415,5414,5407,5373,5374,5370,5367
5398,5398,5398,5398,5401,5399,5399,5395,5400,5416,5413,5393,5373,5377,5376,5375
5398,5398,5398,5398,5397,5394,5394,5395,5396,5417,5411,5393,5385,5386,5383,5375
5398,5397,5396,5385,5401,5398,5417,5386,5400,5416,5375,5383,5373,5377,5376,5375
5398,5397,5396,5385,5401,5398,5412,5388,5400,5415,5367,5387,5373,5374,5370,5367
5418,5419,5420,5421,5422,5418,5423,5424,5425,5426,5427,5428,5429,5430,5431,5427
5418,5418,5418,5418,5432,5433,5433,5422,5434,5435,5426,5425,5427,5431,5430,5429
5418,5422,5425,5429,5419,5418,5436,5437,5420,5438,5439,5440,5421,5441,5442,5439
5418,5418,5418,5418,5443,5444,5444,5422,5445,5446,5436,5425,5439,5440,5437,5429
5418,5422,5425,5429,5443,5444,5429,5447,5445,5439,5448,5449,5439,5450,5451,5452
5418,5422,5425,5429,5432,5433,5429,5447,5434,5427,5453,5449,5427,5454,5455,5452
5418,5418,5418,5418,5443,5456,5456,5419,5445,5457,5438,5420,5439,5442,5441,5421
5418,5418,5418,5418,5419,5458,5458,5432,5420,5423,5459,5434,5421,5424,5428,5427
5418,5443,5445,5439,5419,5456,5439,5450,5420,5421,5460,5451,5421,5461,5462,5452
5418,5419,5420,5421,5432,5458,5421,5461,5434,5427,5463,5462,5427,5454,5455,5452
5418,5432,5434,5427,5443,5418,5459,5428,5445,5457,5421,5424,5439,5442,5441,5421
5418,5432,5434,5427,5443,5418,5435,5431,5445,5446,5429,5430,5439,5440,5437,5429
5452,5462,5461,5421,5449,5452,5464,5441,5447,5465,5439,5442,5429,5437,5440,5439
5452,5452,5452,5452,5451,5460,5460,5462,5450,5466,5464,5461,5439,5442,5441,5421
5452,5449,5447,5429,5462,5452,5467,5430,5461,5468,5427,5431,5421,5424,5428,5427
5452,5452,5452,5452,5455,5463,5463,5462,5454,5469,5468,5461,5427,5428,5424,5421
5452,5452,5452,5452,5455,5453,5453,5449,5454,5470,5467,5447,5427,5431,5430,5429
5452,5452,5452,5452,5451,5448,5448,5449,5450,5471,5465,5447,5439,5440,5437,5429
5452,5451,5450,5439,5455,5452,5471,5440,5454,5470,5429,5437,5427,5431,5430,5429
5452,5451,5450,5439,5455,5452,5466,5442,5454,5469,5421,5441,5427,5428,5424,5421
5472,5473,5474,5475,5476,5472,5477,5478,5479,5480,5481,5482,5483,5484,5485,5481
5472,5472,5472,5472,5486,5487,5487,5476,5488,5489,5480,5479,5481,5485,5484,5483
5472,5476,5479,5483,5473,5472,5490,5491,5474,5492,5493,5494,5475,5495,5496,5493
5472,5472,5472,5472,5497,5498,5498,5476,5499,5500,5490,5479,5493,5494,5491,5483
5472,5476,5479,5483,5497,5498,5483,5501,5499,5493,5502,5503,5493,5504,5505,5506
5472,5476,5479,5483,5486,5487,5483,5501,5488,5481,5507,5503,5481,5508,5509,5506
5472,5472,5472,5472,5497,5510,5510,5473,5499,5511,5492,5474,5493,5496,5495,5475
5472,5472,5472,5472,5473,5512,5512,5486,5474,5477,5513,5488,5475,5478,5482,5481
5472,5497,5499,5493,5473,5510,5493,5504,5474,5475,5514,5505,5475,5515,5516,5506
5472,5473,5474,5475,5486,5512,5475,5515,5488,5481,5517,5516,5481,5508,5509,5506
5472,5486,5488,5481,5497,5472,5513,5482,5499,5511,5475,5478,5493,5496,5495,5475
5472,5486,5488,5481,5497,5472,5489,5485,5499,5500,5483,5484,5493,5494,5491,5483
5506,5516,5515,5475,5503,5506,5518,5495,5501,5519,5493,5496,5483,5491,5494,5493
5506,5506,5506,5506,5505,5514,5514,5516,5504,5520,5518,5515,5493,5496,5495,5475
5506,5503,5501,5483,5516,5506,5521,5484,5515,5522,5481,5485,5475,5478,5482,5481
5506,5506,5506,5506,5509,5517,5517,5516,5508,5523,5522,5515,5481,5482,5478,5475
5506,5506,5506,5506,5509,5507,5507,5503,5508,5524,5521,5501,5481,5485,5484,5483
5506,5506,5506,5506,5505,5502,5502,5503,5504,5525,5519,5501,5493,5494,5491,5483
5506,5505,5504,5493,5509,5506,5525,5494,5508,5524,5483,5491,5481,5485,5484,5483
5506,5505,5504,5493,5509,5506,5520,5496,5508,5523,5475,5495,5481,5482,5478,5475
5525
0.176,-0.582,-0.753
0.175,-0.631,-0.688
0.225,-0.644,-0.58
0.235,-0.677,-0.506
0.144,-0.598,-0.759
0.143,-0.647,-0.694
0.18,-0.665,-0.602
0.19,-0.699,-0.529
0.099,-0.622,-0.765
0.098,-0.666,-0.707
0.117,-0.69,-0.611
0.117,-0.72,-0.545
0.062,-0.622,-0.765
0.062,-0.666,-0.707
0.062,-0.69,-0.611
0.062,-0.72,-0.545
0.176,-0.54,-0.808
0.144,-0.556,-0.814
0.099,-0.584,-0.815
0.062,-0.584,-0.815
0.23,-0.452,-0.835
0.179,-0.475,-0.841
0.118,-0.507,-0.855
0.062,-0.507,-0.855
0.239,-0.397,-0.876
0.188,-0.42,-0.882
0.118,-0.456,-0.891
0.062,-0.456,-0.891
0.238,-0.551,-0.743
0.319,-0.504,-0.726
0.377,-0.467,-0.712
0.238,-0.6,-0.678
0.323,-0.553,-0.63
0.381,-0.516,-0.615
0.291,-0.612,-0.546
0.41,-0.536,-0.495
0.428,-0.528,-0.469
0.301,-0.646,-0.473
0.404,-0.557,-0.389
0.422,-0.549,-0.363
0.239,-0.509,-0.798
0.282,-0.428,-0.829
0.291,-0.373,-0.87
0.316,-0.474,-0.784
0.356,-0.385,-0.818
0.359,-0.336,-0.861
0.375,-0.437,-0.77
0.4,-0.35,-0.806
0.403,-0.301,-0.849
0.198,-0.427,0.522
0.181,-0.473,0.481
0.162,-0.549,0.426
0.151,-0.56,0.365
0.237,-0.422,0.519
0.22,-0.468,0.478
0.201,-0.548,0.43
0.19,-0.559,0.369
0.289,-0.419,0.516
0.279,-0.468,0.481
0.261,-0.557,0.443
0.24,-0.571,0.387
0.328,-0.415,0.514
0.318,-0.463,0.479
0.3,-0.555,0.448
0.279,-0.569,0.393
0.221,-0.363,0.58
0.26,-0.358,0.578
0.304,-0.346,0.569
0.343,-0.342,0.567
0.272,-0.269,0.642
0.297,-0.26,0.639
0.334,-0.251,0.642
0.359,-0.243,0.637
0.282,-0.196,0.695
0.307,-0.187,0.692
0.339,-0.173,0.69
0.363,-0.165,0.685
0.177,-0.43,0.523
0.149,-0.436,0.527
0.129,-0.441,0.529
0.16,-0.475,0.483
0.142,-0.48,0.486
0.121,-0.484,0.488
0.153,-0.549,0.426
0.137,-0.553,0.433
0.129,-0.557,0.436
0.141,-0.56,0.364
0.134,-0.57,0.376
0.126,-0.573,0.378
0.2,-0.365,0.582
0.244,-0.278,0.646
0.253,-0.206,0.698
0.16,-0.375,0.585
0.211,-0.286,0.652
0.215,-0.218,0.705
0.139,-0.379,0.587
0.182,-0.293,0.655
0.186,-0.226,0.708
0.295,-0.096,0.767
0.271,0.031,0.878
0.272,0.143,0.931
0.266,-0.106,0.77
0.239,0.019,0.89
0.241,0.131,0.944
0.221,-0.119,0.783
0.194,0.005,0.898
0.197,0.117,0.957
0.192,-0.126,0.786
0.159,-0.003,0.904
0.162,0.109,0.963
0.32,-0.087,0.763
0.345,-0.071,0.753
0.369,-0.063,0.748
0.3,0.042,0.866
0.335,0.063,0.852
0.362,0.072,0.836
0.301,0.153,0.919
0.336,0.173,0.899
0.363,0.183,0.882
0.292,0.558,1.056
0.288,0.431,1.085
0.274,0.26,0.987
0.314,0.558,1.037
0.311,0.431,1.066
0.303,0.271,0.976
0.345,0.562,1.012
0.336,0.448,1.044
0.336,0.283,0.945
0.365,0.563,0.99
0.356,0.449,1.022
0.363,0.292,0.928
0.296,0.73,1.016
0.319,0.73,0.997
0.357,0.73,0.966
0.377,0.73,0.944
0.298,0.972,0.884
0.337,0.966,0.863
0.387,0.967,0.839
0.418,0.96,0.808
0.298,1.044,0.723
0.336,1.039,0.702
0.393,1.034,0.679
0.424,1.028,0.648
0.266,0.559,1.076
0.235,0.563,1.109
0.205,0.566,1.121
0.263,0.432,1.106
0.232,0.42,1.137
0.201,0.423,1.149
0.242,0.249,1.0
0.2,0.246,1.024
0.165,0.238,1.03
0.271,0.731,1.037
0.265,0.976,0.902
0.264,1.049,0.741
0.239,0.741,1.074
0.225,0.981,0.931
0.219,1.056,0.766
0.209,0.744,1.086
0.189,0.988,0.941
0.183,1.063,0.776
0.295,1.218,0.337
0.334,1.213,0.316
0.407,1.192,0.308
0.438,1.185,0.277
0.285,1.266,-0.24
0.33,1.251,-0.245
0.385,1.237,-0.267
0.427,1.214,-0.276
0.282,1.332,-0.658
0.327,1.317,-0.663
0.386,1.292,-0.667
0.427,1.268,-0.675
0.262,1.223,0.355
0.246,1.28,-0.236
0.242,1.345,-0.654
0.207,1.235,0.369
0.197,1.289,-0.217
0.189,1.358,-0.647
0.17,1.242,0.379
0.156,1.295,-0.214
0.148,1.364,-0.644
0.376,-0.41,0.51
0.366,-0.458,0.476
0.348,-0.553,0.455
0.328,-0.566,0.399
0.445,-0.413,0.518
0.435,-0.461,0.491
0.413,-0.531,0.46
0.393,-0.557,0.414
0.487,-0.394,0.502
0.477,-0.443,0.476
0.46,-0.521,0.45
0.44,-0.546,0.405
0.391,-0.336,0.563
0.462,-0.332,0.562
0.504,-0.313,0.547
0.409,-0.226,0.626
0.475,-0.221,0.631
0.519,-0.202,0.607
0.414,-0.149,0.674
0.486,-0.134,0.665
0.53,-0.115,0.641
0.42,-0.047,0.737
0.499,-0.028,0.706
0.544,-0.009,0.681
0.425,0.094,0.798
0.509,0.121,0.78
0.564,0.135,0.73
0.425,0.205,0.845
0.513,0.233,0.801
0.569,0.247,0.751
0.425,0.565,0.925
0.417,0.451,0.956
0.425,0.314,0.89
0.506,0.568,0.837
0.507,0.478,0.872
0.517,0.327,0.818
0.559,0.566,0.764
0.56,0.477,0.8
0.573,0.342,0.768
0.437,0.732,0.878
0.504,0.713,0.779
0.557,0.712,0.707
0.467,0.949,0.758
0.524,0.943,0.663
0.559,0.929,0.602
0.473,1.017,0.598
0.525,1.002,0.518
0.56,0.988,0.458
0.487,1.174,0.226
0.527,1.137,0.188
0.563,1.123,0.128
0.472,1.189,-0.285
0.526,1.161,-0.348
0.567,1.131,-0.362
0.473,1.243,-0.684
0.528,1.202,-0.701
0.568,1.172,-0.715
0.55,-0.343,-0.66
0.54,-0.406,-0.59
0.528,-0.454,-0.47
0.511,-0.493,-0.385
0.502,-0.385,-0.677
0.491,-0.448,-0.607
0.501,-0.471,-0.466
0.484,-0.51,-0.381
0.431,-0.432,-0.698
0.435,-0.482,-0.602
0.455,-0.511,-0.473
0.449,-0.531,-0.367
0.558,-0.294,-0.715
0.51,-0.336,-0.731
0.429,-0.403,-0.756
0.57,-0.183,-0.731
0.523,-0.236,-0.752
0.457,-0.305,-0.791
0.57,-0.133,-0.784
0.523,-0.185,-0.806
0.459,-0.256,-0.834
0.596,-0.304,-0.644
0.604,-0.255,-0.699
0.607,-0.141,-0.714
0.607,-0.091,-0.767
0.656,-0.248,-0.624
0.656,-0.191,-0.67
0.658,-0.083,-0.69
0.655,-0.034,-0.743
0.691,-0.199,-0.604
0.69,-0.142,-0.65
0.686,-0.037,-0.669
0.683,0.013,-0.722
0.586,-0.367,-0.575
0.657,-0.31,-0.573
0.692,-0.261,-0.553
0.581,-0.419,-0.478
0.645,-0.375,-0.468
0.688,-0.327,-0.467
0.564,-0.458,-0.393
0.646,-0.424,-0.404
0.689,-0.377,-0.403
0.504,-0.51,-0.348
0.483,-0.521,-0.296
0.47,-0.531,-0.258
0.477,-0.527,-0.344
0.445,-0.532,-0.29
0.433,-0.542,-0.252
0.446,-0.54,-0.319
0.405,-0.548,-0.278
0.384,-0.552,-0.234
0.42,-0.558,-0.316
0.37,-0.559,-0.264
0.349,-0.563,-0.221
0.557,-0.476,-0.356
0.647,-0.452,-0.369
0.69,-0.404,-0.368
0.551,-0.5,-0.307
0.639,-0.481,-0.325
0.699,-0.441,-0.32
0.539,-0.51,-0.269
0.639,-0.495,-0.282
0.7,-0.455,-0.277
0.453,-0.539,-0.202
0.451,-0.539,-0.18
0.451,-0.53,-0.15
0.454,-0.532,-0.128
0.409,-0.549,-0.208
0.406,-0.549,-0.186
0.416,-0.541,-0.172
0.419,-0.543,-0.15
0.355,-0.558,-0.166
0.351,-0.558,-0.152
0.351,-0.56,-0.133
0.354,-0.561,-0.118
0.315,-0.566,-0.144
0.312,-0.566,-0.13
0.315,-0.562,-0.11
0.317,-0.562,-0.096
0.456,-0.538,-0.22
0.411,-0.548,-0.225
0.361,-0.558,-0.191
0.322,-0.566,-0.169
0.465,-0.535,-0.241
0.428,-0.546,-0.235
0.373,-0.553,-0.212
0.338,-0.564,-0.198
0.531,-0.521,-0.192
0.533,-0.521,-0.21
0.534,-0.514,-0.253
0.627,-0.518,-0.125
0.642,-0.514,-0.178
0.64,-0.513,-0.23
0.692,-0.481,-0.096
0.708,-0.477,-0.149
0.7,-0.473,-0.225
0.528,-0.521,-0.17
0.61,-0.521,-0.067
0.676,-0.485,-0.039
0.515,-0.511,-0.11
0.607,-0.497,0.013
0.625,-0.484,0.027
0.518,-0.513,-0.088
0.578,-0.491,0.066
0.596,-0.479,0.08
0.534,-0.373,0.485
0.524,-0.421,0.458
0.503,-0.511,0.442
0.484,-0.536,0.396
0.57,-0.315,0.443
0.56,-0.377,0.416
0.551,-0.477,0.415
0.538,-0.521,0.364
0.606,-0.29,0.41
0.597,-0.352,0.383
0.581,-0.458,0.387
0.568,-0.503,0.336
0.551,-0.292,0.529
0.581,-0.241,0.475
0.617,-0.216,0.443
0.559,-0.186,0.584
0.598,-0.145,0.538
0.629,-0.126,0.505
0.57,-0.098,0.618
0.605,-0.07,0.567
0.636,-0.05,0.534
0.584,0.008,0.659
0.613,0.02,0.601
0.644,0.039,0.568
0.599,0.144,0.699
0.625,0.161,0.651
0.651,0.163,0.61
0.604,0.256,0.719
0.63,0.256,0.658
0.656,0.258,0.617
0.598,0.566,0.712
0.599,0.477,0.747
0.608,0.351,0.736
0.637,0.564,0.633
0.636,0.485,0.677
0.635,0.346,0.664
0.669,0.557,0.576
0.668,0.478,0.621
0.661,0.348,0.623
0.596,0.711,0.654
0.638,0.696,0.558
0.67,0.688,0.502
0.596,0.914,0.539
0.638,0.892,0.445
0.664,0.873,0.377
0.597,0.973,0.394
0.636,0.949,0.305
0.663,0.93,0.238
0.599,1.108,0.065
0.634,1.063,0.026
0.66,1.044,-0.042
0.595,1.11,-0.371
0.635,1.088,-0.441
0.657,1.062,-0.454
0.596,1.151,-0.725
0.634,1.124,-0.74
0.656,1.099,-0.753
0.247,-0.347,-0.913
0.197,-0.37,-0.919
0.119,-0.406,-0.928
0.063,-0.406,-0.928
0.234,-0.262,-0.952
0.187,-0.285,-0.951
0.117,-0.316,-0.954
0.064,-0.316,-0.954
0.234,-0.237,-1.009
0.186,-0.261,-1.009
0.118,-0.287,-1.009
0.065,-0.287,-1.009
0.299,-0.324,-0.906
0.288,-0.235,-0.953
0.287,-0.21,-1.01
0.362,-0.281,-0.91
0.35,-0.188,-0.944
0.349,-0.158,-1.011
0.406,-0.246,-0.897
0.397,-0.151,-0.948
0.396,-0.121,-1.014
0.57,-0.075,-0.846
0.523,-0.127,-0.867
0.462,-0.201,-0.882
0.552,-0.018,-0.949
0.515,-0.065,-0.944
0.445,-0.115,-0.951
0.548,0.005,-1.03
0.511,-0.041,-1.026
0.443,-0.085,-1.018
0.607,-0.033,-0.829
0.596,0.038,-0.954
0.593,0.062,-1.035
0.649,0.045,-0.827
0.632,0.113,-0.936
0.628,0.153,-1.043
0.677,0.092,-0.806
0.666,0.177,-0.938
0.662,0.217,-1.046
0.872,0.255,0.1
0.872,0.22,0.133
0.87,0.188,0.192
0.862,0.15,0.219
0.879,0.225,0.064
0.879,0.19,0.097
0.884,0.168,0.148
0.876,0.129,0.175
0.882,0.179,0.016
0.883,0.151,0.039
0.877,0.14,0.078
0.879,0.118,0.105
0.883,0.166,-0.029
0.884,0.138,-0.006
0.887,0.114,0.036
0.888,0.091,0.064
0.874,0.383,-0.022
0.881,0.354,-0.058
0.879,0.308,-0.09
0.881,0.296,-0.135
0.878,0.613,-0.139
0.885,0.591,-0.161
0.886,0.565,-0.192
0.885,0.542,-0.213
0.87,0.694,-0.297
0.877,0.672,-0.319
0.878,0.637,-0.344
0.877,0.613,-0.365
0.86,0.304,0.159
0.844,0.384,0.228
0.821,0.43,0.287
0.86,0.27,0.192
0.841,0.339,0.274
0.819,0.385,0.333
0.856,0.21,0.24
0.824,0.278,0.311
0.801,0.295,0.357
0.848,0.171,0.267
0.818,0.214,0.319
0.795,0.231,0.365
0.862,0.433,0.037
0.868,0.642,-0.11
0.86,0.723,-0.268
0.85,0.509,0.1
0.845,0.694,-0.066
0.838,0.763,-0.231
0.828,0.556,0.159
0.83,0.713,-0.031
0.823,0.782,-0.196
0.715,0.067,-0.698
0.71,0.145,-0.781
0.691,0.224,-0.94
0.687,0.264,-1.047
0.744,0.143,-0.66
0.74,0.227,-0.763
0.726,0.291,-0.918
0.722,0.326,-1.046
0.769,0.199,-0.634
0.764,0.283,-0.738
0.751,0.338,-0.914
0.746,0.373,-1.043
0.718,0.017,-0.645
0.747,0.094,-0.601
0.772,0.151,-0.575
0.722,-0.098,-0.632
0.746,-0.019,-0.602
0.769,0.028,-0.578
0.723,-0.154,-0.586
0.745,-0.084,-0.561
0.768,-0.037,-0.537
0.724,-0.217,-0.535
0.745,-0.144,-0.524
0.768,-0.097,-0.5
0.725,-0.287,-0.466
0.743,-0.211,-0.447
0.765,-0.164,-0.43
0.726,-0.336,-0.403
0.744,-0.26,-0.396
0.766,-0.213,-0.379
0.727,-0.364,-0.368
0.746,-0.299,-0.355
0.767,-0.251,-0.339
0.738,-0.415,-0.317
0.759,-0.34,-0.307
0.775,-0.304,-0.283
0.738,-0.43,-0.274
0.759,-0.364,-0.256
0.776,-0.328,-0.233
0.739,-0.447,-0.222
0.759,-0.386,-0.21
0.776,-0.35,-0.186
0.743,-0.457,-0.134
0.759,-0.398,-0.121
0.777,-0.359,-0.116
0.728,-0.461,-0.081
0.75,-0.401,-0.071
0.768,-0.362,-0.065
0.711,-0.465,-0.023
0.742,-0.404,-0.026
0.76,-0.365,-0.021
0.638,-0.476,0.037
0.691,-0.392,0.069
0.724,-0.35,0.063
0.609,-0.47,0.09
0.686,-0.39,0.084
0.72,-0.347,0.078
0.677,-0.263,0.331
0.667,-0.316,0.304
0.645,-0.391,0.273
0.625,-0.432,0.234
0.699,-0.25,0.304
0.689,-0.304,0.277
0.674,-0.352,0.245
0.653,-0.393,0.205
0.719,-0.227,0.263
0.714,-0.253,0.248
0.697,-0.306,0.213
0.692,-0.329,0.193
0.744,-0.211,0.241
0.739,-0.237,0.226
0.731,-0.265,0.197
0.726,-0.288,0.177
0.691,-0.188,0.37
0.713,-0.175,0.342
0.73,-0.165,0.299
0.755,-0.149,0.276
0.699,-0.086,0.423
0.722,-0.077,0.392
0.75,-0.064,0.356
0.772,-0.065,0.324
0.706,-0.005,0.45
0.729,0.004,0.42
0.758,0.005,0.376
0.78,0.004,0.344
0.658,-0.274,0.355
0.628,-0.275,0.391
0.647,-0.328,0.328
0.618,-0.337,0.364
0.625,-0.419,0.294
0.608,-0.441,0.361
0.604,-0.46,0.255
0.595,-0.486,0.31
0.672,-0.199,0.394
0.678,-0.094,0.45
0.685,-0.013,0.478
0.639,-0.201,0.423
0.651,-0.112,0.481
0.658,-0.037,0.51
0.712,0.072,0.476
0.715,0.178,0.508
0.72,0.259,0.508
0.691,0.064,0.503
0.696,0.182,0.541
0.701,0.263,0.541
0.666,0.053,0.543
0.671,0.164,0.578
0.676,0.26,0.585
0.735,0.081,0.446
0.766,0.07,0.394
0.787,0.07,0.362
0.738,0.174,0.465
0.768,0.179,0.416
0.788,0.164,0.374
0.743,0.255,0.465
0.774,0.247,0.407
0.739,0.526,0.449
0.738,0.462,0.499
0.725,0.341,0.508
0.766,0.504,0.398
0.765,0.439,0.448
0.749,0.337,0.465
0.804,0.466,0.333
0.801,0.421,0.379
0.78,0.311,0.398
0.741,0.654,0.349
0.768,0.632,0.298
0.81,0.592,0.205
0.73,0.812,0.184
0.756,0.788,0.113
0.802,0.747,0.034
0.728,0.868,0.032
0.754,0.844,-0.038
0.795,0.816,-0.131
0.72,0.542,0.485
0.691,0.552,0.537
0.719,0.478,0.536
0.69,0.473,0.582
0.707,0.345,0.541
0.681,0.35,0.591
0.721,0.67,0.386
0.707,0.833,0.244
0.706,0.889,0.092
0.692,0.683,0.463
0.688,0.856,0.317
0.687,0.913,0.177
0.725,0.956,-0.206
0.751,0.932,-0.276
0.788,0.894,-0.319
0.816,0.86,-0.384
0.726,0.964,-0.553
0.742,0.939,-0.567
0.773,0.907,-0.642
0.785,0.881,-0.658
0.725,0.99,-0.805
0.741,0.964,-0.819
0.763,0.933,-0.843
0.775,0.906,-0.86
0.703,0.977,-0.146
0.706,0.997,-0.535
0.705,1.022,-0.787
0.684,1.027,-0.102
0.683,1.033,-0.468
0.681,1.069,-0.767
0.861,0.782,-0.47
0.868,0.76,-0.492
0.869,0.716,-0.513
0.868,0.692,-0.533
0.831,0.766,-0.743
0.84,0.718,-0.764
0.843,0.666,-0.784
0.84,0.615,-0.799
0.82,0.785,-0.936
0.829,0.737,-0.956
0.833,0.667,-0.971
0.831,0.616,-0.985
0.852,0.811,-0.441
0.823,0.807,-0.725
0.812,0.826,-0.918
0.831,0.841,-0.419
0.801,0.846,-0.681
0.792,0.871,-0.882
0.862,0.496,-0.468
0.848,0.566,-0.615
0.82,0.524,-0.846
0.809,0.531,-1.009
0.853,0.45,-0.503
0.839,0.52,-0.65
0.812,0.497,-0.853
0.801,0.504,-1.015
0.84,0.387,-0.546
0.828,0.458,-0.667
0.798,0.448,-0.884
0.789,0.469,-1.023
0.825,0.338,-0.574
0.813,0.409,-0.694
0.788,0.422,-0.89
0.778,0.443,-1.028
0.872,0.444,-0.357
0.864,0.398,-0.392
0.847,0.346,-0.478
0.832,0.297,-0.505
0.88,0.268,-0.294
0.873,0.257,-0.363
0.856,0.201,-0.444
0.835,0.165,-0.499
0.883,0.162,-0.232
0.876,0.151,-0.3
0.858,0.138,-0.395
0.837,0.101,-0.451
0.869,0.533,-0.44
0.875,0.577,-0.395
0.879,0.481,-0.329
0.884,0.506,-0.244
0.887,0.277,-0.234
0.883,0.279,-0.194
0.89,0.171,-0.172
0.885,0.15,-0.088
0.855,0.603,-0.587
0.866,0.657,-0.564
0.828,0.549,-0.84
0.839,0.589,-0.806
0.817,0.556,-1.003
0.829,0.59,-0.993
0.402,-0.506,0.304
0.394,-0.511,0.266
0.404,-0.612,0.197
0.393,-0.575,0.194
0.458,-0.503,0.307
0.45,-0.508,0.269
0.452,-0.61,0.202
0.441,-0.573,0.199
0.509,-0.491,0.229
0.505,-0.49,0.213
0.483,-0.534,0.166
0.486,-0.518,0.166
0.552,-0.475,0.197
0.548,-0.474,0.181
0.521,-0.507,0.153
0.525,-0.491,0.153
0.41,-0.501,0.338
0.466,-0.498,0.34
0.519,-0.494,0.27
0.562,-0.478,0.238
0.428,-0.562,0.377
0.472,-0.552,0.368
0.53,-0.549,0.332
0.56,-0.53,0.304
0.332,-0.51,0.301
0.27,-0.5,0.205
0.205,-0.494,0.181
0.324,-0.515,0.263
0.266,-0.495,0.186
0.201,-0.49,0.163
0.344,-0.615,0.19
0.221,-0.498,0.151
0.217,-0.498,0.137
0.332,-0.577,0.187
0.221,-0.504,0.133
0.217,-0.503,0.119
0.34,-0.505,0.335
0.381,-0.573,0.386
0.286,-0.516,0.272
0.304,-0.582,0.334
0.22,-0.511,0.248
0.255,-0.584,0.328
0.141,-0.485,0.21
0.14,-0.48,0.188
0.139,-0.506,0.161
0.144,-0.517,0.141
0.156,-0.48,0.196
0.156,-0.476,0.174
0.156,-0.5,0.147
0.161,-0.511,0.127
0.185,-0.493,0.174
0.181,-0.488,0.156
0.196,-0.502,0.144
0.195,-0.507,0.126
0.142,-0.494,0.261
0.158,-0.49,0.247
0.2,-0.509,0.241
0.141,-0.569,0.315
0.181,-0.568,0.319
0.216,-0.586,0.322
0.127,-0.489,0.223
0.128,-0.509,0.247
0.117,-0.519,0.259
0.126,-0.484,0.2
0.125,-0.506,0.231
0.115,-0.517,0.243
0.12,-0.513,0.177
0.119,-0.526,0.207
0.103,-0.534,0.227
0.125,-0.524,0.157
0.118,-0.536,0.194
0.102,-0.543,0.214
0.128,-0.498,0.274
0.132,-0.57,0.314
0.133,-0.516,0.285
0.133,-0.581,0.338
0.123,-0.526,0.297
0.124,-0.584,0.341
0.152,-0.534,0.111
0.132,-0.541,0.127
0.117,-0.561,0.162
0.101,-0.568,0.182
0.17,-0.554,0.073
0.151,-0.574,0.091
0.133,-0.587,0.126
0.109,-0.604,0.142
0.182,-0.575,0.047
0.162,-0.595,0.065
0.135,-0.614,0.096
0.111,-0.632,0.112
0.169,-0.528,0.097
0.178,-0.547,0.066
0.189,-0.567,0.04
0.194,-0.516,0.097
0.188,-0.531,0.066
0.191,-0.542,0.039
0.216,-0.512,0.091
0.197,-0.53,0.057
0.199,-0.541,0.03
0.587,-0.477,0.097
0.568,-0.465,0.115
0.567,-0.456,0.133
0.569,-0.49,0.083
0.566,-0.469,0.111
0.565,-0.46,0.128
0.525,-0.518,-0.028
0.497,-0.572,0.039
0.525,-0.542,0.084
0.462,-0.537,-0.068
0.461,-0.591,0.024
0.488,-0.561,0.069
0.6,-0.468,0.107
0.685,-0.389,0.087
0.719,-0.347,0.081
0.626,-0.46,0.109
0.703,-0.351,0.111
0.716,-0.338,0.108
0.624,-0.452,0.127
0.703,-0.35,0.114
0.716,-0.336,0.111
0.791,0.004,0.326
0.783,-0.065,0.306
0.772,-0.137,0.26
0.76,-0.2,0.225
0.805,-0.006,0.302
0.797,-0.061,0.3
0.795,-0.143,0.247
0.794,-0.189,0.216
0.816,-0.007,0.284
0.808,-0.061,0.282
0.813,-0.128,0.238
0.812,-0.173,0.207
0.799,0.07,0.344
0.813,0.046,0.304
0.824,0.045,0.286
0.811,0.146,0.328
0.839,0.129,0.296
0.853,0.107,0.249
0.72,-0.32,0.143
0.719,-0.326,0.134
0.716,-0.335,0.114
0.752,-0.288,0.133
0.75,-0.294,0.125
0.728,-0.323,0.111
0.728,-0.324,0.108
0.806,-0.218,0.104
0.806,-0.218,0.097
0.791,-0.247,0.098
0.791,-0.246,0.09
0.814,-0.206,0.1
0.814,-0.206,0.092
0.814,-0.204,0.082
0.815,-0.204,0.074
0.722,-0.312,0.154
0.753,-0.279,0.145
0.805,-0.219,0.119
0.813,-0.207,0.114
0.724,-0.298,0.168
0.746,-0.271,0.157
0.787,-0.236,0.142
0.812,-0.211,0.134
0.749,-0.261,0.167
0.786,-0.231,0.156
0.812,-0.207,0.148
0.684,-0.357,0.153
0.627,-0.401,0.154
0.589,-0.436,0.166
0.683,-0.363,0.145
0.621,-0.404,0.142
0.583,-0.439,0.154
0.703,-0.348,0.117
0.623,-0.446,0.139
0.566,-0.45,0.145
0.686,-0.349,0.165
0.69,-0.339,0.183
0.638,-0.396,0.173
0.645,-0.409,0.19
0.601,-0.431,0.185
0.617,-0.447,0.219
0.728,-0.326,0.105
0.752,-0.305,0.075
0.753,-0.305,0.072
0.792,-0.245,0.08
0.792,-0.248,0.066
0.792,-0.246,0.055
0.815,-0.203,0.064
0.816,-0.202,0.049
0.817,-0.2,0.039
0.757,-0.307,0.057
0.782,-0.312,-0.013
0.791,-0.309,-0.058
0.795,-0.24,0.027
0.804,-0.24,-0.031
0.807,-0.234,-0.059
0.819,-0.194,0.011
0.822,-0.186,-0.027
0.825,-0.179,-0.055
0.8,-0.306,-0.108
0.8,-0.298,-0.152
0.799,-0.276,-0.199
0.808,-0.231,-0.069
0.812,-0.226,-0.118
0.813,-0.22,-0.127
0.826,-0.177,-0.065
0.831,-0.174,-0.08
0.831,-0.169,-0.09
0.799,-0.252,-0.249
0.791,-0.197,-0.32
0.79,-0.158,-0.36
0.817,-0.184,-0.188
0.817,-0.127,-0.269
0.818,-0.085,-0.326
0.835,-0.133,-0.15
0.838,-0.083,-0.23
0.839,-0.041,-0.287
0.789,0.006,-0.515
0.821,0.074,-0.494
0.79,0.071,-0.555
0.819,0.137,-0.542
0.789,0.192,-0.556
0.819,0.256,-0.528
0.787,0.24,-0.615
0.812,0.297,-0.597
0.789,-0.054,-0.478
0.823,0.022,-0.454
0.839,0.049,-0.412
0.789,-0.109,-0.412
0.819,-0.046,-0.378
0.84,-0.002,-0.339
0.782,0.324,-0.719
0.8,0.368,-0.717
0.761,0.359,-0.913
0.779,0.4,-0.894
0.757,0.394,-1.041
0.77,0.422,-1.033
0.576,-0.448,0.17
0.567,-0.47,0.186
0.569,-0.451,0.159
0.563,-0.469,0.17
0.567,-0.449,0.147
0.535,-0.498,0.148
0.568,-0.455,0.135
0.539,-0.482,0.148
0.587,-0.444,0.19
0.596,-0.476,0.24
0.576,-0.472,0.228
0.587,-0.513,0.279
0.427,-0.548,-0.09
0.425,-0.61,0.01
0.453,-0.58,0.055
0.362,-0.562,-0.075
0.366,-0.61,0.017
0.391,-0.581,0.039
0.325,-0.564,-0.053
0.326,-0.601,0.025
0.35,-0.572,0.046
0.173,-0.345,0.124
0.186,-0.392,0.122
0.204,-0.456,0.121
0.178,-0.366,0.011
0.191,-0.414,0.01
0.201,-0.455,0.109
0.214,-0.503,0.108
0.3,-0.464,-0.02
0.289,-0.492,0.004
0.324,-0.536,0.074
0.302,-0.562,0.055
0.387,-0.477,-0.034
0.376,-0.506,-0.01
0.372,-0.546,0.065
0.169,-0.325,0.228
0.27,-0.414,0.278
0.374,-0.414,0.297
0.182,-0.372,0.226
0.276,-0.462,0.247
0.38,-0.462,0.266
0.208,-0.456,0.134
0.315,-0.522,0.182
0.376,-0.52,0.189
0.465,-0.414,0.313
0.524,-0.361,0.238
0.58,-0.345,0.197
0.471,-0.462,0.282
0.508,-0.405,0.225
0.563,-0.389,0.184
0.424,-0.518,0.194
0.497,-0.47,0.166
0.535,-0.444,0.153
0.606,-0.345,0.123
0.598,-0.369,0.125
0.531,-0.459,0.137
0.607,-0.337,0.138
0.599,-0.362,0.141
0.532,-0.457,0.138
0.598,-0.339,0.183
0.582,-0.383,0.17
0.549,-0.434,0.148
0.605,-0.353,0.103
0.591,-0.394,0.073
0.57,-0.407,0.057
0.596,-0.378,0.105
0.567,-0.44,0.076
0.545,-0.453,0.061
0.529,-0.463,0.132
0.549,-0.516,0.122
0.512,-0.535,0.108
0.522,-0.436,0.024
0.45,-0.487,-0.045
0.498,-0.482,0.027
0.439,-0.516,-0.021
0.477,-0.554,0.094
0.412,-0.554,0.058
0.212,-0.512,0.08
0.233,-0.527,0.019
0.235,-0.537,-0.008
0.278,-0.591,0.034
0.281,-0.565,-0.025
0.273,-0.564,-0.068
0.093,-0.548,0.225
0.077,-0.555,0.242
0.062,-0.555,0.242
0.092,-0.572,0.193
0.077,-0.578,0.2
0.062,-0.578,0.2
0.097,-0.613,0.15
0.08,-0.628,0.159
0.062,-0.628,0.159
0.099,-0.64,0.12
0.08,-0.659,0.124
0.062,-0.659,0.124
0.094,-0.538,0.238
0.101,-0.53,0.259
0.104,-0.533,0.274
0.077,-0.547,0.257
0.086,-0.558,0.283
0.086,-0.562,0.299
0.062,-0.547,0.257
0.062,-0.558,0.283
0.062,-0.562,0.299
0.109,-0.54,0.313
0.106,-0.592,0.347
0.108,-0.581,0.385
0.086,-0.571,0.329
0.083,-0.611,0.367
0.083,-0.598,0.395
0.062,-0.571,0.329
0.062,-0.611,0.367
0.062,-0.598,0.395
0.109,-0.445,0.53
0.083,-0.457,0.533
0.062,-0.457,0.533
0.101,-0.488,0.489
0.083,-0.5,0.493
0.062,-0.5,0.493
0.11,-0.565,0.442
0.083,-0.574,0.449
0.062,-0.574,0.449
0.119,-0.383,0.589
0.145,-0.302,0.658
0.149,-0.235,0.711
0.083,-0.396,0.588
0.1,-0.315,0.662
0.1,-0.25,0.713
0.062,-0.396,0.588
0.062,-0.315,0.662
0.062,-0.25,0.713
0.155,-0.136,0.789
0.13,-0.009,0.909
0.133,0.103,0.968
0.1,-0.148,0.793
0.092,-0.017,0.909
0.092,0.098,0.969
0.062,-0.148,0.793
0.062,-0.017,0.909
0.062,0.098,0.969
0.164,0.569,1.137
0.106,0.578,1.146
0.062,0.578,1.146
0.161,0.427,1.165
0.106,0.427,1.173
0.062,0.427,1.173
0.136,0.232,1.035
0.092,0.234,1.04
0.062,0.234,1.04
0.168,0.747,1.102
0.153,0.994,0.951
0.148,1.069,0.786
0.106,0.762,1.112
0.1,1.013,0.959
0.1,1.088,0.789
0.062,0.762,1.112
0.062,1.013,0.959
0.062,1.088,0.789
0.135,1.249,0.389
0.13,1.299,-0.212
0.122,1.368,-0.642
0.1,1.264,0.388
0.088,1.3,-0.21
0.088,1.364,-0.643
0.062,1.264,0.388
0.062,1.3,-0.21
0.062,1.364,-0.643
0.288,-0.571,-0.129
0.284,-0.571,-0.115
0.27,-0.563,-0.083
0.25,-0.567,-0.101
0.25,-0.552,-0.068
0.238,-0.55,-0.041
0.232,-0.581,-0.078
0.232,-0.566,-0.046
0.202,-0.554,-0.004
0.294,-0.571,-0.153
0.267,-0.569,-0.134
0.249,-0.583,-0.111
0.318,-0.571,-0.191
0.292,-0.575,-0.164
0.273,-0.582,-0.154
0.328,-0.569,-0.213
0.305,-0.581,-0.197
0.286,-0.588,-0.187
0.113,-0.655,0.085
0.1,-0.664,0.093
0.08,-0.683,0.098
0.062,-0.683,0.098
0.113,-0.694,0.056
0.099,-0.702,0.062
0.08,-0.72,0.069
0.062,-0.72,0.069
0.115,-0.714,0.027
0.1,-0.722,0.032
0.08,-0.738,0.039
0.062,-0.738,0.039
0.137,-0.638,0.07
0.145,-0.675,0.044
0.146,-0.695,0.015
0.172,-0.613,0.043
0.188,-0.654,0.027
0.193,-0.671,0.002
0.192,-0.593,0.025
0.21,-0.629,0.007
0.215,-0.646,-0.018
0.116,-0.725,0.01
0.101,-0.733,0.016
0.08,-0.75,0.019
0.062,-0.75,0.019
0.119,-0.747,-0.024
0.104,-0.757,-0.02
0.081,-0.767,-0.016
0.062,-0.767,-0.016
0.119,-0.732,-0.037
0.104,-0.742,-0.034
0.081,-0.758,-0.038
0.062,-0.758,-0.038
0.147,-0.706,-0.001
0.147,-0.727,-0.031
0.147,-0.712,-0.044
0.195,-0.681,-0.012
0.196,-0.689,-0.044
0.185,-0.685,-0.057
0.217,-0.656,-0.032
0.213,-0.663,-0.06
0.202,-0.659,-0.073
0.19,-0.656,-0.088
0.173,-0.682,-0.072
0.147,-0.695,-0.06
0.119,-0.714,-0.053
0.166,-0.631,-0.083
0.15,-0.64,-0.067
0.139,-0.656,-0.013
0.119,-0.67,-0.011
0.151,-0.626,-0.095
0.135,-0.635,-0.079
0.136,-0.64,-0.03
0.116,-0.654,-0.028
0.212,-0.644,-0.082
0.2,-0.641,-0.097
0.197,-0.616,-0.096
0.182,-0.625,-0.099
0.177,-0.625,-0.095
0.201,-0.606,-0.113
0.186,-0.614,-0.116
0.162,-0.62,-0.106
0.223,-0.648,-0.069
0.211,-0.609,-0.094
0.215,-0.599,-0.11
0.233,-0.639,-0.046
0.217,-0.6,-0.074
0.233,-0.588,-0.093
0.23,-0.629,-0.032
0.217,-0.593,-0.059
0.104,-0.725,-0.05
0.081,-0.748,-0.062
0.062,-0.748,-0.062
0.098,-0.684,-0.009
0.088,-0.705,-0.069
0.062,-0.705,-0.069
0.095,-0.668,-0.026
0.088,-0.686,-0.087
0.062,-0.686,-0.087
0.112,-0.634,-0.048
0.091,-0.649,-0.046
0.088,-0.678,-0.094
0.062,-0.678,-0.094
0.105,-0.643,-0.095
0.093,-0.649,-0.093
0.076,-0.668,-0.11
0.063,-0.668,-0.11
0.104,-0.652,-0.122
0.092,-0.658,-0.12
0.076,-0.671,-0.12
0.062,-0.671,-0.12
0.132,-0.621,-0.05
0.115,-0.638,-0.096
0.114,-0.647,-0.122
0.127,-0.632,-0.086
0.125,-0.633,-0.114
0.125,-0.638,-0.124
0.143,-0.623,-0.102
0.134,-0.63,-0.117
0.135,-0.635,-0.127
0.154,-0.617,-0.113
0.16,-0.62,-0.125
0.161,-0.625,-0.135
0.189,-0.607,-0.128
0.192,-0.611,-0.134
0.196,-0.617,-0.146
0.204,-0.599,-0.125
0.218,-0.603,-0.144
0.222,-0.609,-0.157
0.217,-0.591,-0.123
0.238,-0.597,-0.152
0.242,-0.603,-0.165
0.249,-0.59,-0.126
0.254,-0.589,-0.144
0.267,-0.595,-0.178
0.35,-0.566,-0.256
0.326,-0.59,-0.247
0.307,-0.597,-0.237
0.427,-0.555,-0.305
0.364,-0.594,-0.306
0.341,-0.617,-0.301
0.43,-0.546,-0.353
0.369,-0.607,-0.358
0.346,-0.631,-0.354
0.238,-0.633,-0.211
0.213,-0.641,-0.201
0.161,-0.655,-0.192
0.135,-0.664,-0.185
0.267,-0.682,-0.277
0.236,-0.7,-0.271
0.191,-0.706,-0.262
0.156,-0.714,-0.256
0.267,-0.705,-0.335
0.236,-0.722,-0.329
0.189,-0.731,-0.322
0.154,-0.739,-0.316
0.258,-0.627,-0.219
0.295,-0.666,-0.283
0.295,-0.689,-0.34
0.288,-0.604,-0.227
0.318,-0.64,-0.297
0.323,-0.654,-0.35
0.126,-0.668,-0.182
0.112,-0.666,-0.182
0.102,-0.671,-0.181
0.142,-0.718,-0.253
0.124,-0.724,-0.249
0.109,-0.725,-0.248
0.14,-0.743,-0.313
0.121,-0.745,-0.308
0.106,-0.746,-0.306
0.09,-0.677,-0.18
0.074,-0.686,-0.178
0.061,-0.686,-0.178
0.094,-0.726,-0.246
0.072,-0.73,-0.247
0.057,-0.73,-0.247
0.091,-0.747,-0.305
0.071,-0.746,-0.305
0.056,-0.746,-0.305
0.193,-0.555,0.005
0.199,-0.586,0.018
0.217,-0.578,-0.027
0.226,-0.611,-0.007
0.754,-0.239,0.187
0.786,-0.225,0.175
0.812,-0.201,0.168
0.756,-0.226,0.21
0.793,-0.206,0.205
0.812,-0.19,0.196
1.384,0.215,0.378
1.433,0.237,0.406
1.488,0.261,0.466
1.54,0.28,0.492
1.387,0.192,0.393
1.436,0.213,0.421
1.491,0.243,0.482
1.544,0.261,0.508
1.418,0.154,0.378
1.462,0.179,0.407
1.52,0.199,0.466
1.566,0.221,0.493
1.432,0.139,0.361
1.477,0.163,0.389
1.528,0.185,0.447
1.574,0.207,0.474
1.348,0.2,0.357
1.351,0.176,0.372
1.382,0.135,0.356
1.397,0.119,0.338
1.297,0.176,0.333
1.309,0.152,0.353
1.337,0.104,0.335
1.352,0.083,0.314
1.257,0.157,0.33
1.269,0.133,0.35
1.296,0.084,0.327
1.311,0.063,0.306
1.378,0.261,0.348
1.342,0.246,0.327
1.273,0.225,0.293
1.233,0.206,0.29
1.386,0.315,0.284
1.33,0.299,0.249
1.263,0.28,0.23
1.204,0.256,0.204
1.409,0.331,0.236
1.352,0.315,0.201
1.271,0.291,0.164
1.212,0.267,0.138
1.428,0.283,0.376
1.429,0.326,0.312
1.452,0.342,0.264
1.48,0.298,0.434
1.486,0.341,0.36
1.501,0.352,0.314
1.532,0.317,0.46
1.527,0.345,0.392
1.542,0.356,0.346
1.428,0.345,0.195
1.477,0.331,0.139
1.509,0.304,0.113
1.472,0.357,0.222
1.513,0.344,0.182
1.545,0.318,0.156
1.51,0.359,0.285
1.558,0.352,0.238
1.577,0.331,0.225
1.551,0.363,0.317
1.591,0.353,0.285
1.61,0.332,0.272
1.372,0.33,0.159
1.28,0.302,0.092
1.221,0.278,0.066
1.422,0.31,0.074
1.336,0.29,0.02
1.272,0.26,-0.031
1.453,0.284,0.048
1.372,0.253,-0.032
1.308,0.222,-0.083
1.527,0.289,0.098
1.471,0.269,0.034
1.389,0.236,-0.055
1.325,0.205,-0.106
1.543,0.239,0.106
1.491,0.215,0.043
1.412,0.177,-0.04
1.349,0.146,-0.087
1.542,0.219,0.126
1.491,0.195,0.062
1.417,0.154,-0.016
1.354,0.122,-0.064
1.563,0.303,0.141
1.576,0.255,0.148
1.576,0.235,0.167
1.593,0.314,0.215
1.608,0.263,0.219
1.607,0.245,0.236
1.626,0.316,0.262
1.636,0.272,0.266
1.635,0.255,0.283
1.542,0.2,0.144
1.491,0.176,0.08
1.422,0.134,0.004
1.358,0.103,-0.044
1.523,0.188,0.199
1.474,0.166,0.139
1.404,0.121,0.065
1.346,0.091,0.018
1.507,0.208,0.205
1.459,0.186,0.146
1.39,0.144,0.074
1.332,0.113,0.028
1.576,0.216,0.186
1.557,0.203,0.241
1.541,0.224,0.248
1.606,0.226,0.254
1.585,0.205,0.313
1.571,0.226,0.32
1.634,0.236,0.301
1.619,0.216,0.357
1.605,0.237,0.364
1.491,0.229,0.212
1.443,0.207,0.153
1.375,0.168,0.085
1.317,0.138,0.039
1.52,0.254,0.135
1.468,0.234,0.075
1.388,0.203,-0.008
1.328,0.174,-0.054
1.502,0.272,0.143
1.45,0.252,0.083
1.373,0.219,0.012
1.312,0.19,-0.034
1.526,0.244,0.254
1.555,0.268,0.176
1.537,0.287,0.184
1.556,0.249,0.327
1.584,0.274,0.257
1.57,0.298,0.253
1.59,0.26,0.371
1.616,0.28,0.303
1.602,0.304,0.299
1.477,0.298,0.154
1.443,0.298,0.208
1.428,0.289,0.241
1.513,0.312,0.195
1.484,0.313,0.234
1.469,0.303,0.266
1.559,0.317,0.25
1.52,0.328,0.296
1.513,0.323,0.317
1.591,0.323,0.296
1.559,0.333,0.329
1.552,0.329,0.35
1.425,0.277,0.094
1.341,0.252,0.053
1.28,0.224,0.006
1.388,0.279,0.174
1.298,0.25,0.117
1.241,0.223,0.092
1.373,0.269,0.206
1.291,0.242,0.177
1.234,0.215,0.152
1.411,0.278,0.276
1.398,0.247,0.332
1.404,0.211,0.347
1.452,0.293,0.302
1.444,0.268,0.361
1.45,0.231,0.376
1.502,0.316,0.352
1.493,0.288,0.419
1.501,0.254,0.433
1.54,0.321,0.385
1.541,0.305,0.447
1.55,0.271,0.46
1.356,0.258,0.241
1.285,0.236,0.223
1.228,0.209,0.198
1.361,0.231,0.308
1.294,0.203,0.274
1.252,0.183,0.266
1.366,0.194,0.324
1.315,0.167,0.295
1.273,0.148,0.287
1.408,0.185,0.358
1.44,0.171,0.296
1.45,0.146,0.3
1.454,0.206,0.387
1.481,0.193,0.32
1.491,0.168,0.325
1.508,0.228,0.443
1.531,0.215,0.365
1.541,0.189,0.37
1.556,0.245,0.47
1.571,0.234,0.392
1.581,0.208,0.398
1.37,0.169,0.334
1.329,0.143,0.309
1.287,0.123,0.301
1.402,0.15,0.273
1.352,0.116,0.244
1.309,0.095,0.233
1.413,0.125,0.277
1.365,0.088,0.25
1.322,0.066,0.239
1.457,0.128,0.304
1.443,0.128,0.349
1.498,0.15,0.328
1.487,0.152,0.378
1.549,0.168,0.375
1.535,0.172,0.43
1.589,0.187,0.402
1.582,0.194,0.457
1.419,0.108,0.281
1.373,0.07,0.255
1.33,0.048,0.244
1.408,0.109,0.327
1.361,0.07,0.301
1.32,0.05,0.293
1.582,0.295,0.512
1.648,0.309,0.536
1.695,0.325,0.529
1.586,0.276,0.528
1.651,0.295,0.547
1.697,0.311,0.54
1.601,0.238,0.513
1.659,0.261,0.535
1.699,0.279,0.531
1.609,0.224,0.495
1.661,0.248,0.522
1.7,0.267,0.518
1.575,0.332,0.481
1.568,0.349,0.424
1.583,0.36,0.378
1.644,0.33,0.521
1.633,0.36,0.452
1.635,0.358,0.426
1.69,0.346,0.514
1.68,0.36,0.475
1.682,0.358,0.449
1.592,0.367,0.349
1.618,0.355,0.323
1.637,0.334,0.31
1.637,0.356,0.404
1.671,0.328,0.369
1.676,0.31,0.357
1.684,0.356,0.427
1.701,0.333,0.405
1.705,0.315,0.392
1.652,0.317,0.299
1.656,0.279,0.299
1.655,0.261,0.316
1.678,0.299,0.35
1.684,0.273,0.367
1.682,0.264,0.377
1.707,0.304,0.385
1.704,0.28,0.383
1.702,0.272,0.392
1.654,0.242,0.334
1.641,0.223,0.386
1.627,0.244,0.392
1.68,0.254,0.388
1.666,0.243,0.42
1.662,0.256,0.428
1.7,0.261,0.403
1.697,0.256,0.435
1.694,0.268,0.443
1.612,0.267,0.399
1.641,0.285,0.339
1.627,0.309,0.335
1.659,0.267,0.436
1.667,0.288,0.383
1.662,0.301,0.384
1.691,0.28,0.451
1.697,0.295,0.415
1.692,0.308,0.416
1.616,0.328,0.332
1.594,0.338,0.359
1.587,0.333,0.38
1.657,0.313,0.386
1.631,0.319,0.415
1.631,0.32,0.427
1.687,0.32,0.417
1.672,0.322,0.437
1.672,0.322,0.449
1.576,0.326,0.415
1.58,0.319,0.469
1.588,0.285,0.482
1.63,0.321,0.447
1.637,0.315,0.506
1.646,0.299,0.512
1.672,0.324,0.468
1.681,0.33,0.505
1.69,0.314,0.511
1.594,0.259,0.493
1.606,0.251,0.416
1.616,0.225,0.421
1.655,0.283,0.518
1.658,0.266,0.457
1.662,0.247,0.46
1.699,0.298,0.517
1.698,0.285,0.467
1.702,0.267,0.47
1.624,0.204,0.426
1.617,0.211,0.477
1.665,0.233,0.462
1.662,0.238,0.512
1.705,0.253,0.472
1.702,0.256,0.508
1.724,0.335,0.524
1.761,0.359,0.488
1.763,0.357,0.456
1.727,0.321,0.535
1.787,0.355,0.487
1.789,0.353,0.456
1.747,0.301,0.526
1.829,0.351,0.5
1.83,0.353,0.447
1.748,0.289,0.513
1.846,0.331,0.498
1.847,0.333,0.445
1.72,0.356,0.509
1.682,0.358,0.453
1.763,0.357,0.459
1.719,0.336,0.427
1.765,0.355,0.428
1.723,0.317,0.414
1.736,0.363,0.488
1.702,0.36,0.486
1.739,0.361,0.457
1.704,0.358,0.46
1.705,0.356,0.438
1.741,0.358,0.428
1.672,0.323,0.456
1.706,0.339,0.504
1.715,0.322,0.51
1.703,0.323,0.434
1.75,0.339,0.457
1.75,0.341,0.48
1.707,0.312,0.432
1.75,0.338,0.432
1.75,0.339,0.454
1.693,0.324,0.46
1.693,0.323,0.448
1.726,0.34,0.454
1.726,0.339,0.431
1.693,0.325,0.48
1.726,0.342,0.479
1.723,0.306,0.516
1.726,0.299,0.474
1.73,0.281,0.477
1.766,0.34,0.481
1.786,0.315,0.468
1.802,0.314,0.466
1.766,0.338,0.455
1.782,0.316,0.455
1.798,0.315,0.454
1.749,0.278,0.503
1.856,0.319,0.497
1.857,0.321,0.445
1.733,0.267,0.479
1.818,0.313,0.465
1.814,0.314,0.452
1.723,0.28,0.457
1.795,0.316,0.44
1.727,0.267,0.448
1.81,0.315,0.439
1.725,0.271,0.423
1.857,0.323,0.395
1.728,0.281,0.412
1.847,0.335,0.396
1.72,0.292,0.464
1.713,0.299,0.431
1.778,0.317,0.442
1.767,0.337,0.432
1.73,0.29,0.402
1.83,0.355,0.397
1.726,0.307,0.407
1.791,0.351,0.427
0.916,0.018,0.238
0.939,0.026,0.256
0.968,0.017,0.29
0.993,0.026,0.304
0.918,-0.001,0.264
0.942,0.007,0.283
0.97,-0.001,0.308
0.995,0.008,0.323
0.956,-0.054,0.249
0.983,-0.042,0.262
1.011,-0.034,0.286
1.038,-0.022,0.297
0.964,-0.079,0.229
0.991,-0.067,0.242
1.021,-0.046,0.266
1.048,-0.034,0.277
0.906,0.014,0.23
0.909,-0.004,0.257
0.934,-0.064,0.238
0.941,-0.089,0.218
0.901,-0.007,0.221
0.899,-0.027,0.243
0.906,-0.072,0.219
0.909,-0.098,0.203
0.888,-0.01,0.22
0.886,-0.031,0.243
0.882,-0.082,0.212
0.885,-0.108,0.197
0.911,0.055,0.185
0.901,0.051,0.177
0.905,0.031,0.18
0.893,0.027,0.179
0.933,0.056,0.093
0.922,0.047,0.089
0.913,0.036,0.089
0.901,0.042,0.096
0.944,0.06,0.03
0.933,0.051,0.026
0.913,0.03,0.033
0.902,0.036,0.04
0.934,0.063,0.204
0.963,0.08,0.103
0.974,0.085,0.04
0.963,0.068,0.235
0.995,0.11,0.147
1.004,0.118,0.073
0.988,0.077,0.249
1.023,0.135,0.16
1.032,0.143,0.086
0.953,0.064,-0.017
0.963,0.066,-0.086
0.982,0.045,-0.124
0.983,0.088,-0.007
0.982,0.081,-0.091
1.001,0.06,-0.13
1.013,0.126,-0.003
1.017,0.109,-0.095
1.027,0.065,-0.157
1.041,0.152,0.01
1.039,0.12,-0.099
1.048,0.076,-0.161
0.942,0.055,-0.021
0.914,0.024,-0.02
0.902,0.031,-0.013
0.933,0.043,-0.077
0.93,-0.001,-0.059
0.897,-0.011,-0.077
0.952,0.022,-0.116
0.924,-0.035,-0.099
0.891,-0.046,-0.116
0.993,0.033,-0.147
0.963,0.01,-0.138
0.922,-0.044,-0.109
0.89,-0.054,-0.126
1.028,-0.012,-0.157
0.984,-0.026,-0.144
0.926,-0.047,-0.109
0.885,-0.065,-0.093
1.027,-0.034,-0.14
0.983,-0.048,-0.127
0.925,-0.053,-0.097
0.883,-0.071,-0.081
1.012,0.047,-0.152
1.04,-0.008,-0.161
1.04,-0.03,-0.143
1.031,0.049,-0.18
1.058,-0.001,-0.158
1.054,-0.019,-0.135
1.052,0.06,-0.184
1.07,0.004,-0.157
1.067,-0.013,-0.134
1.027,-0.058,-0.12
0.983,-0.072,-0.107
0.924,-0.057,-0.089
0.882,-0.075,-0.073
0.997,-0.062,-0.07
0.969,-0.071,-0.076
0.916,-0.06,-0.056
0.889,-0.074,-0.056
0.98,-0.047,-0.049
0.951,-0.056,-0.056
0.909,-0.064,-0.052
0.881,-0.079,-0.052
1.039,-0.054,-0.124
1.014,-0.057,-0.066
0.996,-0.042,-0.045
1.05,-0.041,-0.106
1.009,-0.07,-0.042
1.011,-0.041,-0.018
1.062,-0.035,-0.105
1.025,-0.063,-0.042
1.027,-0.034,-0.018
0.968,-0.037,-0.035
0.939,-0.046,-0.042
0.882,-0.079,-0.038
0.854,-0.094,-0.038
0.977,0.011,-0.08
0.918,-0.028,-0.071
0.855,-0.068,-0.069
0.809,-0.117,-0.046
0.964,0.02,-0.066
0.905,-0.019,-0.057
0.828,-0.078,-0.051
0.782,-0.128,-0.029
0.984,-0.032,-0.031
1.006,0.03,-0.084
0.993,0.038,-0.07
1.013,-0.014,0.004
1.035,0.02,-0.091
1.04,0.047,-0.069
1.029,-0.007,0.004
1.066,0.034,-0.094
1.071,0.062,-0.072
0.944,0.033,-0.045
0.918,0.002,-0.007
0.91,-0.002,0.023
0.972,0.051,-0.049
0.962,0.038,0.004
0.954,0.033,0.035
1.048,0.087,-0.037
1.02,0.08,0.024
1.009,0.073,0.073
1.079,0.101,-0.041
1.065,0.111,0.042
1.054,0.103,0.091
0.885,-0.006,-0.036
0.817,-0.083,-0.044
0.771,-0.132,-0.021
0.877,-0.03,-0.018
0.797,-0.078,0.006
0.769,-0.123,0.006
0.87,-0.035,0.012
0.806,-0.074,0.016
0.778,-0.119,0.016
0.904,-0.006,0.048
0.913,0.008,0.088
0.919,-0.007,0.108
0.947,0.029,0.059
0.94,0.028,0.128
0.947,0.013,0.147
0.996,0.063,0.132
0.964,0.041,0.194
0.961,-0.008,0.23
1.041,0.094,0.15
1.001,0.058,0.227
0.999,0.008,0.263
0.863,-0.039,0.037
0.828,-0.064,0.043
0.8,-0.109,0.043
0.898,-0.003,0.067
0.862,-0.058,0.061
0.848,-0.082,0.057
0.904,-0.018,0.087
0.888,-0.061,0.087
0.873,-0.084,0.083
0.927,-0.026,0.132
0.941,-0.083,0.123
0.95,-0.098,0.149
0.955,-0.006,0.172
0.971,-0.064,0.134
0.98,-0.079,0.159
0.96,-0.031,0.247
0.994,-0.034,0.178
1.015,-0.052,0.183
0.998,-0.015,0.28
1.026,-0.019,0.189
1.047,-0.037,0.195
0.912,-0.037,0.111
0.905,-0.062,0.105
0.891,-0.086,0.101
0.927,-0.092,0.119
0.915,-0.116,0.122
0.901,-0.126,0.121
0.937,-0.107,0.144
0.917,-0.121,0.147
0.903,-0.131,0.146
0.958,-0.111,0.17
0.97,-0.099,0.215
0.988,-0.091,0.18
0.996,-0.087,0.228
1.034,-0.068,0.189
1.031,-0.058,0.246
1.065,-0.053,0.2
1.058,-0.045,0.257
0.944,-0.12,0.165
0.919,-0.124,0.164
0.905,-0.134,0.163
0.947,-0.109,0.204
0.911,-0.113,0.194
0.887,-0.122,0.187
0.876,-0.014,0.22
0.873,-0.034,0.242
0.868,-0.088,0.209
0.871,-0.113,0.193
0.858,0.0,0.224
0.85,-0.034,0.245
0.849,-0.084,0.214
0.85,-0.12,0.194
0.848,0.0,0.233
0.84,-0.034,0.254
0.838,-0.094,0.215
0.838,-0.13,0.196
0.88,0.024,0.178
0.869,0.05,0.195
0.859,0.05,0.203
0.887,0.05,0.104
0.888,0.08,0.107
0.884,0.09,0.122
0.888,0.044,0.048
0.892,0.082,0.048
0.888,0.039,-0.004
0.897,0.084,-0.029
0.893,0.093,-0.014
0.875,-0.018,-0.089
0.893,0.074,-0.113
0.886,0.075,-0.139
0.868,-0.053,-0.128
0.882,0.01,-0.157
0.875,0.011,-0.182
0.867,-0.062,-0.138
0.875,-0.03,-0.184
0.868,-0.029,-0.209
0.872,-0.071,-0.088
0.867,-0.063,-0.107
0.861,-0.076,-0.108
0.87,-0.077,-0.076
0.861,-0.097,-0.072
0.854,-0.11,-0.073
0.869,-0.081,-0.068
0.859,-0.102,-0.066
0.852,-0.115,-0.067
0.875,-0.082,-0.055
0.856,-0.105,-0.055
0.849,-0.118,-0.056
0.868,-0.086,-0.052
0.856,-0.104,-0.047
0.848,-0.118,-0.048
0.841,-0.101,-0.038
0.854,-0.104,-0.035
0.847,-0.117,-0.035
0.795,-0.132,-0.039
0.841,-0.094,-0.034
0.848,-0.109,-0.02
0.768,-0.143,-0.022
0.84,-0.092,-0.021
0.847,-0.107,-0.007
0.757,-0.147,-0.014
0.84,-0.09,-0.012
0.847,-0.105,0.002
0.758,-0.141,0.006
0.839,-0.086,0.005
0.846,-0.102,0.016
0.767,-0.137,0.017
0.838,-0.09,0.013
0.845,-0.106,0.024
0.789,-0.126,0.044
0.833,-0.1,0.033
0.84,-0.116,0.044
0.835,-0.102,0.053
0.838,-0.119,0.058
0.827,-0.14,0.061
0.86,-0.104,0.079
0.839,-0.129,0.078
0.828,-0.151,0.081
0.878,-0.106,0.097
0.839,-0.138,0.095
0.829,-0.16,0.098
0.885,-0.137,0.12
0.863,-0.16,0.121
0.848,-0.172,0.121
0.887,-0.142,0.144
0.865,-0.156,0.141
0.85,-0.169,0.141
0.889,-0.146,0.162
0.867,-0.153,0.161
0.852,-0.165,0.161
0.873,-0.128,0.184
0.85,-0.138,0.184
0.839,-0.148,0.186
0.837,-0.153,-0.079
0.844,-0.127,-0.074
0.841,-0.117,-0.139
0.851,-0.093,-0.11
0.851,-0.057,-0.208
0.857,-0.028,-0.244
0.852,-0.015,-0.264
0.864,0.012,-0.217
0.836,-0.159,-0.069
0.843,-0.133,-0.069
0.833,-0.158,-0.064
0.839,-0.135,-0.056
0.831,-0.16,-0.053
0.838,-0.135,-0.048
0.828,-0.167,-0.025
0.837,-0.134,-0.036
0.833,-0.167,0.001
0.859,-0.131,0.001
0.831,-0.172,0.029
0.858,-0.129,0.014
0.817,-0.201,0.098
0.82,-0.167,0.083
0.817,-0.201,0.09
0.819,-0.156,0.064
0.83,-0.176,0.071
0.852,-0.143,0.062
0.83,-0.176,0.064
0.856,-0.133,0.042
0.816,-0.202,0.113
0.821,-0.176,0.101
0.824,-0.2,0.131
0.836,-0.183,0.122
0.824,-0.196,0.145
0.837,-0.179,0.141
0.857,-0.128,0.023
0.858,-0.129,0.034
0.83,-0.174,0.04
0.831,-0.174,0.053
0.827,-0.141,0.197
0.827,-0.159,0.188
0.839,-0.176,0.161
0.823,-0.164,0.202
0.823,-0.181,0.191
0.823,-0.19,0.164
0.826,-0.105,0.217
0.826,-0.033,0.265
0.834,0.001,0.245
0.824,-0.118,0.233
0.817,-0.062,0.266
0.826,-0.008,0.268
0.853,0.024,-0.316
0.86,0.086,-0.356
0.867,0.057,-0.232
0.877,0.11,-0.276
0.877,0.056,-0.197
0.884,0.121,-0.208
0.834,0.044,0.27
0.867,0.087,0.205
0.845,0.051,0.215
0.875,0.116,0.164
0.891,0.13,-0.148
0.886,0.122,-0.065
0.888,0.12,-0.154
0.892,0.116,-0.042
1.076,0.118,0.099
1.109,0.136,0.102
1.132,0.15,0.11
1.086,0.126,0.05
1.12,0.143,0.039
1.142,0.157,0.047
1.118,0.119,-0.046
1.142,0.142,-0.048
1.179,0.162,-0.042
1.11,0.08,-0.077
1.159,0.107,-0.099
1.196,0.127,-0.093
1.063,0.109,0.159
1.035,0.072,0.256
1.033,0.023,0.292
1.1,0.13,0.16
1.077,0.106,0.254
1.091,0.063,0.29
1.123,0.144,0.168
1.118,0.127,0.257
1.133,0.084,0.294
1.031,0.0,0.309
1.063,-0.002,0.202
1.084,-0.02,0.208
1.099,0.038,0.311
1.127,0.023,0.218
1.139,-0.007,0.224
1.14,0.059,0.315
1.167,0.038,0.225
1.179,0.008,0.23
1.103,-0.036,0.213
1.09,-0.031,0.27
1.081,-0.02,0.289
1.147,-0.029,0.228
1.136,-0.017,0.285
1.127,-0.005,0.304
1.187,-0.014,0.234
1.171,-0.004,0.289
1.162,0.008,0.307
1.07,-0.008,0.31
1.031,0.021,0.344
1.029,0.039,0.326
1.114,0.012,0.328
1.081,0.049,0.372
1.075,0.071,0.35
1.149,0.024,0.332
1.12,0.068,0.376
1.114,0.09,0.354
1.024,0.09,0.271
1.045,0.155,0.17
1.053,0.163,0.096
1.06,0.128,0.293
1.075,0.185,0.181
1.082,0.192,0.099
1.099,0.147,0.297
1.099,0.201,0.188
1.107,0.209,0.106
1.062,0.171,0.02
1.073,0.138,-0.106
1.083,0.094,-0.168
1.091,0.201,0.015
1.102,0.166,-0.106
1.131,0.121,-0.171
1.115,0.217,0.022
1.136,0.185,-0.095
1.165,0.14,-0.161
1.087,0.078,-0.191
1.112,0.025,-0.153
1.109,0.007,-0.131
1.142,0.104,-0.196
1.162,0.054,-0.157
1.167,0.033,-0.133
1.175,0.123,-0.185
1.204,0.074,-0.146
1.208,0.053,-0.123
1.104,-0.015,-0.101
1.075,-0.042,-0.043
1.077,-0.013,-0.019
1.17,0.018,-0.117
1.142,-0.004,-0.067
1.139,0.018,-0.058
1.212,0.038,-0.106
1.19,0.02,-0.059
1.186,0.041,-0.051
1.078,0.014,0.004
1.105,0.052,-0.099
1.134,0.044,-0.047
1.167,0.091,-0.122
1.182,0.067,-0.04
1.204,0.111,-0.116
1.163,0.17,0.121
1.202,0.2,0.138
1.174,0.177,0.058
1.209,0.208,0.078
1.217,0.182,-0.036
1.248,0.208,-0.019
1.234,0.147,-0.087
1.28,0.175,-0.059
1.154,0.163,0.178
1.159,0.148,0.261
1.174,0.105,0.298
1.196,0.194,0.184
1.21,0.164,0.257
1.231,0.128,0.278
1.182,0.08,0.318
1.21,0.054,0.232
1.222,0.024,0.237
1.246,0.104,0.293
1.269,0.074,0.222
1.282,0.046,0.229
1.23,0.002,0.241
1.215,0.013,0.294
1.206,0.025,0.312
1.29,0.028,0.233
1.278,0.03,0.285
1.268,0.043,0.297
1.194,0.041,0.337
1.163,0.089,0.38
1.157,0.111,0.358
1.253,0.064,0.318
1.225,0.113,0.346
1.214,0.137,0.326
1.142,0.168,0.301
1.129,0.222,0.197
1.137,0.23,0.115
1.19,0.186,0.286
1.172,0.243,0.191
1.18,0.254,0.124
1.145,0.238,0.031
1.182,0.211,-0.081
1.211,0.166,-0.147
1.189,0.265,0.052
1.231,0.24,-0.063
1.268,0.203,-0.115
1.222,0.149,-0.171
1.249,0.095,-0.135
1.254,0.074,-0.112
1.284,0.186,-0.139
1.31,0.126,-0.116
1.316,0.103,-0.092
1.257,0.059,-0.095
1.238,0.043,-0.052
1.235,0.064,-0.043
1.32,0.084,-0.072
1.307,0.07,-0.013
1.293,0.093,-0.003
1.231,0.091,-0.032
1.242,0.132,-0.11
1.278,0.117,0.007
1.296,0.159,-0.079
0.347,-0.634,-0.368
0.324,-0.657,-0.364
0.295,-0.696,-0.359
0.267,-0.712,-0.354
0.317,-0.637,-0.371
0.292,-0.655,-0.372
0.262,-0.691,-0.371
0.235,-0.707,-0.371
0.304,-0.635,-0.376
0.279,-0.654,-0.378
0.249,-0.683,-0.384
0.222,-0.698,-0.384
0.37,-0.61,-0.372
0.347,-0.615,-0.369
0.334,-0.614,-0.374
0.424,-0.55,-0.352
0.421,-0.552,-0.36
0.415,-0.555,-0.363
0.416,-0.553,-0.362
0.411,-0.561,-0.369
0.405,-0.564,-0.371
0.237,-0.73,-0.347
0.188,-0.738,-0.34
0.154,-0.747,-0.334
0.207,-0.722,-0.371
0.161,-0.738,-0.361
0.131,-0.747,-0.359
0.195,-0.714,-0.384
0.155,-0.729,-0.379
0.125,-0.738,-0.377
0.139,-0.751,-0.332
0.12,-0.752,-0.328
0.105,-0.753,-0.327
0.121,-0.75,-0.359
0.105,-0.749,-0.357
0.094,-0.749,-0.357
0.116,-0.741,-0.376
0.102,-0.739,-0.376
0.092,-0.739,-0.376
0.09,-0.754,-0.325
0.07,-0.751,-0.326
0.055,-0.751,-0.326
0.084,-0.749,-0.357
0.067,-0.749,-0.357
0.056,-0.749,-0.357
0.081,-0.739,-0.375
0.067,-0.739,-0.375
0.056,-0.739,-0.375
0.238,-0.686,-0.488
0.304,-0.654,-0.454
0.398,-0.561,-0.388
0.221,-0.715,-0.474
0.286,-0.679,-0.455
0.39,-0.581,-0.388
0.216,-0.718,-0.455
0.282,-0.682,-0.435
0.384,-0.584,-0.39
0.193,-0.707,-0.51
0.18,-0.737,-0.487
0.176,-0.741,-0.467
0.117,-0.73,-0.524
0.109,-0.755,-0.498
0.109,-0.754,-0.475
0.062,-0.73,-0.524
0.061,-0.755,-0.498
0.061,-0.754,-0.475
0.145,1.386,-0.779
0.148,1.364,-0.963
0.148,1.364,-1.1
0.12,1.389,-0.777
0.122,1.368,-0.962
0.122,1.368,-1.099
0.088,1.383,-0.779
0.088,1.364,-0.963
0.088,1.364,-1.1
0.062,1.383,-0.779
0.062,1.364,-0.963
0.062,1.364,-1.1
0.186,1.38,-0.782
0.241,1.366,-0.787
0.28,1.353,-0.791
0.188,1.358,-0.964
0.242,1.345,-0.97
0.282,1.332,-0.972
0.188,1.358,-1.101
0.242,1.345,-1.104
0.282,1.332,-1.106
0.326,1.338,-0.796
0.386,1.309,-0.797
0.428,1.286,-0.805
0.327,1.317,-0.974
0.386,1.292,-0.978
0.427,1.268,-0.982
0.327,1.317,-1.108
0.386,1.292,-1.11
0.427,1.268,-1.113
0.473,1.261,-0.814
0.528,1.217,-0.825
0.569,1.186,-0.839
0.472,1.243,-0.985
0.527,1.203,-1.0
0.568,1.172,-1.005
0.472,1.243,-1.117
0.527,1.203,-1.124
0.568,1.172,-1.13
0.597,1.166,-0.848
0.633,1.138,-0.857
0.655,1.113,-0.87
0.596,1.152,-1.009
0.633,1.124,-1.022
0.656,1.099,-1.028
0.596,1.152,-1.134
0.633,1.124,-1.14
0.656,1.099,-1.145
0.725,1.001,-0.913
0.725,0.99,-1.059
0.725,0.99,-1.167
0.705,1.033,-0.895
0.705,1.023,-1.051
0.705,1.023,-1.16
0.681,1.083,-0.884
0.681,1.069,-1.034
0.681,1.069,-1.151
0.741,0.975,-0.927
0.758,0.945,-0.941
0.771,0.919,-0.958
0.741,0.965,-1.064
0.763,0.933,-1.084
0.775,0.906,-1.091
0.741,0.965,-1.173
0.763,0.933,-1.183
0.775,0.906,-1.19
0.787,0.883,-0.98
0.808,0.834,-1.003
0.816,0.793,-1.021
0.792,0.871,-1.1
0.813,0.824,-1.129
0.82,0.785,-1.136
0.792,0.871,-1.199
0.813,0.824,-1.215
0.82,0.785,-1.222
0.825,0.746,-1.042
0.829,0.667,-1.048
0.827,0.616,-1.062
0.83,0.735,-1.145
0.833,0.667,-1.159
0.831,0.616,-1.165
0.83,0.735,-1.231
0.833,0.667,-1.236
0.831,0.616,-1.243
0.825,0.59,-1.07
0.812,0.559,-1.075
0.804,0.534,-1.082
0.829,0.59,-1.169
0.817,0.556,-1.177
0.809,0.531,-1.179
0.829,0.59,-1.246
0.817,0.556,-1.25
0.809,0.531,-1.252
0.796,0.507,-1.088
0.784,0.48,-1.091
0.773,0.454,-1.097
0.801,0.504,-1.182
0.789,0.469,-1.189
0.778,0.443,-1.191
0.801,0.504,-1.255
0.789,0.469,-1.258
0.778,0.443,-1.261
0.765,0.432,-1.102
0.754,0.412,-1.106
0.744,0.391,-1.107
0.77,0.422,-1.193
0.757,0.394,-1.199
0.746,0.373,-1.199
0.77,0.422,-1.262
0.757,0.394,-1.266
0.746,0.373,-1.267
0.719,0.344,-1.111
0.685,0.287,-1.11
0.66,0.24,-1.108
0.722,0.326,-1.201
0.687,0.264,-1.202
0.662,0.217,-1.201
0.722,0.326,-1.268
0.687,0.264,-1.268
0.662,0.217,-1.268
0.626,0.177,-1.106
0.589,0.081,-1.102
0.545,0.025,-1.096
0.628,0.153,-1.2
0.593,0.062,-1.194
0.548,0.005,-1.192
0.628,0.153,-1.267
0.593,0.062,-1.263
0.548,0.005,-1.261
0.508,-0.022,-1.092
0.442,-0.056,-1.083
0.395,-0.092,-1.08
0.511,-0.041,-1.19
0.443,-0.085,-1.184
0.396,-0.121,-1.183
0.511,-0.041,-1.259
0.443,-0.085,-1.256
0.396,-0.121,-1.255
0.348,-0.129,-1.077
0.286,-0.182,-1.077
0.233,-0.209,-1.077
0.349,-0.158,-1.181
0.288,-0.213,-1.18
0.234,-0.237,-1.18
0.349,-0.158,-1.253
0.288,-0.213,-1.253
0.234,-0.237,-1.253
0.185,-0.232,-1.076
0.118,-0.253,-1.074
0.065,-0.253,-1.074
0.187,-0.258,-1.179
0.116,-0.27,-1.179
0.065,-0.27,-1.179
0.187,-0.258,-1.252
0.116,-0.267,-1.253
0.065,-0.267,-1.253
-0.063,-0.582,-0.753
-0.062,-0.631,-0.688
-0.113,-0.644,-0.58
-0.123,-0.677,-0.506
-0.126,-0.552,-0.743
-0.126,-0.601,-0.678
-0.181,-0.613,-0.547
-0.191,-0.646,-0.473
-0.206,-0.504,-0.726
-0.211,-0.553,-0.63
-0.305,-0.535,-0.49
-0.301,-0.555,-0.384
-0.264,-0.467,-0.712
-0.269,-0.516,-0.615
-0.32,-0.528,-0.47
-0.316,-0.549,-0.363
-0.064,-0.54,-0.808
-0.127,-0.51,-0.798
-0.203,-0.474,-0.784
-0.261,-0.437,-0.77
-0.117,-0.452,-0.835
-0.169,-0.429,-0.829
-0.244,-0.385,-0.818
-0.288,-0.35,-0.806
-0.126,-0.397,-0.876
-0.179,-0.374,-0.87
-0.246,-0.336,-0.861
-0.29,-0.301,-0.849
-0.028,-0.599,-0.759
0.023,-0.622,-0.765
-0.027,-0.648,-0.694
0.023,-0.666,-0.707
-0.064,-0.666,-0.603
0.004,-0.69,-0.611
-0.074,-0.699,-0.529
0.004,-0.72,-0.545
-0.028,-0.557,-0.814
-0.063,-0.476,-0.842
-0.072,-0.421,-0.882
0.023,-0.584,-0.815
0.002,-0.507,-0.855
0.003,-0.456,-0.891
-0.085,-0.427,0.522
-0.069,-0.473,0.481
-0.049,-0.549,0.426
-0.038,-0.56,0.365
-0.064,-0.43,0.523
-0.048,-0.475,0.483
-0.04,-0.549,0.426
-0.029,-0.56,0.364
-0.037,-0.437,0.527
-0.029,-0.48,0.486
-0.024,-0.553,0.434
-0.022,-0.57,0.376
-0.016,-0.441,0.529
-0.009,-0.484,0.488
-0.016,-0.557,0.436
-0.013,-0.573,0.378
-0.108,-0.363,0.58
-0.087,-0.365,0.582
-0.047,-0.375,0.586
-0.027,-0.379,0.587
-0.16,-0.269,0.642
-0.131,-0.278,0.646
-0.099,-0.286,0.652
-0.069,-0.293,0.655
-0.169,-0.196,0.695
-0.141,-0.206,0.698
-0.103,-0.219,0.705
-0.073,-0.226,0.708
-0.124,-0.422,0.519
-0.147,-0.358,0.578
-0.184,-0.26,0.639
-0.194,-0.187,0.692
-0.176,-0.419,0.516
-0.192,-0.346,0.569
-0.222,-0.251,0.642
-0.226,-0.173,0.69
-0.215,-0.415,0.514
-0.231,-0.342,0.567
-0.246,-0.243,0.637
-0.251,-0.165,0.685
-0.108,-0.468,0.478
-0.166,-0.468,0.481
-0.205,-0.463,0.479
-0.088,-0.548,0.43
-0.148,-0.557,0.443
-0.187,-0.555,0.448
-0.077,-0.559,0.369
-0.127,-0.571,0.387
-0.167,-0.569,0.393
-0.182,-0.096,0.767
-0.158,0.031,0.878
-0.16,0.143,0.931
-0.207,-0.087,0.763
-0.187,0.042,0.866
-0.189,0.153,0.919
-0.232,-0.071,0.753
-0.223,0.063,0.852
-0.223,0.173,0.899
-0.257,-0.063,0.748
-0.25,0.072,0.836
-0.25,0.183,0.882
-0.154,-0.106,0.77
-0.109,-0.12,0.784
-0.079,-0.126,0.786
-0.127,0.019,0.89
-0.081,0.004,0.898
-0.047,-0.003,0.904
-0.128,0.131,0.944
-0.084,0.116,0.957
-0.049,0.109,0.963
-0.179,0.558,1.056
-0.175,0.431,1.085
-0.161,0.26,0.987
-0.154,0.559,1.076
-0.15,0.432,1.106
-0.13,0.249,1.0
-0.123,0.563,1.11
-0.119,0.42,1.138
-0.087,0.245,1.024
-0.092,0.566,1.121
-0.089,0.423,1.149
-0.052,0.238,1.03
-0.184,0.73,1.016
-0.158,0.731,1.037
-0.127,0.741,1.075
-0.096,0.744,1.086
-0.186,0.972,0.884
-0.152,0.976,0.902
-0.112,0.981,0.932
-0.076,0.988,0.941
-0.185,1.044,0.723
-0.151,1.049,0.741
-0.107,1.056,0.766
-0.071,1.063,0.776
-0.202,0.558,1.037
-0.207,0.73,0.997
-0.225,0.966,0.863
-0.224,1.039,0.702
-0.232,0.562,1.012
-0.244,0.73,0.966
-0.274,0.967,0.839
-0.28,1.034,0.679
-0.252,0.563,0.99
-0.264,0.73,0.944
-0.305,0.96,0.808
-0.311,1.028,0.648
-0.198,0.431,1.066
-0.224,0.448,1.044
-0.244,0.449,1.022
-0.19,0.271,0.976
-0.223,0.283,0.945
-0.25,0.292,0.928
-0.183,1.218,0.337
-0.149,1.223,0.355
-0.094,1.236,0.37
-0.058,1.242,0.379
-0.173,1.266,-0.24
-0.133,1.28,-0.236
-0.084,1.289,-0.217
-0.043,1.295,-0.214
-0.169,1.332,-0.658
-0.13,1.345,-0.654
-0.076,1.358,-0.647
-0.035,1.364,-0.644
-0.221,1.213,0.316
-0.218,1.251,-0.245
-0.214,1.317,-0.663
-0.294,1.192,0.308
-0.273,1.237,-0.267
-0.273,1.292,-0.667
-0.325,1.185,0.277
-0.314,1.214,-0.276
-0.315,1.268,-0.675
-0.263,-0.41,0.51
-0.333,-0.413,0.518
-0.374,-0.394,0.502
-0.253,-0.458,0.476
-0.322,-0.461,0.491
-0.364,-0.443,0.476
-0.235,-0.553,0.455
-0.3,-0.531,0.46
-0.347,-0.521,0.45
-0.215,-0.566,0.399
-0.281,-0.557,0.414
-0.327,-0.546,0.405
-0.278,-0.336,0.563
-0.296,-0.226,0.626
-0.301,-0.149,0.674
-0.35,-0.332,0.562
-0.362,-0.221,0.631
-0.373,-0.134,0.665
-0.391,-0.313,0.547
-0.406,-0.202,0.607
-0.417,-0.115,0.641
-0.307,-0.047,0.737
-0.312,0.094,0.798
-0.312,0.205,0.845
-0.387,-0.028,0.706
-0.396,0.121,0.78
-0.401,0.233,0.801
-0.431,-0.009,0.681
-0.451,0.135,0.73
-0.456,0.247,0.751
-0.312,0.565,0.925
-0.393,0.568,0.837
-0.446,0.566,0.764
-0.304,0.451,0.956
-0.394,0.478,0.872
-0.447,0.477,0.8
-0.312,0.314,0.89
-0.405,0.327,0.818
-0.46,0.342,0.768
-0.324,0.732,0.878
-0.355,0.949,0.758
-0.361,1.017,0.598
-0.391,0.713,0.779
-0.412,0.943,0.663
-0.413,1.002,0.518
-0.444,0.712,0.707
-0.447,0.929,0.602
-0.448,0.988,0.458
-0.375,1.174,0.226
-0.359,1.189,-0.285
-0.36,1.243,-0.684
-0.415,1.137,0.188
-0.413,1.161,-0.348
-0.415,1.202,-0.701
-0.45,1.123,0.128
-0.454,1.131,-0.362
-0.456,1.172,-0.715
-0.437,-0.343,-0.66
-0.445,-0.294,-0.715
-0.457,-0.183,-0.731
-0.457,-0.133,-0.784
-0.484,-0.304,-0.644
-0.492,-0.255,-0.699
-0.495,-0.141,-0.714
-0.494,-0.091,-0.767
-0.544,-0.248,-0.624
-0.543,-0.191,-0.67
-0.546,-0.083,-0.69
-0.542,-0.034,-0.743
-0.578,-0.199,-0.604
-0.578,-0.142,-0.65
-0.574,-0.037,-0.669
-0.57,0.013,-0.722
-0.427,-0.406,-0.59
-0.473,-0.367,-0.575
-0.545,-0.31,-0.573
-0.58,-0.261,-0.553
-0.415,-0.454,-0.47
-0.468,-0.419,-0.478
-0.532,-0.375,-0.468
-0.575,-0.327,-0.467
-0.398,-0.493,-0.385
-0.451,-0.458,-0.393
-0.534,-0.424,-0.404
-0.577,-0.377,-0.403
-0.389,-0.385,-0.677
-0.318,-0.432,-0.698
-0.379,-0.448,-0.607
-0.324,-0.482,-0.602
-0.39,-0.47,-0.466
-0.347,-0.513,-0.472
-0.373,-0.51,-0.381
-0.343,-0.534,-0.366
-0.397,-0.336,-0.731
-0.315,-0.403,-0.756
-0.411,-0.236,-0.752
-0.344,-0.305,-0.791
-0.41,-0.185,-0.806
-0.347,-0.256,-0.834
-0.391,-0.51,-0.348
-0.444,-0.475,-0.356
-0.535,-0.452,-0.369
-0.578,-0.404,-0.368
-0.37,-0.521,-0.296
-0.438,-0.499,-0.308
-0.526,-0.481,-0.325
-0.587,-0.441,-0.32
-0.358,-0.531,-0.258
-0.426,-0.509,-0.269
-0.527,-0.495,-0.282
-0.587,-0.455,-0.277
-0.366,-0.527,-0.344
-0.335,-0.532,-0.291
-0.323,-0.541,-0.252
-0.341,-0.543,-0.319
-0.299,-0.549,-0.276
-0.277,-0.552,-0.234
-0.314,-0.558,-0.316
-0.266,-0.559,-0.263
-0.244,-0.563,-0.221
-0.341,-0.539,-0.202
-0.343,-0.538,-0.22
-0.353,-0.535,-0.241
-0.419,-0.521,-0.192
-0.421,-0.521,-0.21
-0.421,-0.513,-0.253
-0.514,-0.518,-0.125
-0.53,-0.514,-0.178
-0.527,-0.513,-0.23
-0.58,-0.481,-0.096
-0.595,-0.477,-0.149
-0.588,-0.473,-0.225
-0.338,-0.539,-0.18
-0.416,-0.521,-0.17
-0.498,-0.521,-0.067
-0.563,-0.485,-0.039
-0.338,-0.53,-0.15
-0.402,-0.511,-0.11
-0.494,-0.497,0.013
-0.512,-0.485,0.027
-0.342,-0.532,-0.128
-0.405,-0.513,-0.088
-0.466,-0.491,0.066
-0.484,-0.479,0.08
-0.297,-0.549,-0.208
-0.294,-0.549,-0.186
-0.303,-0.541,-0.172
-0.306,-0.543,-0.15
-0.242,-0.558,-0.166
-0.238,-0.558,-0.152
-0.238,-0.56,-0.133
-0.241,-0.561,-0.118
-0.203,-0.566,-0.144
-0.198,-0.566,-0.13
-0.202,-0.562,-0.11
-0.205,-0.562,-0.096
-0.299,-0.548,-0.225
-0.25,-0.558,-0.191
-0.211,-0.566,-0.169
-0.317,-0.546,-0.236
-0.265,-0.554,-0.21
-0.232,-0.564,-0.198
-0.422,-0.373,0.485
-0.457,-0.315,0.443
-0.494,-0.29,0.41
-0.412,-0.421,0.458
-0.448,-0.377,0.416
-0.484,-0.352,0.383
-0.391,-0.511,0.442
-0.438,-0.477,0.415
-0.468,-0.458,0.387
-0.371,-0.536,0.396
-0.425,-0.521,0.364
-0.455,-0.503,0.336
-0.439,-0.292,0.529
-0.447,-0.186,0.584
-0.458,-0.098,0.618
-0.468,-0.241,0.475
-0.486,-0.145,0.538
-0.493,-0.07,0.567
-0.505,-0.216,0.443
-0.516,-0.126,0.505
-0.523,-0.05,0.534
-0.471,0.008,0.659
-0.486,0.144,0.699
-0.491,0.256,0.719
-0.501,0.02,0.601
-0.513,0.161,0.651
-0.518,0.256,0.658
-0.531,0.039,0.568
-0.538,0.163,0.61
-0.543,0.258,0.617
-0.485,0.566,0.712
-0.524,0.564,0.633
-0.556,0.557,0.576
-0.486,0.477,0.747
-0.523,0.485,0.677
-0.555,0.478,0.621
-0.495,0.351,0.736
-0.523,0.346,0.664
-0.548,0.348,0.623
-0.483,0.711,0.654
-0.484,0.914,0.539
-0.485,0.973,0.394
-0.525,0.696,0.558
-0.525,0.892,0.445
-0.524,0.949,0.305
-0.558,0.688,0.502
-0.552,0.873,0.377
-0.55,0.93,0.238
-0.487,1.108,0.065
-0.482,1.11,-0.371
-0.484,1.151,-0.725
-0.521,1.063,0.026
-0.523,1.088,-0.441
-0.521,1.124,-0.74
-0.548,1.044,-0.042
-0.545,1.062,-0.454
-0.543,1.099,-0.753
-0.135,-0.347,-0.913
-0.187,-0.324,-0.907
-0.249,-0.281,-0.91
-0.293,-0.246,-0.897
-0.122,-0.262,-0.952
-0.176,-0.236,-0.953
-0.237,-0.188,-0.944
-0.285,-0.151,-0.948
-0.121,-0.237,-1.009
-0.175,-0.212,-1.01
-0.236,-0.158,-1.011
-0.284,-0.121,-1.014
-0.08,-0.371,-0.919
-0.07,-0.286,-0.951
-0.069,-0.262,-1.009
0.003,-0.406,-0.928
0.007,-0.316,-0.954
0.007,-0.287,-1.009
-0.457,-0.075,-0.846
-0.439,-0.018,-0.949
-0.435,0.005,-1.03
-0.494,-0.033,-0.829
-0.484,0.038,-0.954
-0.48,0.062,-1.035
-0.537,0.045,-0.827
-0.519,0.113,-0.936
-0.515,0.153,-1.043
-0.565,0.092,-0.806
-0.554,0.177,-0.938
-0.549,0.217,-1.046
-0.41,-0.127,-0.867
-0.35,-0.201,-0.882
-0.402,-0.065,-0.944
-0.332,-0.115,-0.951
-0.399,-0.041,-1.026
-0.331,-0.085,-1.018
-0.759,0.255,0.1
-0.759,0.22,0.133
-0.758,0.188,0.192
-0.75,0.15,0.219
-0.748,0.304,0.159
-0.747,0.27,0.192
-0.743,0.21,0.24
-0.735,0.171,0.267
-0.731,0.384,0.228
-0.729,0.339,0.274
-0.711,0.278,0.311
-0.705,0.214,0.319
-0.709,0.43,0.287
-0.706,0.385,0.333
-0.688,0.295,0.357
-0.682,0.231,0.365
-0.761,0.383,-0.022
-0.749,0.433,0.037
-0.738,0.509,0.1
-0.715,0.556,0.159
-0.765,0.613,-0.139
-0.756,0.642,-0.11
-0.732,0.694,-0.066
-0.717,0.713,-0.031
-0.757,0.694,-0.297
-0.748,0.723,-0.268
-0.726,0.763,-0.231
-0.711,0.782,-0.196
-0.766,0.225,0.064
-0.769,0.179,0.016
-0.771,0.166,-0.029
-0.766,0.19,0.097
-0.77,0.151,0.039
-0.771,0.138,-0.006
-0.771,0.168,0.148
-0.765,0.14,0.078
-0.774,0.114,0.036
-0.763,0.129,0.175
-0.766,0.118,0.105
-0.775,0.091,0.064
-0.768,0.354,-0.058
-0.772,0.591,-0.161
-0.764,0.672,-0.319
-0.766,0.308,-0.09
-0.774,0.565,-0.192
-0.765,0.637,-0.344
-0.768,0.296,-0.135
-0.773,0.542,-0.213
-0.764,0.613,-0.365
-0.602,0.067,-0.698
-0.597,0.145,-0.781
-0.579,0.224,-0.94
-0.575,0.264,-1.047
-0.632,0.143,-0.66
-0.627,0.227,-0.763
-0.614,0.291,-0.918
-0.609,0.326,-1.046
-0.656,0.199,-0.634
-0.652,0.283,-0.738
-0.638,0.338,-0.914
-0.633,0.373,-1.043
-0.606,0.017,-0.645
-0.634,0.094,-0.601
-0.659,0.151,-0.575
-0.609,-0.098,-0.632
-0.633,-0.019,-0.602
-0.656,0.028,-0.578
-0.61,-0.154,-0.586
-0.633,-0.084,-0.561
-0.655,-0.037,-0.537
-0.611,-0.217,-0.535
-0.632,-0.144,-0.524
-0.655,-0.097,-0.5
-0.612,-0.287,-0.466
-0.631,-0.211,-0.447
-0.652,-0.164,-0.43
-0.614,-0.336,-0.403
-0.632,-0.26,-0.396
-0.653,-0.213,-0.379
-0.615,-0.364,-0.368
-0.633,-0.299,-0.355
-0.654,-0.251,-0.339
-0.625,-0.415,-0.317
-0.646,-0.34,-0.307
-0.663,-0.304,-0.283
-0.626,-0.43,-0.274
-0.646,-0.364,-0.256
-0.663,-0.328,-0.233
-0.626,-0.447,-0.222
-0.647,-0.386,-0.21
-0.663,-0.35,-0.186
-0.63,-0.457,-0.134
-0.647,-0.398,-0.121
-0.664,-0.359,-0.116
-0.615,-0.461,-0.081
-0.638,-0.401,-0.071
-0.655,-0.362,-0.065
-0.598,-0.465,-0.023
-0.63,-0.404,-0.026
-0.647,-0.365,-0.021
-0.524,-0.476,0.037
-0.578,-0.392,0.069
-0.611,-0.35,0.063
-0.496,-0.47,0.09
-0.574,-0.39,0.084
-0.607,-0.347,0.078
-0.564,-0.263,0.331
-0.554,-0.316,0.304
-0.533,-0.391,0.273
-0.512,-0.432,0.234
-0.545,-0.274,0.355
-0.535,-0.328,0.328
-0.512,-0.419,0.294
-0.491,-0.46,0.255
-0.515,-0.275,0.391
-0.506,-0.337,0.364
-0.496,-0.441,0.361
-0.483,-0.486,0.31
-0.579,-0.188,0.37
-0.559,-0.199,0.394
-0.526,-0.201,0.423
-0.587,-0.086,0.423
-0.566,-0.094,0.45
-0.539,-0.112,0.481
-0.593,-0.005,0.45
-0.572,-0.013,0.478
-0.545,-0.037,0.51
-0.586,-0.25,0.304
-0.606,-0.227,0.263
-0.631,-0.211,0.241
-0.576,-0.304,0.277
-0.602,-0.253,0.248
-0.626,-0.237,0.226
-0.561,-0.352,0.245
-0.584,-0.306,0.213
-0.619,-0.265,0.197
-0.54,-0.393,0.205
-0.58,-0.329,0.193
-0.614,-0.288,0.177
-0.601,-0.175,0.342
-0.609,-0.077,0.392
-0.616,0.004,0.42
-0.618,-0.165,0.299
-0.637,-0.064,0.356
-0.645,0.005,0.376
-0.642,-0.149,0.276
-0.659,-0.065,0.324
-0.667,0.004,0.344
-0.599,0.072,0.476
-0.602,0.178,0.508
-0.607,0.259,0.508
-0.622,0.081,0.446
-0.626,0.174,0.465
-0.631,0.255,0.465
-0.653,0.07,0.394
-0.655,0.179,0.416
-0.662,0.247,0.407
-0.674,0.07,0.362
-0.676,0.164,0.374
-0.579,0.064,0.503
-0.554,0.053,0.543
-0.584,0.182,0.541
-0.558,0.164,0.578
-0.589,0.263,0.541
-0.563,0.26,0.585
-0.626,0.526,0.449
-0.626,0.462,0.499
-0.612,0.341,0.508
-0.607,0.542,0.485
-0.606,0.478,0.536
-0.594,0.345,0.541
-0.578,0.552,0.537
-0.577,0.473,0.582
-0.568,0.35,0.591
-0.628,0.654,0.349
-0.609,0.67,0.386
-0.58,0.683,0.463
-0.617,0.812,0.184
-0.595,0.833,0.244
-0.576,0.856,0.317
-0.615,0.868,0.032
-0.593,0.889,0.092
-0.574,0.913,0.177
-0.653,0.504,0.398
-0.691,0.466,0.333
-0.652,0.439,0.448
-0.688,0.421,0.379
-0.636,0.337,0.465
-0.668,0.311,0.398
-0.655,0.632,0.298
-0.643,0.788,0.113
-0.641,0.844,-0.038
-0.697,0.592,0.205
-0.689,0.747,0.034
-0.683,0.816,-0.131
-0.613,0.956,-0.206
-0.59,0.977,-0.146
-0.571,1.027,-0.102
-0.613,0.964,-0.553
-0.593,0.997,-0.535
-0.57,1.033,-0.468
-0.613,0.99,-0.805
-0.593,1.022,-0.787
-0.569,1.069,-0.767
-0.639,0.932,-0.276
-0.629,0.939,-0.567
-0.629,0.964,-0.819
-0.675,0.894,-0.319
-0.66,0.907,-0.642
-0.65,0.933,-0.843
-0.703,0.86,-0.384
-0.672,0.881,-0.658
-0.663,0.906,-0.86
-0.748,0.782,-0.47
-0.739,0.811,-0.441
-0.718,0.841,-0.419
-0.718,0.766,-0.743
-0.71,0.807,-0.725
-0.689,0.846,-0.681
-0.708,0.785,-0.936
-0.7,0.826,-0.918
-0.679,0.871,-0.882
-0.755,0.76,-0.492
-0.727,0.718,-0.764
-0.717,0.737,-0.956
-0.756,0.716,-0.513
-0.73,0.666,-0.784
-0.72,0.667,-0.971
-0.755,0.692,-0.533
-0.728,0.615,-0.799
-0.718,0.616,-0.985
-0.749,0.496,-0.468
-0.76,0.444,-0.357
-0.768,0.268,-0.294
-0.771,0.162,-0.232
-0.756,0.533,-0.44
-0.767,0.481,-0.329
-0.774,0.277,-0.234
-0.777,0.171,-0.172
-0.763,0.577,-0.395
-0.771,0.506,-0.244
-0.77,0.279,-0.194
-0.773,0.15,-0.088
-0.735,0.566,-0.615
-0.742,0.603,-0.587
-0.753,0.657,-0.564
-0.707,0.524,-0.846
-0.715,0.549,-0.84
-0.727,0.589,-0.806
-0.696,0.531,-1.009
-0.704,0.556,-1.003
-0.717,0.59,-0.993
-0.74,0.45,-0.503
-0.726,0.52,-0.65
-0.699,0.497,-0.853
-0.688,0.504,-1.015
-0.728,0.387,-0.546
-0.715,0.458,-0.667
-0.685,0.448,-0.884
-0.676,0.469,-1.023
-0.712,0.338,-0.574
-0.7,0.409,-0.694
-0.675,0.422,-0.89
-0.665,0.443,-1.028
-0.751,0.398,-0.392
-0.735,0.346,-0.478
-0.719,0.297,-0.505
-0.76,0.257,-0.363
-0.743,0.201,-0.444
-0.722,0.165,-0.499
-0.763,0.151,-0.3
-0.746,0.138,-0.395
-0.725,0.101,-0.451
-0.289,-0.506,0.304
-0.22,-0.51,0.301
-0.158,-0.5,0.205
-0.092,-0.494,0.181
-0.296,-0.501,0.338
-0.226,-0.505,0.335
-0.172,-0.516,0.272
-0.106,-0.511,0.249
-0.315,-0.562,0.377
-0.269,-0.573,0.386
-0.191,-0.582,0.334
-0.143,-0.584,0.328
-0.282,-0.511,0.266
-0.297,-0.612,0.197
-0.286,-0.575,0.194
-0.212,-0.515,0.263
-0.236,-0.614,0.19
-0.225,-0.577,0.187
-0.154,-0.495,0.186
-0.114,-0.498,0.151
-0.114,-0.504,0.133
-0.088,-0.49,0.162
-0.11,-0.498,0.138
-0.11,-0.503,0.119
-0.345,-0.503,0.307
-0.396,-0.491,0.229
-0.439,-0.475,0.197
-0.338,-0.508,0.269
-0.393,-0.49,0.214
-0.436,-0.474,0.182
-0.345,-0.61,0.202
-0.376,-0.533,0.166
-0.414,-0.507,0.153
-0.334,-0.573,0.199
-0.379,-0.518,0.166
-0.417,-0.491,0.153
-0.352,-0.498,0.34
-0.359,-0.552,0.368
-0.405,-0.494,0.271
-0.417,-0.549,0.332
-0.448,-0.478,0.239
-0.447,-0.53,0.304
-0.028,-0.485,0.21
-0.027,-0.48,0.188
-0.027,-0.506,0.161
-0.031,-0.517,0.141
-0.014,-0.489,0.223
-0.013,-0.484,0.2
-0.007,-0.513,0.177
-0.012,-0.524,0.157
-0.016,-0.509,0.247
-0.014,-0.506,0.232
-0.008,-0.527,0.208
-0.007,-0.536,0.195
-0.004,-0.519,0.259
-0.002,-0.517,0.243
0.01,-0.534,0.227
0.01,-0.543,0.214
-0.03,-0.494,0.261
-0.015,-0.498,0.274
-0.022,-0.516,0.286
-0.01,-0.526,0.297
-0.029,-0.569,0.315
-0.019,-0.57,0.314
-0.02,-0.581,0.338
-0.012,-0.584,0.341
-0.044,-0.48,0.196
-0.045,-0.49,0.247
-0.068,-0.568,0.319
-0.072,-0.493,0.174
-0.086,-0.509,0.241
-0.103,-0.586,0.322
-0.043,-0.476,0.174
-0.068,-0.488,0.155
-0.045,-0.5,0.146
-0.087,-0.502,0.144
-0.05,-0.511,0.126
-0.086,-0.507,0.126
-0.039,-0.534,0.111
-0.058,-0.528,0.096
-0.085,-0.516,0.097
-0.109,-0.512,0.091
-0.058,-0.554,0.073
-0.065,-0.547,0.066
-0.076,-0.531,0.067
-0.085,-0.53,0.057
-0.069,-0.575,0.047
-0.077,-0.567,0.04
-0.078,-0.542,0.039
-0.087,-0.541,0.03
-0.019,-0.541,0.127
-0.038,-0.574,0.091
-0.05,-0.595,0.065
-0.006,-0.561,0.163
-0.022,-0.588,0.127
-0.024,-0.615,0.097
0.012,-0.568,0.182
0.004,-0.604,0.142
0.002,-0.632,0.112
-0.475,-0.477,0.096
-0.488,-0.468,0.106
-0.573,-0.389,0.087
-0.606,-0.347,0.081
-0.461,-0.464,0.116
-0.517,-0.46,0.11
-0.59,-0.351,0.111
-0.604,-0.338,0.108
-0.46,-0.456,0.133
-0.516,-0.452,0.127
-0.59,-0.35,0.114
-0.604,-0.336,0.111
-0.457,-0.49,0.082
-0.459,-0.468,0.112
-0.458,-0.46,0.128
-0.414,-0.518,-0.028
-0.39,-0.572,0.039
-0.418,-0.542,0.084
-0.351,-0.537,-0.068
-0.353,-0.592,0.024
-0.381,-0.561,0.069
-0.679,0.004,0.326
-0.692,-0.006,0.302
-0.704,-0.007,0.284
-0.671,-0.065,0.306
-0.684,-0.061,0.3
-0.695,-0.061,0.282
-0.659,-0.137,0.26
-0.682,-0.143,0.247
-0.7,-0.128,0.238
-0.648,-0.2,0.225
-0.681,-0.189,0.216
-0.699,-0.173,0.207
-0.686,0.07,0.344
-0.698,0.146,0.328
-0.7,0.046,0.304
-0.726,0.129,0.296
-0.712,0.045,0.286
-0.741,0.107,0.249
-0.607,-0.32,0.143
-0.571,-0.357,0.153
-0.514,-0.401,0.154
-0.477,-0.436,0.166
-0.606,-0.326,0.134
-0.57,-0.363,0.145
-0.509,-0.404,0.143
-0.471,-0.439,0.155
-0.604,-0.335,0.114
-0.59,-0.348,0.117
-0.515,-0.446,0.138
-0.459,-0.45,0.144
-0.609,-0.312,0.154
-0.573,-0.349,0.165
-0.525,-0.396,0.174
-0.487,-0.431,0.186
-0.611,-0.298,0.168
-0.577,-0.339,0.183
-0.532,-0.409,0.19
-0.504,-0.447,0.219
-0.639,-0.288,0.133
-0.638,-0.294,0.125
-0.615,-0.323,0.111
-0.615,-0.324,0.108
-0.693,-0.218,0.104
-0.693,-0.218,0.097
-0.678,-0.247,0.098
-0.678,-0.246,0.09
-0.701,-0.206,0.1
-0.701,-0.206,0.092
-0.702,-0.204,0.082
-0.702,-0.204,0.074
-0.641,-0.279,0.145
-0.693,-0.219,0.119
-0.701,-0.207,0.114
-0.634,-0.271,0.157
-0.674,-0.236,0.142
-0.7,-0.211,0.134
-0.636,-0.261,0.167
-0.674,-0.231,0.156
-0.699,-0.207,0.148
-0.615,-0.325,0.105
-0.639,-0.305,0.075
-0.64,-0.305,0.072
-0.679,-0.245,0.08
-0.679,-0.248,0.066
-0.68,-0.246,0.055
-0.702,-0.203,0.064
-0.703,-0.202,0.049
-0.704,-0.2,0.039
-0.645,-0.307,0.057
-0.67,-0.312,-0.013
-0.678,-0.309,-0.058
-0.682,-0.24,0.027
-0.691,-0.24,-0.031
-0.694,-0.234,-0.059
-0.706,-0.194,0.011
-0.709,-0.186,-0.027
-0.712,-0.179,-0.055
-0.687,-0.306,-0.108
-0.687,-0.298,-0.152
-0.687,-0.276,-0.199
-0.696,-0.231,-0.069
-0.7,-0.226,-0.118
-0.7,-0.22,-0.127
-0.714,-0.177,-0.065
-0.718,-0.174,-0.08
-0.719,-0.169,-0.09
-0.687,-0.252,-0.249
-0.678,-0.197,-0.32
-0.677,-0.158,-0.36
-0.704,-0.184,-0.188
-0.704,-0.127,-0.269
-0.705,-0.085,-0.326
-0.722,-0.133,-0.15
-0.726,-0.083,-0.23
-0.727,-0.041,-0.287
-0.677,0.006,-0.515
-0.709,0.074,-0.494
-0.677,0.071,-0.555
-0.706,0.137,-0.542
-0.677,0.192,-0.556
-0.707,0.256,-0.528
-0.674,0.24,-0.615
-0.7,0.297,-0.597
-0.676,-0.054,-0.478
-0.711,0.022,-0.454
-0.727,0.049,-0.412
-0.676,-0.109,-0.412
-0.706,-0.046,-0.378
-0.728,-0.002,-0.339
-0.67,0.324,-0.719
-0.687,0.368,-0.717
-0.649,0.359,-0.913
-0.666,0.4,-0.894
-0.644,0.394,-1.041
-0.657,0.422,-1.033
-0.463,-0.448,0.17
-0.454,-0.47,0.186
-0.458,-0.451,0.159
-0.451,-0.469,0.171
-0.459,-0.449,0.146
-0.428,-0.497,0.148
-0.46,-0.455,0.135
-0.431,-0.482,0.148
-0.474,-0.443,0.19
-0.462,-0.472,0.228
-0.483,-0.476,0.24
-0.475,-0.513,0.279
-0.315,-0.548,-0.09
-0.318,-0.61,0.01
-0.346,-0.58,0.055
-0.25,-0.562,-0.075
-0.259,-0.61,0.017
-0.283,-0.581,0.039
-0.214,-0.564,-0.052
-0.218,-0.602,0.025
-0.243,-0.572,0.046
-0.061,-0.345,0.124
-0.075,-0.392,0.122
-0.095,-0.456,0.121
-0.065,-0.366,0.011
-0.08,-0.414,0.01
-0.092,-0.455,0.109
-0.106,-0.503,0.108
-0.188,-0.464,-0.02
-0.178,-0.492,0.004
-0.216,-0.536,0.074
-0.195,-0.562,0.055
-0.275,-0.477,-0.034
-0.265,-0.506,-0.01
-0.264,-0.546,0.065
-0.057,-0.325,0.228
-0.157,-0.414,0.278
-0.261,-0.414,0.297
-0.071,-0.372,0.226
-0.165,-0.462,0.247
-0.269,-0.462,0.266
-0.099,-0.456,0.134
-0.208,-0.522,0.182
-0.269,-0.52,0.189
-0.352,-0.414,0.313
-0.411,-0.361,0.238
-0.467,-0.345,0.197
-0.36,-0.462,0.282
-0.397,-0.405,0.225
-0.452,-0.389,0.184
-0.317,-0.518,0.194
-0.389,-0.47,0.166
-0.428,-0.444,0.153
-0.494,-0.345,0.123
-0.486,-0.369,0.125
-0.425,-0.459,0.137
-0.495,-0.337,0.138
-0.487,-0.362,0.141
-0.425,-0.457,0.138
-0.486,-0.339,0.183
-0.471,-0.383,0.17
-0.441,-0.435,0.148
-0.492,-0.353,0.103
-0.478,-0.394,0.073
-0.457,-0.407,0.057
-0.485,-0.378,0.105
-0.456,-0.44,0.076
-0.434,-0.453,0.061
-0.423,-0.463,0.132
-0.441,-0.516,0.122
-0.405,-0.536,0.107
-0.41,-0.436,0.024
-0.338,-0.487,-0.045
-0.387,-0.482,0.027
-0.328,-0.516,-0.021
-0.37,-0.554,0.093
-0.305,-0.555,0.057
-0.105,-0.512,0.079
-0.17,-0.591,0.034
-0.121,-0.526,0.02
-0.169,-0.565,-0.025
-0.122,-0.537,-0.008
-0.16,-0.564,-0.068
0.022,-0.548,0.227
0.023,-0.573,0.194
0.019,-0.614,0.15
0.017,-0.641,0.121
0.044,-0.555,0.242
0.044,-0.578,0.2
0.042,-0.628,0.159
0.042,-0.659,0.124
0.022,-0.539,0.239
0.044,-0.547,0.257
0.014,-0.53,0.259
0.036,-0.558,0.283
0.012,-0.533,0.275
0.036,-0.562,0.299
0.006,-0.54,0.313
0.036,-0.571,0.329
0.01,-0.593,0.347
0.038,-0.611,0.367
0.009,-0.582,0.385
0.038,-0.598,0.395
0.007,-0.446,0.53
0.015,-0.489,0.489
0.006,-0.565,0.443
0.038,-0.457,0.533
0.038,-0.5,0.493
0.038,-0.574,0.449
-0.003,-0.384,0.589
0.038,-0.396,0.588
-0.029,-0.302,0.658
0.021,-0.315,0.662
-0.033,-0.235,0.711
0.021,-0.25,0.713
-0.039,-0.136,0.789
0.021,-0.148,0.793
-0.014,-0.009,0.91
0.029,-0.017,0.909
-0.017,0.103,0.968
0.029,0.098,0.969
-0.048,0.569,1.138
-0.045,0.427,1.165
-0.02,0.232,1.036
0.015,0.578,1.146
0.015,0.427,1.173
0.029,0.234,1.04
-0.052,0.748,1.103
0.015,0.762,1.112
-0.037,0.995,0.952
0.022,1.013,0.959
-0.032,1.07,0.786
0.022,1.088,0.789
-0.019,1.249,0.39
0.022,1.264,0.388
-0.014,1.299,-0.212
0.033,1.3,-0.21
-0.006,1.368,-0.642
0.033,1.364,-0.643
-0.175,-0.571,-0.129
-0.138,-0.567,-0.101
-0.12,-0.581,-0.078
-0.171,-0.571,-0.115
-0.137,-0.552,-0.068
-0.119,-0.566,-0.046
-0.157,-0.563,-0.083
-0.124,-0.55,-0.041
-0.088,-0.554,-0.004
-0.183,-0.571,-0.154
-0.154,-0.569,-0.134
-0.136,-0.583,-0.111
-0.21,-0.571,-0.189
-0.181,-0.574,-0.164
-0.16,-0.582,-0.154
-0.222,-0.569,-0.212
-0.195,-0.58,-0.197
-0.174,-0.588,-0.187
0.0,-0.655,0.085
-0.025,-0.639,0.071
-0.06,-0.613,0.043
-0.079,-0.593,0.025
0.0,-0.694,0.056
-0.033,-0.676,0.045
-0.076,-0.654,0.027
-0.098,-0.629,0.007
-0.002,-0.714,0.027
-0.034,-0.696,0.016
-0.08,-0.671,0.002
-0.102,-0.646,-0.018
0.015,-0.665,0.094
0.017,-0.704,0.062
0.015,-0.723,0.033
0.042,-0.683,0.098
0.041,-0.72,0.069
0.041,-0.738,0.039
-0.003,-0.725,0.01
-0.035,-0.707,-0.001
-0.082,-0.681,-0.012
-0.104,-0.656,-0.032
-0.006,-0.747,-0.024
-0.035,-0.728,-0.03
-0.084,-0.689,-0.044
-0.1,-0.663,-0.06
-0.006,-0.732,-0.037
-0.035,-0.713,-0.044
-0.073,-0.685,-0.057
-0.09,-0.659,-0.073
0.015,-0.734,0.016
0.012,-0.759,-0.02
0.012,-0.744,-0.033
0.041,-0.75,0.019
0.04,-0.767,-0.016
0.04,-0.758,-0.038
-0.099,-0.644,-0.082
-0.11,-0.648,-0.069
-0.12,-0.639,-0.046
-0.117,-0.629,-0.032
-0.084,-0.616,-0.096
-0.098,-0.609,-0.094
-0.105,-0.6,-0.074
-0.105,-0.593,-0.059
-0.088,-0.606,-0.113
-0.102,-0.599,-0.11
-0.12,-0.588,-0.093
-0.077,-0.656,-0.088
-0.087,-0.641,-0.097
-0.053,-0.631,-0.083
-0.064,-0.625,-0.095
-0.07,-0.625,-0.099
-0.038,-0.626,-0.095
-0.049,-0.62,-0.106
-0.074,-0.614,-0.116
-0.061,-0.682,-0.072
-0.037,-0.64,-0.067
-0.022,-0.635,-0.079
-0.035,-0.696,-0.06
-0.027,-0.657,-0.012
-0.024,-0.641,-0.029
-0.007,-0.714,-0.053
-0.006,-0.67,-0.011
-0.003,-0.654,-0.028
0.011,-0.726,-0.049
0.018,-0.684,-0.009
0.021,-0.668,-0.026
0.04,-0.748,-0.062
0.034,-0.705,-0.069
0.034,-0.686,-0.087
0.0,-0.634,-0.048
-0.021,-0.622,-0.05
-0.014,-0.632,-0.086
-0.03,-0.623,-0.102
0.008,-0.643,-0.095
-0.002,-0.639,-0.095
-0.012,-0.633,-0.114
-0.022,-0.63,-0.117
0.008,-0.652,-0.122
-0.001,-0.647,-0.122
-0.012,-0.638,-0.124
-0.022,-0.635,-0.127
0.025,-0.649,-0.046
0.023,-0.649,-0.093
0.024,-0.658,-0.12
0.034,-0.678,-0.094
0.045,-0.668,-0.11
0.045,-0.671,-0.12
-0.041,-0.617,-0.113
-0.076,-0.607,-0.128
-0.091,-0.599,-0.125
-0.048,-0.62,-0.125
-0.08,-0.611,-0.134
-0.105,-0.603,-0.144
-0.048,-0.625,-0.135
-0.083,-0.617,-0.146
-0.109,-0.609,-0.157
-0.105,-0.591,-0.123
-0.136,-0.59,-0.126
-0.125,-0.597,-0.152
-0.141,-0.588,-0.145
-0.129,-0.603,-0.165
-0.154,-0.594,-0.178
-0.244,-0.566,-0.255
-0.215,-0.589,-0.247
-0.194,-0.597,-0.237
-0.32,-0.556,-0.307
-0.255,-0.593,-0.306
-0.23,-0.617,-0.301
-0.322,-0.546,-0.354
-0.258,-0.607,-0.358
-0.233,-0.631,-0.354
-0.125,-0.633,-0.211
-0.145,-0.627,-0.219
-0.175,-0.603,-0.228
-0.154,-0.682,-0.277
-0.183,-0.666,-0.283
-0.206,-0.64,-0.297
-0.155,-0.705,-0.335
-0.183,-0.689,-0.34
-0.209,-0.653,-0.35
-0.1,-0.641,-0.201
-0.124,-0.7,-0.271
-0.124,-0.722,-0.329
-0.049,-0.655,-0.192
-0.078,-0.706,-0.262
-0.076,-0.731,-0.322
-0.023,-0.664,-0.185
-0.044,-0.714,-0.256
-0.042,-0.739,-0.316
-0.013,-0.668,-0.182
-0.03,-0.718,-0.253
-0.027,-0.743,-0.313
0.001,-0.667,-0.181
-0.011,-0.724,-0.249
-0.008,-0.745,-0.308
0.011,-0.671,-0.181
0.003,-0.725,-0.248
0.006,-0.746,-0.306
0.026,-0.678,-0.179
0.018,-0.726,-0.246
0.021,-0.747,-0.305
0.043,-0.686,-0.178
0.042,-0.73,-0.247
0.041,-0.746,-0.305
-0.08,-0.555,0.005
-0.104,-0.578,-0.027
-0.087,-0.586,0.018
-0.113,-0.611,-0.007
-0.641,-0.239,0.187
-0.673,-0.225,0.175
-0.699,-0.201,0.168
-0.643,-0.226,0.21
-0.681,-0.206,0.205
-0.699,-0.19,0.196
-1.271,0.215,0.378
-1.235,0.2,0.357
-1.184,0.176,0.333
-1.144,0.157,0.33
-1.274,0.192,0.393
-1.238,0.176,0.372
-1.196,0.152,0.353
-1.156,0.133,0.35
-1.305,0.154,0.378
-1.27,0.135,0.356
-1.224,0.104,0.335
-1.183,0.084,0.327
-1.32,0.139,0.361
-1.285,0.119,0.338
-1.239,0.083,0.314
-1.198,0.063,0.306
-1.321,0.237,0.406
-1.324,0.213,0.421
-1.349,0.179,0.407
-1.364,0.163,0.389
-1.375,0.261,0.466
-1.379,0.243,0.482
-1.407,0.199,0.466
-1.415,0.185,0.447
-1.427,0.28,0.492
-1.431,0.261,0.508
-1.453,0.221,0.493
-1.462,0.207,0.474
-1.265,0.261,0.348
-1.315,0.283,0.376
-1.367,0.298,0.434
-1.419,0.317,0.46
-1.273,0.315,0.284
-1.317,0.326,0.312
-1.374,0.341,0.36
-1.415,0.345,0.392
-1.296,0.331,0.236
-1.339,0.342,0.264
-1.388,0.352,0.314
-1.429,0.356,0.346
-1.229,0.246,0.327
-1.217,0.299,0.249
-1.239,0.315,0.201
-1.161,0.225,0.293
-1.151,0.28,0.23
-1.159,0.291,0.164
-1.121,0.206,0.29
-1.091,0.256,0.204
-1.1,0.267,0.138
-1.316,0.345,0.195
-1.365,0.331,0.139
-1.396,0.304,0.113
-1.259,0.33,0.159
-1.309,0.31,0.074
-1.341,0.284,0.048
-1.168,0.302,0.092
-1.223,0.29,0.02
-1.26,0.253,-0.032
-1.109,0.278,0.066
-1.159,0.26,-0.031
-1.195,0.222,-0.083
-1.359,0.357,0.222
-1.397,0.359,0.285
-1.439,0.363,0.317
-1.401,0.344,0.182
-1.446,0.352,0.238
-1.479,0.353,0.285
-1.432,0.318,0.156
-1.465,0.331,0.225
-1.498,0.332,0.272
-1.414,0.289,0.098
-1.45,0.303,0.141
-1.48,0.314,0.215
-1.513,0.316,0.262
-1.43,0.239,0.106
-1.464,0.255,0.148
-1.495,0.263,0.219
-1.523,0.272,0.266
-1.43,0.219,0.126
-1.463,0.235,0.167
-1.494,0.245,0.236
-1.522,0.255,0.283
-1.359,0.269,0.034
-1.378,0.215,0.043
-1.378,0.195,0.062
-1.276,0.236,-0.055
-1.3,0.177,-0.04
-1.305,0.154,-0.016
-1.212,0.205,-0.106
-1.236,0.146,-0.087
-1.241,0.122,-0.064
-1.429,0.2,0.144
-1.463,0.216,0.186
-1.493,0.226,0.254
-1.521,0.236,0.301
-1.41,0.188,0.199
-1.444,0.203,0.241
-1.472,0.205,0.313
-1.506,0.216,0.357
-1.394,0.208,0.205
-1.429,0.224,0.248
-1.458,0.226,0.32
-1.492,0.237,0.364
-1.378,0.176,0.08
-1.361,0.166,0.139
-1.346,0.186,0.146
-1.309,0.134,0.004
-1.291,0.121,0.065
-1.277,0.144,0.074
-1.246,0.103,-0.044
-1.233,0.091,0.018
-1.219,0.113,0.028
-1.379,0.229,0.212
-1.413,0.244,0.254
-1.443,0.249,0.327
-1.477,0.26,0.371
-1.407,0.254,0.135
-1.443,0.268,0.176
-1.471,0.274,0.257
-1.503,0.28,0.303
-1.389,0.272,0.143
-1.425,0.287,0.184
-1.457,0.298,0.253
-1.489,0.304,0.299
-1.33,0.207,0.153
-1.355,0.234,0.075
-1.337,0.252,0.083
-1.262,0.168,0.085
-1.276,0.203,-0.008
-1.26,0.219,0.012
-1.204,0.138,0.039
-1.215,0.174,-0.054
-1.199,0.19,-0.034
-1.364,0.298,0.154
-1.33,0.298,0.208
-1.315,0.289,0.241
-1.312,0.277,0.094
-1.275,0.279,0.174
-1.26,0.269,0.206
-1.228,0.252,0.053
-1.185,0.25,0.117
-1.178,0.242,0.177
-1.167,0.224,0.006
-1.129,0.223,0.092
-1.121,0.215,0.152
-1.4,0.312,0.195
-1.446,0.317,0.25
-1.478,0.323,0.296
-1.372,0.313,0.234
-1.407,0.328,0.296
-1.446,0.333,0.329
-1.356,0.303,0.266
-1.4,0.323,0.317
-1.439,0.329,0.35
-1.298,0.278,0.276
-1.285,0.247,0.332
-1.291,0.211,0.347
-1.243,0.258,0.241
-1.248,0.231,0.308
-1.254,0.194,0.324
-1.172,0.236,0.223
-1.181,0.203,0.274
-1.202,0.167,0.295
-1.116,0.209,0.198
-1.139,0.183,0.266
-1.16,0.148,0.287
-1.34,0.293,0.302
-1.389,0.316,0.352
-1.428,0.321,0.385
-1.332,0.268,0.361
-1.38,0.288,0.419
-1.429,0.305,0.447
-1.337,0.231,0.376
-1.389,0.254,0.433
-1.437,0.271,0.46
-1.295,0.185,0.358
-1.328,0.171,0.296
-1.338,0.146,0.3
-1.258,0.169,0.334
-1.29,0.15,0.273
-1.3,0.125,0.277
-1.217,0.143,0.309
-1.239,0.116,0.244
-1.252,0.088,0.25
-1.175,0.123,0.301
-1.197,0.095,0.233
-1.21,0.066,0.239
-1.341,0.206,0.387
-1.395,0.228,0.443
-1.443,0.245,0.47
-1.368,0.193,0.32
-1.418,0.215,0.365
-1.458,0.234,0.392
-1.378,0.168,0.325
-1.428,0.189,0.37
-1.468,0.208,0.398
-1.344,0.128,0.304
-1.33,0.128,0.349
-1.307,0.108,0.281
-1.295,0.109,0.327
-1.26,0.07,0.255
-1.248,0.07,0.301
-1.218,0.048,0.244
-1.207,0.05,0.293
-1.385,0.15,0.328
-1.436,0.168,0.375
-1.476,0.187,0.402
-1.375,0.152,0.378
-1.423,0.172,0.43
-1.469,0.194,0.457
-1.47,0.295,0.512
-1.474,0.276,0.528
-1.488,0.238,0.513
-1.496,0.224,0.495
-1.536,0.309,0.536
-1.538,0.295,0.547
-1.547,0.261,0.535
-1.548,0.248,0.522
-1.582,0.325,0.529
-1.585,0.311,0.54
-1.586,0.279,0.531
-1.588,0.267,0.518
-1.462,0.332,0.481
-1.531,0.33,0.521
-1.578,0.346,0.514
-1.456,0.349,0.424
-1.521,0.36,0.452
-1.567,0.36,0.475
-1.47,0.36,0.378
-1.523,0.358,0.426
-1.569,0.358,0.449
-1.48,0.367,0.349
-1.524,0.356,0.404
-1.571,0.356,0.427
-1.505,0.355,0.323
-1.559,0.328,0.369
-1.588,0.333,0.405
-1.524,0.334,0.31
-1.563,0.31,0.357
-1.592,0.315,0.392
-1.539,0.317,0.299
-1.565,0.299,0.35
-1.595,0.304,0.385
-1.543,0.279,0.299
-1.571,0.273,0.367
-1.591,0.28,0.383
-1.542,0.261,0.316
-1.569,0.264,0.377
-1.589,0.272,0.392
-1.541,0.242,0.334
-1.567,0.254,0.388
-1.587,0.261,0.403
-1.529,0.223,0.386
-1.553,0.243,0.42
-1.585,0.256,0.435
-1.515,0.244,0.392
-1.55,0.256,0.428
-1.581,0.268,0.443
-1.5,0.267,0.399
-1.546,0.267,0.436
-1.578,0.28,0.451
-1.528,0.285,0.339
-1.554,0.288,0.383
-1.584,0.295,0.415
-1.514,0.309,0.335
-1.549,0.301,0.384
-1.579,0.308,0.416
-1.503,0.328,0.332
-1.544,0.313,0.386
-1.574,0.32,0.417
-1.481,0.338,0.359
-1.518,0.319,0.415
-1.559,0.322,0.437
-1.474,0.333,0.38
-1.518,0.32,0.427
-1.559,0.322,0.449
-1.463,0.326,0.415
-1.518,0.321,0.447
-1.559,0.324,0.468
-1.467,0.319,0.469
-1.525,0.315,0.506
-1.569,0.33,0.505
-1.476,0.285,0.482
-1.533,0.299,0.512
-1.578,0.314,0.511
-1.482,0.259,0.493
-1.542,0.283,0.518
-1.586,0.298,0.517
-1.494,0.251,0.416
-1.545,0.266,0.457
-1.585,0.285,0.467
-1.504,0.225,0.421
-1.549,0.247,0.46
-1.589,0.267,0.47
-1.512,0.204,0.426
-1.552,0.233,0.462
-1.592,0.253,0.472
-1.504,0.211,0.477
-1.549,0.238,0.512
-1.589,0.256,0.508
-1.611,0.335,0.524
-1.614,0.321,0.535
-1.634,0.301,0.526
-1.635,0.289,0.513
-1.648,0.359,0.488
-1.674,0.355,0.487
-1.716,0.351,0.5
-1.733,0.331,0.498
-1.65,0.357,0.456
-1.676,0.353,0.456
-1.717,0.353,0.447
-1.734,0.333,0.445
-1.607,0.356,0.509
-1.65,0.357,0.459
-1.652,0.355,0.428
-1.569,0.358,0.453
-1.606,0.336,0.427
-1.61,0.317,0.414
-1.589,0.36,0.486
-1.624,0.363,0.488
-1.591,0.358,0.46
-1.626,0.361,0.457
-1.593,0.356,0.438
-1.628,0.358,0.428
-1.559,0.323,0.456
-1.59,0.323,0.434
-1.595,0.312,0.432
-1.593,0.339,0.504
-1.637,0.339,0.457
-1.638,0.338,0.432
-1.602,0.322,0.51
-1.637,0.341,0.48
-1.637,0.339,0.454
-1.58,0.324,0.46
-1.581,0.323,0.448
-1.613,0.34,0.454
-1.614,0.339,0.431
-1.58,0.325,0.48
-1.613,0.342,0.479
-1.61,0.306,0.516
-1.653,0.34,0.481
-1.654,0.338,0.455
-1.614,0.299,0.474
-1.673,0.315,0.468
-1.67,0.316,0.455
-1.618,0.281,0.477
-1.689,0.314,0.466
-1.686,0.315,0.454
-1.637,0.278,0.503
-1.621,0.267,0.479
-1.743,0.319,0.497
-1.705,0.313,0.465
-1.744,0.321,0.445
-1.701,0.314,0.452
-1.611,0.28,0.457
-1.614,0.267,0.448
-1.613,0.271,0.423
-1.615,0.281,0.412
-1.682,0.316,0.44
-1.697,0.315,0.439
-1.745,0.323,0.395
-1.735,0.335,0.396
-1.607,0.292,0.464
-1.666,0.317,0.442
-1.6,0.299,0.431
-1.654,0.337,0.432
-1.617,0.29,0.402
-1.613,0.307,0.407
-1.718,0.355,0.397
-1.678,0.351,0.427
-0.803,0.018,0.238
-0.793,0.014,0.23
-0.788,-0.007,0.221
-0.776,-0.01,0.22
-0.806,-0.001,0.264
-0.796,-0.004,0.257
-0.786,-0.027,0.243
-0.773,-0.031,0.243
-0.844,-0.054,0.249
-0.821,-0.064,0.238
-0.793,-0.072,0.219
-0.769,-0.082,0.212
-0.851,-0.079,0.229
-0.829,-0.089,0.218
-0.796,-0.098,0.203
-0.772,-0.108,0.197
-0.826,0.026,0.256
-0.829,0.007,0.283
-0.87,-0.042,0.262
-0.878,-0.067,0.242
-0.855,0.017,0.29
-0.857,-0.001,0.308
-0.898,-0.034,0.286
-0.908,-0.046,0.266
-0.881,0.026,0.304
-0.882,0.008,0.323
-0.926,-0.022,0.297
-0.936,-0.034,0.277
-0.798,0.055,0.185
-0.821,0.063,0.204
-0.85,0.068,0.235
-0.876,0.077,0.249
-0.82,0.056,0.093
-0.85,0.08,0.103
-0.883,0.11,0.147
-0.911,0.135,0.16
-0.832,0.06,0.03
-0.861,0.085,0.04
-0.892,0.118,0.073
-0.919,0.143,0.086
-0.788,0.051,0.177
-0.809,0.047,0.089
-0.821,0.051,0.026
-0.793,0.031,0.18
-0.8,0.036,0.089
-0.801,0.03,0.033
-0.78,0.027,0.179
-0.789,0.042,0.096
-0.789,0.036,0.04
-0.84,0.064,-0.017
-0.851,0.066,-0.086
-0.869,0.045,-0.124
-0.829,0.055,-0.021
-0.821,0.043,-0.077
-0.84,0.022,-0.116
-0.801,0.024,-0.02
-0.817,-0.001,-0.059
-0.811,-0.035,-0.099
-0.79,0.031,-0.013
-0.784,-0.011,-0.077
-0.778,-0.046,-0.116
-0.87,0.088,-0.007
-0.901,0.126,-0.003
-0.928,0.152,0.01
-0.87,0.081,-0.091
-0.905,0.109,-0.095
-0.926,0.12,-0.099
-0.888,0.06,-0.13
-0.914,0.065,-0.157
-0.936,0.076,-0.161
-0.88,0.033,-0.147
-0.899,0.047,-0.152
-0.918,0.049,-0.18
-0.939,0.06,-0.184
-0.915,-0.012,-0.157
-0.928,-0.008,-0.161
-0.945,-0.001,-0.158
-0.957,0.004,-0.157
-0.915,-0.034,-0.14
-0.927,-0.03,-0.143
-0.942,-0.019,-0.135
-0.954,-0.013,-0.134
-0.85,0.01,-0.138
-0.871,-0.026,-0.144
-0.871,-0.048,-0.127
-0.81,-0.044,-0.109
-0.814,-0.047,-0.109
-0.812,-0.053,-0.097
-0.777,-0.054,-0.126
-0.772,-0.065,-0.093
-0.771,-0.071,-0.081
-0.914,-0.058,-0.12
-0.926,-0.054,-0.124
-0.937,-0.041,-0.106
-0.949,-0.035,-0.105
-0.885,-0.062,-0.07
-0.901,-0.057,-0.066
-0.897,-0.07,-0.042
-0.913,-0.063,-0.042
-0.867,-0.047,-0.049
-0.883,-0.042,-0.045
-0.899,-0.041,-0.018
-0.915,-0.034,-0.018
-0.87,-0.072,-0.107
-0.856,-0.071,-0.076
-0.838,-0.056,-0.056
-0.811,-0.057,-0.089
-0.803,-0.06,-0.056
-0.796,-0.064,-0.052
-0.769,-0.075,-0.073
-0.776,-0.074,-0.056
-0.769,-0.079,-0.052
-0.855,-0.037,-0.035
-0.871,-0.032,-0.031
-0.9,-0.014,0.004
-0.916,-0.007,0.004
-0.865,0.011,-0.08
-0.893,0.03,-0.084
-0.922,0.02,-0.091
-0.953,0.034,-0.094
-0.851,0.02,-0.066
-0.88,0.038,-0.07
-0.927,0.047,-0.069
-0.959,0.062,-0.072
-0.826,-0.046,-0.042
-0.806,-0.028,-0.071
-0.792,-0.019,-0.057
-0.769,-0.079,-0.038
-0.742,-0.068,-0.069
-0.715,-0.078,-0.051
-0.742,-0.094,-0.038
-0.696,-0.117,-0.046
-0.669,-0.128,-0.029
-0.831,0.033,-0.045
-0.805,0.002,-0.007
-0.797,-0.002,0.023
-0.772,-0.006,-0.036
-0.765,-0.03,-0.018
-0.757,-0.035,0.012
-0.704,-0.083,-0.044
-0.685,-0.078,0.006
-0.693,-0.074,0.016
-0.658,-0.132,-0.021
-0.657,-0.123,0.006
-0.665,-0.119,0.016
-0.86,0.051,-0.049
-0.935,0.087,-0.037
-0.966,0.101,-0.041
-0.849,0.038,0.004
-0.907,0.08,0.024
-0.952,0.111,0.042
-0.841,0.033,0.035
-0.896,0.073,0.073
-0.941,0.103,0.091
-0.791,-0.006,0.048
-0.8,0.008,0.088
-0.806,-0.007,0.108
-0.751,-0.039,0.037
-0.785,-0.003,0.067
-0.792,-0.018,0.087
-0.715,-0.064,0.043
-0.75,-0.058,0.061
-0.775,-0.061,0.087
-0.687,-0.109,0.043
-0.735,-0.082,0.057
-0.76,-0.084,0.083
-0.835,0.029,0.059
-0.884,0.063,0.132
-0.929,0.094,0.15
-0.828,0.028,0.128
-0.851,0.041,0.194
-0.889,0.058,0.227
-0.834,0.013,0.147
-0.848,-0.008,0.23
-0.886,0.008,0.263
-0.814,-0.026,0.132
-0.828,-0.083,0.123
-0.838,-0.098,0.149
-0.799,-0.037,0.111
-0.814,-0.092,0.119
-0.824,-0.107,0.144
-0.793,-0.062,0.105
-0.802,-0.116,0.122
-0.805,-0.121,0.147
-0.778,-0.086,0.101
-0.788,-0.126,0.121
-0.791,-0.131,0.146
-0.842,-0.006,0.172
-0.847,-0.031,0.247
-0.885,-0.015,0.28
-0.858,-0.064,0.134
-0.881,-0.034,0.178
-0.913,-0.019,0.189
-0.868,-0.079,0.159
-0.903,-0.052,0.183
-0.934,-0.037,0.195
-0.845,-0.111,0.17
-0.857,-0.099,0.215
-0.832,-0.12,0.165
-0.834,-0.109,0.204
-0.807,-0.124,0.164
-0.798,-0.113,0.194
-0.793,-0.134,0.163
-0.774,-0.122,0.187
-0.875,-0.091,0.18
-0.921,-0.068,0.189
-0.953,-0.053,0.2
-0.884,-0.087,0.228
-0.918,-0.058,0.246
-0.945,-0.045,0.257
-0.763,-0.014,0.22
-0.745,0.0,0.224
-0.735,0.0,0.233
-0.761,-0.034,0.242
-0.737,-0.034,0.245
-0.727,-0.034,0.254
-0.755,-0.088,0.209
-0.737,-0.084,0.214
-0.725,-0.094,0.215
-0.758,-0.113,0.193
-0.737,-0.12,0.194
-0.726,-0.13,0.196
-0.768,0.024,0.178
-0.774,0.05,0.104
-0.775,0.044,0.048
-0.756,0.05,0.195
-0.775,0.08,0.107
-0.779,0.082,0.048
-0.746,0.05,0.203
-0.772,0.09,0.122
-0.775,0.039,-0.004
-0.762,-0.018,-0.089
-0.756,-0.053,-0.128
-0.784,0.084,-0.029
-0.781,0.074,-0.113
-0.77,0.01,-0.157
-0.78,0.093,-0.014
-0.773,0.075,-0.139
-0.762,0.011,-0.182
-0.754,-0.062,-0.138
-0.76,-0.071,-0.088
-0.758,-0.077,-0.076
-0.763,-0.03,-0.184
-0.755,-0.063,-0.107
-0.748,-0.097,-0.072
-0.755,-0.029,-0.209
-0.748,-0.076,-0.108
-0.741,-0.11,-0.073
-0.757,-0.081,-0.068
-0.762,-0.082,-0.055
-0.755,-0.086,-0.052
-0.747,-0.102,-0.066
-0.744,-0.105,-0.055
-0.743,-0.104,-0.047
-0.74,-0.115,-0.067
-0.736,-0.118,-0.056
-0.735,-0.118,-0.048
-0.728,-0.101,-0.038
-0.683,-0.132,-0.039
-0.655,-0.143,-0.022
-0.742,-0.104,-0.035
-0.728,-0.094,-0.034
-0.727,-0.092,-0.021
-0.734,-0.117,-0.035
-0.735,-0.109,-0.02
-0.734,-0.107,-0.007
-0.645,-0.147,-0.014
-0.646,-0.141,0.006
-0.654,-0.137,0.017
-0.727,-0.09,-0.012
-0.727,-0.086,0.005
-0.725,-0.09,0.013
-0.734,-0.105,0.002
-0.734,-0.102,0.016
-0.732,-0.106,0.024
-0.676,-0.126,0.044
-0.722,-0.102,0.053
-0.748,-0.104,0.079
-0.721,-0.1,0.033
-0.725,-0.119,0.058
-0.726,-0.129,0.078
-0.728,-0.116,0.044
-0.715,-0.14,0.061
-0.715,-0.151,0.081
-0.765,-0.106,0.097
-0.772,-0.137,0.12
-0.775,-0.142,0.144
-0.727,-0.138,0.095
-0.751,-0.16,0.121
-0.752,-0.156,0.141
-0.716,-0.16,0.098
-0.736,-0.172,0.121
-0.737,-0.169,0.141
-0.776,-0.146,0.162
-0.76,-0.128,0.184
-0.754,-0.153,0.161
-0.737,-0.138,0.184
-0.739,-0.165,0.161
-0.726,-0.148,0.186
-0.724,-0.153,-0.079
-0.728,-0.117,-0.139
-0.738,-0.057,-0.208
-0.739,-0.015,-0.264
-0.732,-0.127,-0.074
-0.739,-0.093,-0.11
-0.744,-0.028,-0.244
-0.751,0.012,-0.217
-0.724,-0.159,-0.069
-0.731,-0.133,-0.069
-0.72,-0.158,-0.064
-0.727,-0.135,-0.056
-0.719,-0.16,-0.053
-0.726,-0.135,-0.048
-0.715,-0.167,-0.025
-0.725,-0.134,-0.036
-0.721,-0.167,0.001
-0.746,-0.131,0.001
-0.718,-0.172,0.029
-0.745,-0.129,0.014
-0.704,-0.201,0.098
-0.704,-0.201,0.09
-0.717,-0.176,0.071
-0.718,-0.176,0.064
-0.708,-0.167,0.083
-0.707,-0.156,0.064
-0.739,-0.143,0.062
-0.743,-0.133,0.042
-0.704,-0.202,0.113
-0.708,-0.176,0.101
-0.711,-0.2,0.131
-0.723,-0.183,0.122
-0.711,-0.196,0.145
-0.725,-0.179,0.141
-0.745,-0.128,0.023
-0.745,-0.129,0.034
-0.718,-0.174,0.04
-0.718,-0.174,0.053
-0.714,-0.141,0.197
-0.71,-0.164,0.202
-0.715,-0.159,0.188
-0.71,-0.181,0.191
-0.727,-0.176,0.161
-0.711,-0.19,0.164
-0.714,-0.105,0.217
-0.714,-0.033,0.265
-0.721,0.001,0.245
-0.711,-0.118,0.233
-0.705,-0.062,0.266
-0.713,-0.008,0.268
-0.74,0.024,-0.316
-0.748,0.086,-0.356
-0.754,0.057,-0.232
-0.764,0.11,-0.276
-0.765,0.056,-0.197
-0.772,0.121,-0.208
-0.721,0.044,0.27
-0.754,0.087,0.205
-0.733,0.051,0.215
-0.762,0.116,0.164
-0.778,0.13,-0.148
-0.773,0.122,-0.065
-0.775,0.12,-0.154
-0.779,0.116,-0.042
-0.963,0.118,0.099
-0.974,0.126,0.05
-1.005,0.119,-0.046
-0.997,0.08,-0.077
-0.997,0.136,0.102
-1.007,0.143,0.039
-1.029,0.142,-0.048
-1.046,0.107,-0.099
-1.019,0.15,0.11
-1.03,0.157,0.047
-1.066,0.162,-0.042
-1.084,0.127,-0.093
-0.95,0.109,0.159
-0.987,0.13,0.16
-1.01,0.144,0.168
-0.922,0.072,0.256
-0.964,0.106,0.254
-1.006,0.127,0.257
-0.92,0.023,0.292
-0.978,0.063,0.29
-1.02,0.084,0.294
-0.919,0.0,0.309
-0.986,0.038,0.311
-1.028,0.059,0.315
-0.95,-0.002,0.202
-1.014,0.023,0.218
-1.054,0.038,0.225
-0.972,-0.02,0.208
-1.026,-0.007,0.224
-1.066,0.008,0.23
-0.99,-0.036,0.213
-1.034,-0.029,0.228
-1.074,-0.014,0.234
-0.978,-0.031,0.27
-1.023,-0.017,0.285
-1.058,-0.004,0.289
-0.968,-0.02,0.289
-1.014,-0.005,0.304
-1.049,0.008,0.307
-0.958,-0.008,0.31
-1.001,0.012,0.328
-1.036,0.024,0.332
-0.918,0.021,0.344
-0.969,0.049,0.372
-1.008,0.068,0.376
-0.917,0.039,0.326
-0.963,0.071,0.35
-1.002,0.09,0.354
-0.912,0.09,0.271
-0.947,0.128,0.293
-0.986,0.147,0.297
-0.932,0.155,0.17
-0.962,0.185,0.181
-0.986,0.201,0.188
-0.941,0.163,0.096
-0.97,0.192,0.099
-0.994,0.209,0.106
-0.95,0.171,0.02
-0.978,0.201,0.015
-1.003,0.217,0.022
-0.961,0.138,-0.106
-0.99,0.166,-0.106
-1.023,0.185,-0.095
-0.971,0.094,-0.168
-1.019,0.121,-0.171
-1.052,0.14,-0.161
-0.974,0.078,-0.191
-1.029,0.104,-0.196
-1.063,0.123,-0.185
-0.999,0.025,-0.153
-1.05,0.054,-0.157
-1.091,0.074,-0.146
-0.996,0.007,-0.131
-1.054,0.033,-0.133
-1.096,0.053,-0.123
-0.992,-0.015,-0.101
-1.057,0.018,-0.117
-1.099,0.038,-0.106
-0.962,-0.042,-0.043
-1.029,-0.004,-0.067
-1.077,0.02,-0.059
-0.964,-0.013,-0.019
-1.026,0.018,-0.058
-1.074,0.041,-0.051
-0.966,0.014,0.004
-1.022,0.044,-0.047
-1.069,0.067,-0.04
-0.992,0.052,-0.099
-1.054,0.091,-0.122
-1.091,0.111,-0.116
-1.051,0.17,0.121
-1.061,0.177,0.058
-1.104,0.182,-0.036
-1.121,0.147,-0.087
-1.089,0.2,0.138
-1.096,0.208,0.078
-1.135,0.208,-0.019
-1.167,0.175,-0.059
-1.041,0.163,0.178
-1.083,0.194,0.184
-1.047,0.148,0.261
-1.098,0.164,0.257
-1.061,0.105,0.298
-1.119,0.128,0.278
-1.069,0.08,0.318
-1.133,0.104,0.293
-1.097,0.054,0.232
-1.156,0.074,0.222
-1.109,0.024,0.237
-1.169,0.046,0.229
-1.117,0.002,0.241
-1.177,0.028,0.233
-1.103,0.013,0.294
-1.165,0.03,0.285
-1.093,0.025,0.312
-1.156,0.043,0.297
-1.081,0.041,0.337
-1.141,0.064,0.318
-1.05,0.089,0.38
-1.113,0.113,0.346
-1.044,0.111,0.358
-1.101,0.137,0.326
-1.029,0.168,0.301
-1.077,0.186,0.286
-1.016,0.222,0.197
-1.059,0.243,0.191
-1.024,0.23,0.115
-1.068,0.254,0.124
-1.032,0.238,0.031
-1.077,0.265,0.052
-1.069,0.211,-0.081
-1.119,0.24,-0.063
-1.098,0.166,-0.147
-1.155,0.203,-0.115
-1.109,0.149,-0.171
-1.172,0.186,-0.139
-1.137,0.095,-0.135
-1.198,0.126,-0.116
-1.141,0.074,-0.112
-1.203,0.103,-0.092
-1.145,0.059,-0.095
-1.207,0.084,-0.072
-1.126,0.043,-0.052
-1.194,0.07,-0.013
-1.122,0.064,-0.043
-1.18,0.093,-0.003
-1.118,0.091,-0.032
-1.165,0.117,0.007
-1.129,0.132,-0.11
-1.183,0.159,-0.079
-0.234,-0.634,-0.369
-0.2,-0.637,-0.37
-0.186,-0.635,-0.376
-0.259,-0.61,-0.374
-0.225,-0.616,-0.368
-0.211,-0.614,-0.375
-0.31,-0.553,-0.353
-0.295,-0.55,-0.361
-0.281,-0.554,-0.363
-0.303,-0.556,-0.362
-0.285,-0.561,-0.371
-0.271,-0.564,-0.373
-0.21,-0.657,-0.365
-0.183,-0.696,-0.359
-0.155,-0.712,-0.353
-0.177,-0.656,-0.371
-0.147,-0.691,-0.37
-0.122,-0.706,-0.37
-0.163,-0.654,-0.378
-0.135,-0.683,-0.383
-0.109,-0.698,-0.383
-0.124,-0.73,-0.347
-0.075,-0.738,-0.34
-0.041,-0.747,-0.334
-0.095,-0.722,-0.37
-0.048,-0.737,-0.361
-0.018,-0.747,-0.359
-0.082,-0.714,-0.383
-0.043,-0.729,-0.378
-0.013,-0.738,-0.377
-0.027,-0.751,-0.332
-0.007,-0.752,-0.328
0.007,-0.753,-0.327
-0.009,-0.75,-0.359
0.008,-0.749,-0.357
0.018,-0.749,-0.357
-0.003,-0.741,-0.376
0.011,-0.739,-0.376
0.021,-0.739,-0.376
0.022,-0.754,-0.325
0.04,-0.751,-0.326
0.029,-0.749,-0.357
0.045,-0.749,-0.357
0.032,-0.739,-0.375
0.046,-0.739,-0.375
0.004,-0.73,-0.524
0.01,-0.755,-0.498
0.01,-0.754,-0.475
-0.077,-0.708,-0.511
-0.065,-0.739,-0.488
-0.061,-0.743,-0.468
-0.125,-0.686,-0.488
-0.108,-0.715,-0.474
-0.104,-0.718,-0.455
-0.193,-0.655,-0.455
-0.288,-0.563,-0.383
-0.168,-0.68,-0.456
-0.262,-0.584,-0.392
-0.164,-0.683,-0.436
-0.248,-0.587,-0.395
0.033,1.383,-0.779
0.033,1.364,-0.963
0.033,1.364,-1.1
-0.003,1.39,-0.777
-0.006,1.368,-0.962
-0.006,1.368,-1.099
-0.032,1.386,-0.779
-0.035,1.364,-0.963
-0.035,1.364,-1.1
-0.073,1.38,-0.782
-0.076,1.358,-0.964
-0.076,1.358,-1.101
-0.129,1.366,-0.787
-0.13,1.345,-0.97
-0.13,1.345,-1.104
-0.168,1.353,-0.791
-0.169,1.332,-0.972
-0.169,1.332,-1.106
-0.213,1.338,-0.796
-0.214,1.317,-0.974
-0.214,1.317,-1.108
-0.274,1.309,-0.797
-0.273,1.292,-0.978
-0.273,1.292,-1.11
-0.315,1.286,-0.805
-0.315,1.268,-0.982
-0.315,1.268,-1.113
-0.36,1.261,-0.814
-0.36,1.243,-0.985
-0.36,1.243,-1.117
-0.416,1.217,-0.825
-0.415,1.203,-1.0
-0.415,1.203,-1.124
-0.456,1.186,-0.839
-0.456,1.172,-1.005
-0.456,1.172,-1.13
-0.484,1.166,-0.848
-0.483,1.152,-1.009
-0.483,1.152,-1.134
-0.52,1.138,-0.857
-0.521,1.124,-1.022
-0.521,1.124,-1.14
-0.542,1.113,-0.87
-0.543,1.099,-1.028
-0.543,1.099,-1.145
-0.612,1.001,-0.913
-0.613,0.99,-1.059
-0.613,0.99,-1.167
-0.628,0.975,-0.927
-0.628,0.965,-1.064
-0.628,0.965,-1.173
-0.646,0.945,-0.941
-0.65,0.933,-1.084
-0.65,0.933,-1.183
-0.658,0.919,-0.958
-0.663,0.906,-1.091
-0.663,0.906,-1.19
-0.592,1.033,-0.895
-0.568,1.083,-0.884
-0.592,1.023,-1.051
-0.568,1.069,-1.034
-0.592,1.023,-1.16
-0.568,1.069,-1.151
-0.674,0.883,-0.98
-0.679,0.871,-1.1
-0.679,0.871,-1.199
-0.695,0.834,-1.003
-0.7,0.824,-1.129
-0.7,0.824,-1.215
-0.703,0.793,-1.021
-0.708,0.785,-1.136
-0.708,0.785,-1.222
-0.712,0.746,-1.042
-0.717,0.735,-1.145
-0.717,0.735,-1.231
-0.716,0.667,-1.048
-0.72,0.667,-1.159
-0.72,0.667,-1.236
-0.714,0.616,-1.062
-0.718,0.616,-1.165
-0.718,0.616,-1.243
-0.713,0.59,-1.07
-0.717,0.59,-1.169
-0.717,0.59,-1.246
-0.699,0.559,-1.075
-0.704,0.556,-1.177
-0.704,0.556,-1.25
-0.691,0.534,-1.082
-0.696,0.531,-1.179
-0.696,0.531,-1.252
-0.683,0.507,-1.088
-0.688,0.504,-1.182
-0.688,0.504,-1.255
-0.671,0.48,-1.091
-0.676,0.469,-1.189
-0.676,0.469,-1.258
-0.661,0.454,-1.097
-0.665,0.443,-1.191
-0.665,0.443,-1.261
-0.652,0.432,-1.102
-0.657,0.422,-1.193
-0.657,0.422,-1.262
-0.642,0.412,-1.106
-0.644,0.394,-1.199
-0.644,0.394,-1.266
-0.631,0.391,-1.107
-0.633,0.373,-1.199
-0.633,0.373,-1.267
-0.607,0.344,-1.111
-0.572,0.287,-1.11
-0.547,0.24,-1.108
-0.609,0.326,-1.201
-0.575,0.264,-1.202
-0.549,0.217,-1.201
-0.609,0.326,-1.268
-0.575,0.264,-1.268
-0.549,0.217,-1.268
-0.513,0.177,-1.106
-0.477,0.081,-1.102
-0.432,0.025,-1.096
-0.515,0.153,-1.2
-0.48,0.062,-1.194
-0.435,0.005,-1.192
-0.515,0.153,-1.267
-0.48,0.062,-1.263
-0.435,0.005,-1.261
-0.396,-0.022,-1.092
-0.33,-0.056,-1.083
-0.283,-0.092,-1.08
-0.399,-0.041,-1.19
-0.331,-0.085,-1.184
-0.284,-0.121,-1.183
-0.399,-0.041,-1.259
-0.331,-0.085,-1.256
-0.284,-0.121,-1.255
-0.235,-0.129,-1.077
-0.174,-0.183,-1.077
-0.12,-0.209,-1.077
-0.236,-0.158,-1.181
-0.176,-0.214,-1.18
-0.121,-0.237,-1.18
-0.236,-0.158,-1.253
-0.176,-0.214,-1.253
-0.121,-0.237,-1.253
-0.068,-0.233,-1.076
0.007,-0.253,-1.074
-0.069,-0.259,-1.179
0.008,-0.27,-1.179
-0.069,-0.259,-1.252
0.008,-0.267,-1.253
0.09,-0.731,-0.39
0.079,-0.732,-0.39
0.067,-0.731,-0.39
0.056,-0.731,-0.39
0.092,-0.71,-0.404
0.081,-0.71,-0.404
0.067,-0.71,-0.404
0.056,-0.71,-0.404
0.092,-0.693,-0.405
0.081,-0.693,-0.405
0.067,-0.693,-0.405
0.056,-0.693,-0.405
0.1,-0.731,-0.39
0.102,-0.71,-0.404
0.102,-0.693,-0.405
0.111,-0.734,-0.389
0.116,-0.712,-0.404
0.116,-0.696,-0.405
0.121,-0.731,-0.39
0.125,-0.709,-0.403
0.125,-0.693,-0.404
0.151,-0.722,-0.392
0.15,-0.703,-0.403
0.15,-0.687,-0.404
0.187,-0.709,-0.393
0.184,-0.688,-0.404
0.183,-0.676,-0.404
0.214,-0.693,-0.393
0.206,-0.677,-0.402
0.205,-0.664,-0.402
0.241,-0.678,-0.393
0.225,-0.667,-0.4
0.223,-0.654,-0.4
0.266,-0.653,-0.384
0.247,-0.646,-0.395
0.242,-0.632,-0.396
0.291,-0.634,-0.383
0.265,-0.636,-0.392
0.26,-0.622,-0.393
0.321,-0.612,-0.38
0.302,-0.616,-0.387
0.297,-0.602,-0.388
0.41,-0.557,-0.364
0.399,-0.554,-0.37
0.397,-0.549,-0.371
0.4,-0.566,-0.373
0.389,-0.569,-0.375
0.388,-0.564,-0.375
0.379,-0.586,-0.392
0.375,-0.593,-0.382
0.373,-0.588,-0.382
0.279,-0.685,-0.421
0.275,-0.68,-0.41
0.275,-0.666,-0.406
0.214,-0.721,-0.441
0.216,-0.708,-0.417
0.216,-0.694,-0.414
0.173,-0.743,-0.453
0.174,-0.728,-0.423
0.174,-0.714,-0.42
0.109,-0.753,-0.455
0.108,-0.736,-0.424
0.108,-0.717,-0.418
0.06,-0.753,-0.455
0.061,-0.736,-0.424
0.061,-0.717,-0.418
0.01,-0.753,-0.455
0.011,-0.736,-0.424
0.011,-0.717,-0.418
-0.058,-0.745,-0.454
-0.059,-0.729,-0.423
-0.059,-0.716,-0.42
-0.101,-0.721,-0.441
-0.104,-0.708,-0.417
-0.104,-0.694,-0.414
-0.161,-0.686,-0.422
-0.243,-0.588,-0.396
-0.265,-0.565,-0.374
-0.157,-0.682,-0.41
-0.237,-0.596,-0.383
-0.253,-0.569,-0.375
-0.157,-0.668,-0.407
-0.235,-0.591,-0.383
-0.251,-0.564,-0.375
-0.275,-0.555,-0.365
-0.199,-0.613,-0.38
-0.174,-0.634,-0.382
-0.263,-0.554,-0.37
-0.182,-0.616,-0.387
-0.151,-0.635,-0.392
-0.261,-0.548,-0.371
-0.178,-0.603,-0.388
-0.147,-0.622,-0.393
-0.151,-0.653,-0.383
-0.127,-0.678,-0.392
-0.101,-0.693,-0.392
-0.134,-0.646,-0.395
-0.112,-0.667,-0.401
-0.093,-0.677,-0.402
-0.13,-0.633,-0.396
-0.111,-0.654,-0.401
-0.092,-0.664,-0.402
-0.074,-0.709,-0.392
-0.038,-0.722,-0.392
-0.008,-0.731,-0.39
-0.071,-0.689,-0.404
-0.037,-0.703,-0.404
-0.013,-0.709,-0.404
-0.07,-0.676,-0.404
-0.037,-0.687,-0.405
-0.013,-0.693,-0.405
0.002,-0.734,-0.39
0.013,-0.731,-0.39
0.023,-0.731,-0.39
-0.003,-0.712,-0.404
0.011,-0.71,-0.404
0.021,-0.71,-0.404
-0.003,-0.696,-0.405
0.011,-0.693,-0.405
0.021,-0.693,-0.405
0.034,-0.732,-0.39
0.046,-0.731,-0.39
0.032,-0.71,-0.404
0.046,-0.71,-0.404
0.032,-0.693,-0.405
0.046,-0.693,-0.405
0.234,-0.237,-1.508
0.288,-0.213,-1.508
0.349,-0.158,-1.508
0.396,-0.121,-1.508
0.234,-0.237,-1.431
0.288,-0.213,-1.431
0.349,-0.158,-1.432
0.396,-0.121,-1.432
0.234,-0.237,-1.329
0.288,-0.213,-1.33
0.349,-0.158,-1.329
0.396,-0.121,-1.331
0.187,-0.258,-1.508
0.187,-0.258,-1.431
0.187,-0.258,-1.329
0.116,-0.267,-1.508
0.116,-0.267,-1.431
0.116,-0.264,-1.329
0.065,-0.267,-1.508
0.065,-0.267,-1.431
0.065,-0.264,-1.329
0.008,-0.267,-1.508
-0.069,-0.259,-1.508
-0.121,-0.237,-1.508
0.008,-0.267,-1.431
-0.069,-0.259,-1.431
-0.121,-0.237,-1.431
0.008,-0.264,-1.329
-0.069,-0.259,-1.329
-0.121,-0.237,-1.329
-0.176,-0.214,-1.508
-0.236,-0.158,-1.508
-0.284,-0.121,-1.508
-0.176,-0.214,-1.431
-0.236,-0.158,-1.432
-0.284,-0.121,-1.432
-0.176,-0.214,-1.33
-0.236,-0.158,-1.329
-0.284,-0.121,-1.331
-0.331,-0.085,-1.508
-0.399,-0.041,-1.508
-0.435,0.005,-1.508
-0.331,-0.085,-1.432
-0.399,-0.041,-1.434
-0.435,0.005,-1.434
-0.331,-0.085,-1.332
-0.399,-0.041,-1.333
-0.435,0.005,-1.335
-0.48,0.062,-1.508
-0.515,0.153,-1.508
-0.549,0.217,-1.508
-0.48,0.062,-1.434
-0.515,0.153,-1.436
-0.549,0.217,-1.436
-0.48,0.062,-1.337
-0.515,0.153,-1.339
-0.549,0.217,-1.34
-0.575,0.264,-1.508
-0.609,0.326,-1.508
-0.633,0.373,-1.508
-0.575,0.264,-1.436
-0.609,0.326,-1.436
-0.633,0.373,-1.436
-0.575,0.264,-1.341
-0.609,0.326,-1.341
-0.633,0.373,-1.339
-0.644,0.394,-1.508
-0.657,0.422,-1.508
-0.665,0.443,-1.508
-0.644,0.394,-1.436
-0.657,0.422,-1.434
-0.665,0.443,-1.434
-0.644,0.394,-1.338
-0.657,0.422,-1.337
-0.665,0.443,-1.335
-0.676,0.469,-1.508
-0.688,0.504,-1.508
-0.696,0.531,-1.508
-0.676,0.469,-1.434
-0.688,0.504,-1.431
-0.696,0.531,-1.431
-0.676,0.469,-1.332
-0.688,0.504,-1.332
-0.696,0.531,-1.329
-0.704,0.556,-1.508
-0.717,0.59,-1.508
-0.718,0.616,-1.508
-0.704,0.556,-1.431
-0.717,0.59,-1.428
-0.718,0.616,-1.428
-0.704,0.556,-1.326
-0.717,0.59,-1.325
-0.718,0.616,-1.322
-0.72,0.667,-1.508
-0.717,0.735,-1.508
-0.708,0.785,-1.508
-0.72,0.667,-1.428
-0.717,0.735,-1.422
-0.708,0.785,-1.422
-0.72,0.667,-1.316
-0.717,0.735,-1.317
-0.708,0.785,-1.308
-0.7,0.823,-1.508
-0.679,0.871,-1.508
-0.663,0.906,-1.508
-0.7,0.823,-1.422
-0.679,0.871,-1.413
-0.663,0.906,-1.413
-0.7,0.824,-1.301
-0.679,0.871,-1.295
-0.663,0.906,-1.285
-0.65,0.933,-1.508
-0.628,0.965,-1.508
-0.613,0.99,-1.508
-0.65,0.933,-1.413
-0.628,0.965,-1.406
-0.613,0.99,-1.406
-0.65,0.933,-1.278
-0.628,0.965,-1.275
-0.613,0.99,-1.269
-0.592,1.023,-1.508
-0.568,1.069,-1.508
-0.543,1.099,-1.508
-0.592,1.023,-1.406
-0.568,1.069,-1.399
-0.543,1.099,-1.399
-0.592,1.023,-1.262
-0.568,1.069,-1.26
-0.543,1.099,-1.254
-0.521,1.125,-1.508
-0.483,1.152,-1.508
-0.456,1.172,-1.508
-0.521,1.125,-1.399
-0.483,1.152,-1.395
-0.456,1.172,-1.395
-0.521,1.124,-1.249
-0.483,1.152,-1.247
-0.456,1.172,-1.243
-0.415,1.203,-1.508
-0.36,1.243,-1.508
-0.315,1.268,-1.508
-0.415,1.203,-1.395
-0.36,1.243,-1.39
-0.315,1.268,-1.39
-0.415,1.203,-1.237
-0.36,1.243,-1.235
-0.315,1.268,-1.232
-0.273,1.292,-1.508
-0.214,1.317,-1.508
-0.169,1.332,-1.508
-0.273,1.292,-1.39
-0.214,1.317,-1.387
-0.169,1.332,-1.387
-0.273,1.292,-1.228
-0.214,1.317,-1.229
-0.169,1.332,-1.227
-0.13,1.345,-1.508
-0.076,1.358,-1.508
-0.035,1.364,-1.508
-0.13,1.345,-1.387
-0.076,1.358,-1.386
-0.035,1.364,-1.386
-0.13,1.345,-1.225
-0.076,1.358,-1.224
-0.035,1.364,-1.222
-0.006,1.368,-1.508
0.033,1.364,-1.508
0.062,1.364,-1.508
-0.006,1.368,-1.386
0.033,1.364,-1.386
0.062,1.364,-1.386
-0.006,1.368,-1.222
0.033,1.364,-1.222
0.062,1.364,-1.222
0.088,1.364,-1.508
0.122,1.368,-1.508
0.148,1.364,-1.508
0.088,1.364,-1.386
0.122,1.368,-1.386
0.148,1.364,-1.386
0.088,1.364,-1.222
0.122,1.368,-1.222
0.148,1.364,-1.222
0.188,1.358,-1.508
0.242,1.345,-1.508
0.282,1.332,-1.508
0.188,1.358,-1.386
0.242,1.345,-1.387
0.282,1.332,-1.387
0.188,1.358,-1.224
0.242,1.345,-1.225
0.282,1.332,-1.227
0.327,1.317,-1.508
0.386,1.292,-1.508
0.427,1.268,-1.508
0.327,1.317,-1.387
0.386,1.292,-1.39
0.427,1.268,-1.39
0.327,1.317,-1.229
0.386,1.292,-1.228
0.427,1.268,-1.232
0.472,1.243,-1.508
0.527,1.203,-1.508
0.568,1.172,-1.508
0.472,1.243,-1.39
0.527,1.203,-1.395
0.568,1.172,-1.395
0.472,1.243,-1.235
0.527,1.203,-1.237
0.568,1.172,-1.243
0.596,1.152,-1.508
0.633,1.125,-1.508
0.656,1.099,-1.508
0.596,1.152,-1.395
0.633,1.125,-1.399
0.656,1.099,-1.399
0.596,1.152,-1.247
0.633,1.124,-1.249
0.656,1.099,-1.254
0.681,1.069,-1.508
0.705,1.023,-1.508
0.725,0.99,-1.508
0.681,1.069,-1.399
0.705,1.023,-1.406
0.725,0.99,-1.406
0.681,1.069,-1.26
0.705,1.023,-1.262
0.725,0.99,-1.269
0.741,0.965,-1.508
0.763,0.933,-1.508
0.775,0.906,-1.508
0.741,0.965,-1.406
0.763,0.933,-1.413
0.775,0.906,-1.413
0.741,0.965,-1.275
0.763,0.933,-1.278
0.775,0.906,-1.285
0.792,0.871,-1.508
0.813,0.823,-1.508
0.82,0.785,-1.508
0.792,0.871,-1.413
0.813,0.823,-1.422
0.82,0.785,-1.422
0.792,0.871,-1.295
0.813,0.824,-1.301
0.82,0.785,-1.308
0.83,0.735,-1.508
0.833,0.667,-1.508
0.831,0.616,-1.508
0.83,0.735,-1.422
0.833,0.667,-1.428
0.831,0.616,-1.428
0.83,0.735,-1.317
0.833,0.667,-1.316
0.831,0.616,-1.322
0.829,0.59,-1.508
0.817,0.556,-1.508
0.809,0.531,-1.508
0.829,0.59,-1.428
0.817,0.556,-1.431
0.809,0.531,-1.431
0.829,0.59,-1.325
0.817,0.556,-1.326
0.809,0.531,-1.329
0.801,0.504,-1.508
0.789,0.469,-1.508
0.778,0.443,-1.508
0.801,0.504,-1.431
0.789,0.469,-1.434
0.778,0.443,-1.434
0.801,0.504,-1.332
0.789,0.469,-1.332
0.778,0.443,-1.335
0.77,0.422,-1.508
0.756,0.394,-1.508
0.746,0.373,-1.508
0.77,0.422,-1.434
0.756,0.394,-1.436
0.746,0.373,-1.436
0.77,0.422,-1.337
0.757,0.394,-1.338
0.746,0.373,-1.339
0.722,0.326,-1.508
0.687,0.264,-1.508
0.662,0.217,-1.508
0.722,0.326,-1.436
0.687,0.264,-1.436
0.662,0.217,-1.436
0.722,0.326,-1.341
0.687,0.264,-1.341
0.662,0.217,-1.34
0.628,0.153,-1.508
0.593,0.062,-1.508
0.548,0.005,-1.508
0.628,0.153,-1.436
0.593,0.062,-1.434
0.548,0.005,-1.434
0.628,0.153,-1.339
0.593,0.062,-1.337
0.548,0.005,-1.335
0.511,-0.041,-1.508
0.443,-0.085,-1.508
0.511,-0.041,-1.434
0.443,-0.085,-1.432
0.511,-0.041,-1.333
0.443,-0.085,-1.332
0.092,-0.513,-0.389
0.092,-0.567,-0.394
0.092,-0.639,-0.408
0.081,-0.513,-0.389
0.081,-0.567,-0.394
0.081,-0.639,-0.408
0.067,-0.513,-0.389
0.067,-0.567,-0.394
0.067,-0.639,-0.408
0.056,-0.513,-0.389
0.056,-0.567,-0.394
0.056,-0.639,-0.408
0.102,-0.513,-0.389
0.115,-0.513,-0.388
0.125,-0.513,-0.388
0.102,-0.567,-0.394
0.115,-0.567,-0.393
0.125,-0.567,-0.393
0.102,-0.639,-0.408
0.116,-0.642,-0.407
0.125,-0.639,-0.407
0.149,-0.513,-0.388
0.181,-0.513,-0.387
0.205,-0.513,-0.387
0.149,-0.567,-0.393
0.181,-0.558,-0.392
0.205,-0.558,-0.391
0.15,-0.633,-0.406
0.178,-0.63,-0.405
0.201,-0.619,-0.403
0.222,-0.513,-0.387
0.244,-0.513,-0.387
0.26,-0.513,-0.386
0.222,-0.558,-0.391
0.244,-0.546,-0.389
0.26,-0.546,-0.388
0.219,-0.609,-0.401
0.231,-0.601,-0.398
0.249,-0.591,-0.396
0.299,-0.513,-0.384
0.403,-0.513,-0.37
0.388,-0.513,-0.377
0.299,-0.546,-0.386
0.403,-0.528,-0.37
0.388,-0.528,-0.376
0.286,-0.571,-0.39
0.392,-0.534,-0.372
0.383,-0.55,-0.377
0.366,-0.513,-0.385
0.268,-0.513,-0.399
0.216,-0.513,-0.404
0.366,-0.528,-0.385
0.268,-0.567,-0.402
0.216,-0.567,-0.407
0.368,-0.574,-0.384
0.275,-0.614,-0.393
0.216,-0.641,-0.401
0.17,-0.513,-0.408
0.107,-0.513,-0.406
0.061,-0.513,-0.406
0.17,-0.567,-0.411
0.107,-0.574,-0.41
0.061,-0.574,-0.41
0.174,-0.661,-0.406
0.108,-0.658,-0.401
0.061,-0.658,-0.401
0.011,-0.513,-0.406
-0.055,-0.513,-0.408
-0.104,-0.513,-0.404
0.011,-0.574,-0.41
-0.055,-0.567,-0.411
-0.104,-0.567,-0.407
0.011,-0.658,-0.401
-0.059,-0.663,-0.407
-0.104,-0.641,-0.401
-0.148,-0.513,-0.4
-0.229,-0.513,-0.386
-0.251,-0.513,-0.377
-0.148,-0.567,-0.403
-0.229,-0.528,-0.385
-0.251,-0.528,-0.376
-0.157,-0.615,-0.394
-0.23,-0.576,-0.384
-0.246,-0.55,-0.376
-0.267,-0.513,-0.37
-0.179,-0.513,-0.384
-0.147,-0.513,-0.386
-0.267,-0.528,-0.37
-0.179,-0.546,-0.386
-0.147,-0.546,-0.388
-0.256,-0.534,-0.372
-0.168,-0.572,-0.391
-0.138,-0.591,-0.396
-0.131,-0.513,-0.387
-0.109,-0.513,-0.387
-0.092,-0.513,-0.387
-0.131,-0.546,-0.389
-0.109,-0.558,-0.392
-0.092,-0.558,-0.392
-0.12,-0.602,-0.399
-0.106,-0.609,-0.402
-0.088,-0.619,-0.403
-0.068,-0.513,-0.388
-0.036,-0.513,-0.388
-0.013,-0.513,-0.389
-0.068,-0.558,-0.392
-0.036,-0.567,-0.393
-0.013,-0.567,-0.394
-0.065,-0.63,-0.405
-0.037,-0.633,-0.407
-0.013,-0.639,-0.408
-0.002,-0.513,-0.389
0.011,-0.513,-0.389
0.021,-0.513,-0.389
-0.002,-0.567,-0.394
0.011,-0.567,-0.394
0.021,-0.567,-0.394
-0.003,-0.642,-0.408
0.011,-0.639,-0.408
0.021,-0.639,-0.408
0.032,-0.513,-0.389
0.046,-0.513,-0.389
0.032,-0.567,-0.394
0.046,-0.567,-0.394
0.032,-0.639,-0.408
0.046,-0.639,-0.408
-0.491,-0.332,0.116
-0.491,-0.215,0.116
-0.391,-0.115,0.116
-0.274,-0.115,0.116
-0.491,-0.45,0.116
-0.391,-0.115,-0.002
-0.274,-0.115,-0.002
-0.391,-0.55,0.116
-0.391,-0.55,-0.002
-0.274,-0.332,-0.102
-0.274,-0.215,-0.102
-0.274,-0.55,0.116
-0.274,-0.55,-0.002
-0.274,-0.45,-0.102
-0.491,-0.332,-0.002
-0.491,-0.45,-0.002
-0.391,-0.332,-0.102
-0.391,-0.45,-0.102
-0.391,-0.55,0.233
-0.274,-0.55,0.233
-0.391,-0.115,0.233
-0.274,-0.332,0.333
-0.274,-0.45,0.333
-0.274,-0.115,0.233
-0.274,-0.215,0.333
-0.491,-0.332,0.233
-0.491,-0.45,0.233
-0.391,-0.332,0.333
-0.391,-0.45,0.333
-0.156,-0.55,0.116
-0.056,-0.45,0.233
-0.056,-0.45,0.116
-0.156,-0.332,0.333
-0.056,-0.332,0.233
-0.056,-0.332,0.116
-0.056,-0.45,-0.002
-0.156,-0.332,-0.102
-0.056,-0.332,-0.002
-0.491,-0.215,0.233
-0.391,-0.215,0.333
-0.491,-0.215,-0.002
-0.391,-0.215,-0.102
-0.056,-0.215,0.233
-0.156,-0.115,0.116
-0.056,-0.215,0.116
-0.056,-0.215,-0.002
-0.156,-0.115,0.233
-0.156,-0.55,0.233
-0.156,-0.215,0.333
-0.156,-0.55,-0.002
-0.156,-0.115,-0.002
-0.156,-0.215,-0.102
-0.156,-0.45,-0.102
-0.156,-0.45,0.333
-0.372,-0.486,0.116
-0.372,-0.439,0.116
-0.321,-0.41,0.116
-0.274,-0.41,0.116
-0.372,-0.534,0.116
-0.321,-0.41,0.068
-0.274,-0.41,0.068
-0.321,-0.563,0.116
-0.321,-0.563,0.068
-0.274,-0.486,0.017
-0.274,-0.439,0.017
-0.274,-0.563,0.116
-0.274,-0.563,0.068
-0.274,-0.534,0.017
-0.372,-0.486,0.062
-0.372,-0.534,0.062
-0.327,-0.486,0.017
-0.327,-0.534,0.017
-0.321,-0.563,0.163
-0.274,-0.563,0.163
-0.321,-0.41,0.163
-0.274,-0.486,0.214
-0.274,-0.534,0.214
-0.274,-0.41,0.163
-0.274,-0.439,0.214
-0.372,-0.486,0.169
-0.372,-0.534,0.169
-0.327,-0.486,0.214
-0.327,-0.534,0.214
-0.226,-0.563,0.116
-0.175,-0.534,0.169
-0.175,-0.534,0.116
-0.22,-0.486,0.214
-0.175,-0.486,0.169
-0.175,-0.486,0.116
-0.175,-0.534,0.062
-0.22,-0.486,0.017
-0.175,-0.486,0.062
-0.372,-0.439,0.169
-0.327,-0.439,0.214
-0.372,-0.439,0.062
-0.327,-0.439,0.017
-0.175,-0.439,0.169
-0.226,-0.41,0.116
-0.175,-0.439,0.116
-0.175,-0.439,0.062
-0.226,-0.41,0.163
-0.226,-0.563,0.163
-0.22,-0.439,0.214
-0.226,-0.563,0.068
-0.226,-0.41,0.068
-0.22,-0.439,0.017
-0.22,-0.534,0.017
-0.22,-0.534,0.214
0.166,-0.332,0.116
0.166,-0.215,0.116
0.266,-0.115,0.116
0.384,-0.115,0.116
0.166,-0.45,0.116
0.266,-0.115,-0.002
0.384,-0.115,-0.002
0.266,-0.55,0.116
0.266,-0.55,-0.002
0.384,-0.332,-0.102
0.384,-0.215,-0.102
0.384,-0.55,0.116
0.384,-0.55,-0.002
0.384,-0.45,-0.102
0.166,-0.332,-0.002
0.166,-0.45,-0.002
0.266,-0.332,-0.102
0.266,-0.45,-0.102
0.266,-0.55,0.233
0.384,-0.55,0.233
0.266,-0.115,0.233
0.384,-0.332,0.333
0.384,-0.45,0.333
0.384,-0.115,0.233
0.384,-0.215,0.333
0.166,-0.332,0.233
0.166,-0.45,0.233
0.266,-0.332,0.333
0.266,-0.45,0.333
0.501,-0.55,0.116
0.601,-0.45,0.233
0.601,-0.45,0.116
0.501,-0.332,0.333
0.601,-0.332,0.233
0.601,-0.332,0.116
0.601,-0.45,-0.002
0.501,-0.332,-0.102
0.601,-0.332,-0.002
0.166,-0.215,0.233
0.266,-0.215,0.333
0.166,-0.215,-0.002
0.266,-0.215,-0.102
0.601,-0.215,0.233
0.501,-0.115,0.116
0.601,-0.215,0.116
0.601,-0.215,-0.002
0.501,-0.115,0.233
0.501,-0.55,0.233
0.501,-0.215,0.333
0.501,-0.55,-0.002
0.501,-0.115,-0.002
0.501,-0.215,-0.102
0.501,-0.45,-0.102
0.501,-0.45,0.333
0.285,-0.486,0.116
0.285,-0.439,0.116
0.336,-0.41,0.116
0.384,-0.41,0.116
0.285,-0.534,0.116
0.336,-0.41,0.068
0.384,-0.41,0.068
0.336,-0.563,0.116
0.336,-0.563,0.068
0.384,-0.486,0.017
0.384,-0.439,0.017
0.384,-0.563,0.116
0.384,-0.563,0.068
0.384,-0.534,0.017
0.285,-0.486,0.062
0.285,-0.534,0.062
0.33,-0.486,0.017
0.33,-0.534,0.017
0.336,-0.563,0.163
0.384,-0.563,0.163
0.336,-0.41,0.163
0.384,-0.486,0.214
0.384,-0.534,0.214
0.384,-0.41,0.163
0.384,-0.439,0.214
0.285,-0.486,0.169
0.285,-0.534,0.169
0.33,-0.486,0.214
0.33,-0.534,0.214
0.431,-0.563,0.116
0.482,-0.534,0.169
0.482,-0.534,0.116
0.437,-0.486,0.214
0.482,-0.486,0.169
0.482,-0.486,0.116
0.482,-0.534,0.062
0.437,-0.486,0.017
0.482,-0.486,0.062
0.285,-0.439,0.169
0.33,-0.439,0.214
0.285,-0.439,0.062
0.33,-0.439,0.017
0.482,-0.439,0.169
0.431,-0.41,0.116
0.482,-0.439,0.116
0.482,-0.439,0.062
0.431,-0.41,0.163
0.431,-0.563,0.163
0.437,-0.439,0.214
0.431,-0.563,0.068
0.431,-0.41,0.068
0.437,-0.439,0.017
0.437,-0.534,0.017
0.437,-0.534,0.214
'''
