##
# DG LS8 (sc) Electrical System
#	by Benedikt Wolf (D-ECHO) 05/2020

#	adapted from: Cessna 172p Electrical System


var volts = 0.0;

var bus_volts = 0.0;

var ammeter_ave = 0.0;

# Initialize properties
var electrical	=	props.globals.initNode("/systems/electrical");
var serviceable	=	electrical.initNode("serviceable", 1, "BOOL");
var batt_prop	=	electrical.initNode("battery");
var output_prop	=	electrical.initNode("outputs");
var cb_prop	=	props.globals.initNode("/controls/circuit-breakers");

var volts	=	electrical.initNode("volts", 0.0, "DOUBLE");
var amps	=	electrical.initNode("volts", 0.0, "DOUBLE");
var master_sw	=	props.globals.initNode("/controls/electric/battery-selector-switch", 0, "INT");

var radio	=	output_prop.initNode("comm[0]", 0.0, "DOUBLE");
var flarm	=	output_prop.initNode("flarm", 0.0, "DOUBLE");
var s3		=	output_prop.initNode("S3", 0.0, "DOUBLE");
var xpdr	=	output_prop.initNode("transponder", 0.0, "DOUBLE");

var comm_ptt	=	props.globals.getNode("/instrumentation/comm[0]/ptt");
var xpdr_mode	=	props.globals.initNode("/instrumentation/transponder/inputs/knob-mode", 0, "INT");

# Array of circuit breakers
var circuit_breakers = {
	master: cb_prop.initNode("master", 1, "BOOL"),
};

var freeze_replay	=	props.globals.getNode("/sim/freeze/replay-state");

##
# Battery model class.
#

var BatteryClass = {
	# Constructor
	new: func( ideal_volts, ideal_amps, amp_hours, charge_amps, n ){
		var charge_prop	= batt_prop.getNode( "charge["~n~"]" );
		var charge	= nil;
		if( getprop("/systems/electrical/battery/charge["~n~"]") != nil ){			# If the battery charge has been set from a previous FG instance
			charge = charge_prop.getDoubleValue();
		} else {
			charge = 1.0;
			charge_prop = batt_prop.initNode("charge["~n~"]", 1.0, "DOUBLE");
		}
		var obj = {
			parents: [BatteryClass],
			ideal_volts: ideal_volts,
			ideal_amps: ideal_amps,
			amp_hours: amp_hours,
			charge_amps: charge_amps,
			charge: charge,
			charge_prop: charge_prop,
			n: n 
		};
		return obj;
	},
	# Passing in positive amps means the battery will be discharged.
	# Negative amps indicates a battery charge.
	apply_load: func( amps, dt ){
		var old_charge = me.charge_prop.getDoubleValue();
		if( freeze_replay.getBoolValue() ){
			return me.amp_hours * old_charge;
		}
		var amphrs_used = amps * dt / 3600.0;
		var fraction_used = amphrs_used / me.amp_hours;
		
		var new_charge = std.max(0.0, std.min(old_charge - fraction_used, 1.0));
		
		if (new_charge < 0.1 and old_charge_percent >= 0.1)
			gui.popupTip("Warning: Low battery! Enable alternator or apply external power to recharge battery!", 10);
		me.charge = new_charge;
		me.charge_prop.setDoubleValue( new_charge );
		return me.amp_hours * new_charge;
	},
	# Return output volts based on percent charged.  Currently based on a simple
	# polynomial percent charge vs. volts function.
	get_output_volts: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32 ) / 32;
		return me.ideal_volts * factor;
	},
	# Return output amps available.  This function is totally wrong and should be
	# fixed at some point with a more sensible function based on charge percent.
	# There is probably some physical limits to the number of instantaneous amps
	# a battery can produce (cold cranking amps?)
	get_output_amps: func() {
		var x = 1.0 - me.charge;
		var tmp = -(3.0 * x - 1.0);
		var factor = ( math.pow( tmp, 5) + 32) / 32;
		return me.ideal_amps * factor;
	},
	# Set the current charge instantly to 100 %.
	reset_to_full_charge: func() {
		me.apply_load(-(1.0 - me.charge) * me.amp_hours, 3600);
	},
	# Get current charge
	get_charge: func() {
		return me.charge;
	}
};

############################
####	Battery Packs	####
############################

##	example glider battery: https://shop.segelflugbedarf24.de/Flugzeugausstattung/Akkus-Energieversorgung/Sonnenschein-Dryfit-A512-12V/6-5-Ah::731.html
##				http://www.sonnenschein.org/A500.htm	(A512-6.5S)
##				ideal volts: 12.0
##				ideal amps: 0.325 (max. 80 / 300 for 5 sec))
##				amp hours: 6.5
##				charge amps: 25

var	battery_1 = BatteryClass.new( 12.0, 0.325, 6.5, 25, 0);
var	battery_2 = BatteryClass.new( 12.0, 0.325, 6.5, 25, 0);

var reset_battery = func {
	battery_1.reset_to_full_charge();
	battery_2.reset_to_full_charge();
}
var reset_circuit_breakers = func {
	# Reset circuit breakers
	foreach(var cb; keys(circuit_breakers)){
		circuit_breakers[cb].setBoolValue( 1 );
	}
}

##
# This is the main electrical system update function.
#

var ElectricalSystemUpdater = {
	new : func {
		var m = {
			parents: [ElectricalSystemUpdater]
		};
		# Request that the update function be called each frame
		m.loop = updateloop.UpdateLoop.new(components: [m], update_period: 0.0, enable: 0);
		return m;
	},
	
	enable: func {
		me.loop.reset();
		me.loop.enable();
	},
	
	disable: func {
		me.loop.disable();
	},
	
	reset: func {
		# Do nothing
	},
	
	update: func (dt) {
		update_bus(dt);
	}
};


var update_bus = func (dt) {
	if( !serviceable.getBoolValue() ){
		return;
	}
	var load = 0.0;
	var bus_volts = 0.0;
	
	var master_pos = master_sw.getIntValue();
	
	if( circuit_breakers.master.getBoolValue() and master_pos != 0){
		if( master_pos == 1 ){
			bus_volts = battery_1.get_output_volts();
		} else if ( master_pos == 2 ){
			bus_volts = battery_2.get_output_volts();
		}
	}
	
	# switch state
	load += cockpit_bus( bus_volts );
	
	if ( load > 300 ) {
		circuit_breakers.master.setBoolValue( 0 );
	}
	
	if( master_pos == 1 ){
		battery_1.apply_load( load, dt );
	} else if ( master_pos == 2 ){
		battery_2.apply_load( load, dt );
	}
}

var cockpit_bus = func( bv ) {
	var load = 0.0;
	var bus_volts = bv;
	
	# Electrical Consumers (Instruments)
	# Radio
	radio.setDoubleValue( bus_volts );
	if( comm_ptt.getBoolValue() ){
		load += bus_volts / 15; #transmitting: 1600mA
	}else{
		load += bus_volts / 282.353; #85mA (standby)
	}
	
	# Vario
	s3.setDoubleValue( bus_volts );
	load += bus_volts / 177.78; #135mA
	
	# XPDR Power
	if( xpdr_mode.getValue() != 0 and bus_volts != 0 ){
		xpdr.setDoubleValue( bus_volts );
		load += 5.6 / bus_volts;
	} else {
		xpdr.setDoubleValue( 0.0 );
	}
	
	# FLARM
	flarm.setDoubleValue( bus_volts );
	
	amps.setDoubleValue( load );
	volts.setDoubleValue( bus_volts );
	
	return load;
}


##
# Initialize the electrical system
#

var system_updater = ElectricalSystemUpdater.new();

setlistener("/sim/signals/fdm-initialized", func {
	reset_circuit_breakers();
	
	system_updater.enable();
	print("Electrical System initialized");
});

