# Simulation of LX Instruments Salus started 09/2022 by Bea Wolf based on

# A3XX Lower ECAM Canvas
# Joshua Davidson (Octal450)

# References:
#	https://downloads.lxnavigation.com/downloads/manuals/Salus/LX-Salus_users_manual_1_4.pdf

var Salus_main = [ nil, nil, nil, nil, nil, nil, nil, nil, nil ];
var Salus_start = nil;
var Salus_display = nil;

var base		=	props.globals.getNode("/instrumentation/salus/");


var accel = props.globals.getNode("/accelerations/pilot-g", 1);

var groundspeed = props.globals.getNode("/velocities/groundspeed-kt", 1);
var airspeed = props.globals.getNode("/instrumentation/airspeed-indicator/indicated-speed-kt", 1);
var vario = props.globals.getNode("/instrumentation/vertical-speed-indicator/indicated-speed-mps", 1);
var alt_baro = props.globals.getNode("/instrumentation/altimeter[1]/indicated-altitude-ft", 1);
var qnh = props.globals.getNode("/instrumentation/altimeter[1]/setting-hpa", 1);

var alt_gps = props.globals.getNode("/position/altitude-ft", 1);

var utc = {
	year:	props.globals.getNode("/sim/time/utc/year", 1),
	month:	props.globals.getNode("/sim/time/utc/month", 1),
	day:	props.globals.getNode("/sim/time/utc/day", 1),
	hour:	props.globals.getNode("/sim/time/utc/hour", 1),
	minute:	props.globals.getNode("/sim/time/utc/minute", 1),
	second:	props.globals.getNode("/sim/time/utc/second", 1),
};

var lat = props.globals.getNode("/position/latitude-string",  1);
var lon = props.globals.getNode("/position/longitude-string", 1);

var track = props.globals.getNode("/orientation/track-deg", 1);

var flarm_present = nil;
var flarm_legacy = nil;

var instrument_dir = "Aircraft/DR400-Ecoflyer/Models/Interior/Panel/Instruments/TRX-2000/";

var volts = props.globals.getNode("systems/electrical/outputs/salus", 1);

var instrument_dir = "Aircraft/Song-120/Models/Instruments/Salus/";

var status = 0;		# 0 = off; 1 = starting; 2 = on
var page = 0;		# 0 = main
			# 1 = FLARM Radar
			# 2 = WP Navigation
			# 3 = RTE Navigation
			# 4 = G-Force
			# 5 = AHRS, if module present
			# 6 = GPS Info
			# 7 = Logbook / statistic
			# 8 = Setup

var starting = maketimer( 1.5, func() { status = 2 } );
starting.singleShot = 1;

var canvas_Salus_base = {
	init: func(canvas_group, file) {
		var font_mapper = func(family, weight) {
			if( find( "Liberation Mono", family ) != -1 ){
				if (weight == "bold"){
					return "LiberationFonts/LiberationMono-Bold.ttf";
				}elsif(weight == "normal"){
					return "LiberationFonts/LiberationMono-Regular.ttf";
				}
			}elsif( family == "'Liberation Sans'" ){
				if (weight == "bold"){
					return "LiberationFonts/LiberationSans-Bold.ttf";
				}elsif(weight == "normal"){
					return "LiberationFonts/LiberationSans-Regular.ttf";
				}
			}elsif( family == "'Liberation Serif'" ){
				if (weight == "bold"){
					return "LiberationFonts/LiberationSerif-Bold.ttf";
				}elsif(weight == "normal"){
					return "LiberationFonts/LiberationSerif-Regular.ttf";
				}
			} else {
				print( "No matching Font found: " );
				debug.dump( caller(0) );
				debug.dump( caller(1) );
				debug.dump( caller(2) );
				debug.dump( caller(3) );
				print( family );
				print( weight );
				return "LiberationFonts/LiberationSans-Regular.ttf";
			}
		};

		
		canvas.parsesvg(canvas_group, file, {'font-mapper': font_mapper});

		var svg_keys = me.getKeys();
		
		foreach (var key; svg_keys) {
			me[key] = canvas_group.getElementById(key);
			var clip_el = canvas_group.getElementById(key ~ "_clip");
			if (clip_el != nil) {
				clip_el.setVisible(0);
				var tran_rect = clip_el.getTransformedBounds();
				var clip_rect = sprintf("rect(%d,%d, %d,%d)", 
				tran_rect[1], # 0 ys
				tran_rect[2], # 1 xe
				tran_rect[3], # 2 ye
				tran_rect[0]); #3 xs
				#   coordinates are top,right,bottom,left (ys, xe, ye, xs) ref: l621 of simgear/canvas/CanvasElement.cxx
				me[key].set("clip", clip_rect);
				me[key].set("clip-frame", canvas.Element.PARENT);
			}
		}

		me.page = canvas_group;

		return me;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
		if ( volts.getDoubleValue() > 9.0 ){
			if( status == 0 ){
				status = 1;
				starting.start();
				page = 0;
			}
		} elsif( status != 0 ){
			status = 0;
			page = 0;
		}
		
		if ( status == 2 ) {
			Salus_main[ page ].page.show();
			Salus_main[ page ].update();
			forindex( var i; Salus_main ){
				if( i != page ){
					Salus_main[i].page.hide();
				}
			}
			Salus_start.page.hide();
		} elsif ( status == 1 ) {
			foreach( var el; Salus_main ){
				el.page.hide();
			}
			Salus_start.page.show();
		} else{
			foreach( var el; Salus_main ){
				el.page.hide();
			}
			Salus_start.page.hide();
		}
	},
};

var canvas_Salus_main = [ nil, nil, nil, nil, nil, nil, nil, nil, nil ];
var qnh_popup_visible = 0;
canvas_Salus_main[0] = 	{	# Main Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[0] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return ["airspeed.digits","groundspeed.digits","altitude.digits","wind.string","volts.digits","qnh.digits","gps_status.icon","vertical_speed.arch.up","vertical_speed.arch.down", "qnh_popup", "qnh_popup.qnh", "qnh_popup.alt"];
	},
	update: func() {
		me["airspeed.digits"].setText( sprintf("%3d", math.round( airspeed.getDoubleValue() * KT2MPS * 3.6 ) ) );
		me["altitude.digits"].setText( sprintf("%5d", math.round( alt_baro.getDoubleValue() * FT2M ) ) );
		me["groundspeed.digits"].setText( sprintf("%3d", math.round( groundspeed.getDoubleValue() * KT2MPS * 3.6 ) ) );
		me["volts.digits"].setText( sprintf("%3.1f", volts.getDoubleValue() ) );
		var vs = vario.getDoubleValue();
		if( vs >= 0 ){
			me["vertical_speed.arch.up"].show();
			me["vertical_speed.arch.up"].setRotation( math.min( vs / 5, 1 )  * 45 * D2R );
			me["vertical_speed.arch.down"].hide();
		} else {
			me["vertical_speed.arch.up"].hide();
			me["vertical_speed.arch.down"].show();
			me["vertical_speed.arch.down"].setRotation( math.max( vs / 5, -1 )  * 45 * D2R );
		}
	},
	show_qnh_popup: func( visible = nil ){
		if( visible == nil ) visible = 1 - qnh_popup_visible;
		if( visible ){
			me.update_qnh( 1 );
			me["qnh_popup"].show();
			qnh_popup_visible = 1;
		} else {
			me["qnh_popup"].hide();
			qnh_popup_visible = 0;
		}
	},
	update_qnh: func ( i = 0 ) {
		if( qnh_popup_visible or i ){
			me["qnh_popup.qnh"].setText( sprintf("%4d", math.round( qnh.getDoubleValue() ) ) );
			me["qnh_popup.alt"].setText( sprintf("%5d", math.round( alt_baro.getDoubleValue() * FT2M ) ) );
		}
		me["qnh.digits"].setText( sprintf("%4d", math.round( qnh.getDoubleValue() ) ) );
	},
};
canvas_Salus_main[1] = 	{	# Flarm Radar Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[1] , canvas_Salus_base] };
		m.init(canvas_group, file);
		m.target_status = [ -1, -1, -1, -1, -1 ];

		return m;
	},
	getKeys: func() {
		return ["target0","target1","target2","target3","target4","rel_altitude","vario","range","compass","compass.N","compass.E","compass.S","compass.W",];
	},
	update: func() {
		if( !flarm_present ) {
			for( var i = 0; i <= 4 ; i += 1 ){
				if( me.target_status[ i ] != 0 ){
					me["target"~ i ].hide();
					me.target_status[ i ] = 0;
				}
			}
			return;
		}
		
		var track_deg = 0.0;
		if( groundspeed.getDoubleValue() > 5 ){
			track_deg = track.getDoubleValue();
		}
		
		me["compass"].setRotation( track_deg * D2R );
		me["compass.N"].setRotation( -track_deg * D2R );
		me["compass.E"].setRotation( -track_deg * D2R );
		me["compass.S"].setRotation( -track_deg * D2R );
		me["compass.W"].setRotation( -track_deg * D2R );
		
		var targets = [];
		var x = 0;
		var i = 0;
		while( x < 5 and i <= 30 ){
			if( flarm.targets[i] != nil ){
				append( targets, flarm.targets[i] );
				x += 1;
			}
			i += 1;
		}
		var nearest = [nil, 999999];
		forindex( var i; targets ){
			if( targets[i] != nil and targets[i].last_dist < nearest[1] ){
				nearest = [ i, targets[i].last_dist ];
			}
		}
		
		if( size( targets ) < 5 ){
			for( var i = 4; i >= size( targets ); i -= 1 ){
				if( me.target_status[i] != 0 ){
					me["target"~i].hide();
					me.target_status[i] = 0;
				}
			}
		}
				
		forindex(var i; targets ){
			if(targets[i] != nil ){
				if( me.target_status[i] != 1 ){
					me["target"~i].show();
					me.target_status[i] = 1;
				}
				
				var brg_abs = geo.aircraft_position().course_to( targets[i].pos );
				var brg_rel = track_deg - brg_abs;
				
				var screen_pos = [ 	-math.sin( brg_rel * D2R ) * geo.aircraft_position().distance_to( targets[i].pos) / 3000, 
							-math.cos( brg_rel * D2R ) * geo.aircraft_position().distance_to( targets[i].pos) / 3000 ];
				
				me["target"~i].setTranslation( screen_pos[0] * 120, screen_pos[1] * 120 );
				
				if( !flarm_legacy ){
					me["target"~i].setRotation( ( targets[i].hdg.getDoubleValue() - track_deg ) * D2R );
				}
				
				if( i == nearest[0] ){
					if( !flarm_legacy ){
						me["vario"].setText( sprintf("%2.1f", targets[i].vario.getDoubleValue() * FT2M ) );
					} else {
						me["vario"].setText( "-.-" );
					}
					me["rel_altitude"].setText( sprintf("%+2dm", math.round( targets[i].alt.getDoubleValue() - alt_gps.getDoubleValue() ) ) );
				}
			}else{
				if( me.target_status[i] != 0 ){
					me["target"~i].hide();
					me.target_status[i] = 0;
				}
			}
		}
	}
};
canvas_Salus_main[2] = 	{	# WP Navigation Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[2] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
};
canvas_Salus_main[3] = 	{	# RTE Navigation Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[3] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
};
canvas_Salus_main[4] = 	{	# G-Force Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[4] , canvas_Salus_base] };
		m.init(canvas_group, file);
		m.min_g = 1.0;
		m.max_g = 1.0;
		m.menu = 0; # 0 = menu off, 1 - 4 menu cursor positions
		return m;
	},
	getKeys: func() {
		return ["current_g_force", "min_g_force", "max_g_force", "scale.needle", "menu_gr", "menu.x.circle", "menu.cursor"];
	},
	update: func() {
		var current_g = accel.getDoubleValue();
		
		me["current_g_force"].setText( sprintf("%2.1f", current_g ) );
		me["min_g_force"].setText( sprintf("%2.1f", me.min_g ) ~"G" );
		me["max_g_force"].setText( sprintf("%2.1f", me.max_g ) ~"G" );
		me["scale.needle"].setTranslation( 0, 38 * current_g );
		
		if( current_g > me.max_g ) me.max_g = current_g;
		if( current_g < me.min_g ) me.min_g = current_g;
	},
	show_menu: func( visible ){
		if( visible ){
			me.menu = 2;
			me.update_menu_cursor();
			me["menu_gr"].show();
		} else {
			me.menu = 0;
			me["menu_gr"].hide();
		}
	},
	update_menu_cursor: func {
		if( me.menu == 0 ){
			return;
		} elsif( me.menu == 1 ){
			me["menu.x.circle"].show();
			me["menu.cursor"].hide();
		} else {
			me["menu.x.circle"].hide();
			me["menu.cursor"].setTranslation( 0, ( me.menu - 2 ) * 54 );
			me["menu.cursor"].show();
		}
	},
	reset_g_limits: func {
		var current_g = accel.getDoubleValue();
		me.min_g = current_g;
		me.max_g = current_g;
	},
};

canvas_Salus_main[5] = 	{	# AHRS Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[5] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
};
canvas_Salus_main[6] = 	{	# GPS Info Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[6] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [ "status", "lat", "lon", "time", "date"];
	},
	update: func() {
		# TODO proper GPS simulation, show correct GPS status
		
		me["lat"].setText( "Lat: "~ lat.getValue() );
		me["lon"].setText( "Lon: "~ lon.getValue() );
		
		me["time"].setText( sprintf("Time: %02d:%02d:%02d UTC", utc.hour.getIntValue(), utc.minute.getIntValue(), utc.second.getIntValue() ) );
		me["date"].setText( sprintf("Date: %02d.%02d.%02d", utc.day.getIntValue(), utc.month.getIntValue(), utc.year.getIntValue() ) );
	}
};
canvas_Salus_main[7] = 	{	# Logbook / Statistic Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[7] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
};
canvas_Salus_main[8] = 	{	# Setup Page
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_main[8] , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() {
	}
};

var canvas_Salus_start = {
	new: func(canvas_group, file) {
		var m = { parents: [canvas_Salus_start , canvas_Salus_base] };
		m.init(canvas_group, file);

		return m;
	},
	getKeys: func() {
		return [];
	},
	update: func() { },
};

var base_updater = maketimer( 0.02, canvas_Salus_base.update );
base_updater.simulatedTime = 1;

setlistener("sim/signals/fdm-initialized", func {
	Salus_display = canvas.new({
		"name": "Salus",
		"size": [320, 256],
		"view": [320, 256],
		"mipmapping": 1
	});
	Salus_display.addPlacement({"node": "salus.display"});
	
	if( flarm.flarm_base != nil ){
		flarm_present = 1;
		if( flarm.version != nil and flarm.version >= 202209 ){
			flarm_legacy = 0;
		} else {
			flarm_legacy = 1;
			print( "Old FLARM Module Detected: FLARM Radar limited!" );
		}
	} else {
		flarm_present = 0;
		print( "CAUTION: FLARM MODULE NOT AVAILABLE, SALUS FLARM RADAR NOT AVAILABLE!" );
	}

	Salus_main[0] = canvas_Salus_main[0].new(Salus_display.createGroup(), instrument_dir~"salus.svg");
	Salus_main[1] = canvas_Salus_main[1].new(Salus_display.createGroup(), instrument_dir~"salus-flarm_radar.svg");
	Salus_main[2] = canvas_Salus_main[2].new(Salus_display.createGroup(), instrument_dir~"salus-wp_nav.svg");
	Salus_main[3] = canvas_Salus_main[3].new(Salus_display.createGroup(), instrument_dir~"salus-rte_nav.svg");
	Salus_main[4] = canvas_Salus_main[4].new(Salus_display.createGroup(), instrument_dir~"salus-g_force.svg");
	Salus_main[5] = canvas_Salus_main[5].new(Salus_display.createGroup(), instrument_dir~"salus-ahrs.svg");
	Salus_main[6] = canvas_Salus_main[6].new(Salus_display.createGroup(), instrument_dir~"salus-gps_info.svg");
	Salus_main[7] = canvas_Salus_main[7].new(Salus_display.createGroup(), instrument_dir~"salus-logbook.svg");
	Salus_main[8] = canvas_Salus_main[8].new(Salus_display.createGroup(), instrument_dir~"salus-setup.svg");
	Salus_start = canvas_Salus_start.new(Salus_display.createGroup(), instrument_dir~"salus-start.svg");

	Salus_main[0].show_qnh_popup(0);
	Salus_main[0].update_qnh();
	Salus_main[4].show_menu(0);
	base_updater.start();
});

setlistener( qnh, func {
	Salus_main[0].update_qnh();
});

var enter_button = func( i ){
	if( status != 2 ) return;
	if( i ) {
		# Enter Button Action
		if( page == 0){
			Salus_main[0].show_qnh_popup();
		}elsif( page == 4 ){
			if( Salus_main[4].menu == 0 ){
				Salus_main[4].show_menu(1);
			} elsif( Salus_main[4].menu == 1 or Salus_main[4].menu == 4 ){
				Salus_main[4].show_menu(0);
			} elsif( Salus_main[4].menu == 2 ){
				screen.log.write( "Setting G-Force Range is not yet supported!" );
			} elsif( Salus_main[4].menu == 3 ){
				Salus_main[4].reset_g_limits();
				Salus_main[4].show_menu(0);
			}
		}
	} else {
		# Enter Button Release Action
	}
}

var knob = func( i ){
	if( status != 2 ) return;
	if( qnh_popup_visible ){
		qnh.setDoubleValue( qnh.getDoubleValue() + i );
	} elsif( Salus_main[4].menu > 0 ) {
		Salus_main[4].menu += i;
		if( Salus_main[4].menu < 1 ) Salus_main[4].menu = 4;
		if( Salus_main[4].menu > 4 ) Salus_main[4].menu = 1;
		Salus_main[4].update_menu_cursor();
	} else {
		page += i;
		if( page < 0 ) page = 8;
		if( page > 8 ) page = 0;
	}
}
