# BIRDY ELECTRICAL SYSTEM
# based on turboprop engine electrical system by Syd Adams and C172S electrical system   ####

#	Components:
#		Engine:			Geiger Engineering HPD16
#		Motor Controller:	Geiger Engineering MC300 16 kW
#		Battery:		128 Ah -> probably 1 x 60 Ah + 1 x 68 Ah

# Basic props.nas objects
var electrical = props.globals.getNode("systems/electrical");
var electrical_sw = electrical.initNode("internal-switches");
var output = electrical.getNode("outputs");
var breakers = props.globals.initNode("/controls/circuit-breakers");
var ctrls = props.globals.getNode("/controls/electric");

# Helper functions
var check_or_create = func ( prop, value, type ) {
	var obj = props.globals.getNode(prop, 1);
	if( obj.getValue() == nil ){
		return props.globals.initNode(prop, value, type);
	} else {
		return obj;
	}
}

var eng_power	= props.globals.getNode("/fdm/jsbsim/propulsion/engine/electrical-power-hp", 1);

#	Switches
var switches = {
	battery:		ctrls.initNode("battery-switch",		0,	"BOOL"),
	avionics:		ctrls.initNode("avionics-switch",		0,	"BOOL"),
};

#	Additional (not directly consumer-related) Circuit Breakers
var circuit_breakers = {
};

#	Internal (virtual) switches
var int_switches = {
};

var delta_sec	=	props.globals.getNode("sim/time/delta-sec");

var BatteryClass = {
	new : func( switch, volt, amps, amp_hours, charge_percent, charge_amps, n){
		m = { 
			parents : [BatteryClass],
			switch:		switch,
			serviceable:	electrical.initNode("/systems/electrical/battery["~n~"]/serviceable", 1, "BOOL"),
			temp:		electrical.initNode("battery["~n~"]/temperature", 15.0, "DOUBLE"),
			ideal_volts:	volt,
			ideal_amps:	amps,
			volt_p:		electrical.initNode("battery["~n~"]/volts", 0.0, "DOUBLE"),
			actual_amp_hours:	electrical.initNode("battery["~n~"]/amp-hours", 0.0, "DOUBLE"),
			amp_hours:	amp_hours,
			charge_percent:	charge_percent, 
			charge_amps:	charge_amps,
		};
		return m;
	},
	apply_load : func( load ) {
		var dt = delta_sec.getDoubleValue();
		if( me.switch.getBoolValue() ){
			var amphrs_used = load * dt / 3600.0;
			var percent_used = amphrs_used / me.amp_hours;
			me.charge_percent -= percent_used;
			if ( me.charge_percent < 0.0 ) {
				me.charge_percent = 0.0;
			} elsif ( me.charge_percent > 1.0 ) {
				me.charge_percent = 1.0;
			}
			var output = me.amp_hours * me.charge_percent;
			me.actual_amp_hours.setDoubleValue( output );
			return output;
		}else return 0;
	},
	
	get_output_volts : func {
		if( me.switch.getBoolValue() ){
			var x = 1.0 - me.charge_percent;
			var tmp = -(3.0 * x - 1.0);
			var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
			var output =me.ideal_volts * factor;
			me.volt_p.setDoubleValue( output );
			return output;
		}else return 0;
	},
	
	get_output_amps : func {
		if( me.switch.getBoolValue() ){
			var x = 1.0 - me.charge_percent;
			var tmp = -(3.0 * x - 1.0);
			var factor = (tmp*tmp*tmp*tmp*tmp + 32) / 32;
			var output =me.ideal_amps * factor;
			return output;
		}else return 0;
	}
};

var battery = BatteryClass.new(switches.battery,60.0, 100, 128.0, 1.0, 7.0, 0);

#	Buses
#		Battery Engine Bus (60V)
#		

#	Consumer Class
#		Functions:
#			* power: takes bus_volts, applies to relevant outputs/ property and returns electrical load
#			* automatically trips its circuit breaker when maximum load is exceeded
var consumer = {
	new: func( name, switch, watts, cb_max ){
		m = { parents : [consumer] };
		m.cb = breakers.initNode(name, 1, "BOOL");
		m.switch_type = "none";
		if( switch != nil ){
			m.switch = switch;
			if ( switch.getType() == "DOUBLE" or switch.getType() == "FLOAT" ) {
				m.switch_type = "double";
			} else if ( switch.getType() == "BOOL" ) {
				m.switch_type = "bool";
			} else {
				die("Consumer (non-int) switch of unsupported type: "~ switch.getType() ~ "!");
			}
		} else {
			m.switch = nil;
		}
		m.output = output.initNode(name, 0.0, "DOUBLE");
		m.watts = watts;
		m.cb_max = cb_max;
		return m;
	},
	power: func( bus_volts ){
		if( me.cb.getBoolValue() and bus_volts != 0.0 ){
			if ( me.switch_type == "none" or ( me.switch_type == "bool" and me.switch.getBoolValue() ) ) {
				me.output.setDoubleValue( bus_volts );
				# if( me.watts/bus_volts > me.cb_max ){
				# 	me.cb.setBoolValue( 0 );
				# 	return 0.0;
				# }
				return me.watts / bus_volts;
			} else if ( me.switch_type == "double" ) {
				me.output.setDoubleValue( bus_volts * me.switch.getDoubleValue() );
				# if( me.watts / bus_volts * me.switch.getDoubleValue() > me.cb_max ){
				# 	me.cb.setBoolValue( 0 );
				# 	return 0.0;
				# }
				return me.watts / bus_volts * me.switch.getDoubleValue();
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};
# Consumer with support for integer switches
var consumer_int = {
	new: func( name, switch, watts, int, mode ){
		m = { parents : [consumer_int] };
		m.cb = breakers.initNode(name, 1, "BOOL");
		if ( switch.getType() == "INT" ) {
			m.switch = switch;
			m.int = int;
			# Mode: 0 means "=="; 1 means "!="
			if( mode != nil ){
				m.mode = mode;
			} else {
				m.mode = 0;
			}
		} else {
			die("Consumer (int) switch of unsupported type: "~ switch.getType() ~ "!");
		}
		m.output = output.initNode(name, 0.0, "DOUBLE");
		m.watts = watts;
		return m;
	},
	power: func( bus_volts ){
		if( me.cb.getBoolValue() and bus_volts != 0.0 ){
			if ( ( ( me.mode == 0 and me.switch.getIntValue() == me.int ) or ( me.mode == 1 and me.switch.getIntValue() != me.int ) ) ) {
				me.output.setDoubleValue( bus_volts );
				return me.watts / bus_volts;
			} else {
				me.output.setDoubleValue( 0.0 );
				return 0.0;
			}
		} else {
			me.output.setDoubleValue( 0.0 );
			return 0.0;
		}
	},
};

#	Electrical Bus Class
var bus = {
	new: func( name, on_update, consumers ) {
		m = {	
			parents: [bus],
			name: name,
			volts: check_or_create("systems/electrical/" ~ name ~ "/volts", 0.0, "DOUBLE"),
			amps: check_or_create("systems/electrical/"~ name ~"/amps", 0.0, "DOUBLE"),
			serviceable: check_or_create("systems/electrical/" ~ name ~ "/serviceable", 1, "BOOL"),
			on_update: on_update,
			bus_volts: 0.0,
			consumers: consumers,
			extra_load: 0.0,
			src: "",
		};
		return m;
	},
	update_consumers: func () {
		#print("Update consumers of bus "~ me.name);
		load = 0.0;
		foreach( var c; me.consumers ) {
			load += c.power( me.bus_volts );
		}
		return load;
	},
};
#	Battery Bus
#		connected to main battery
#		powers:
#			Engine Bus
var battery_bus = bus.new(
	"battery-bus",
	func() {
		me.src = "";
		if( me.serviceable.getBoolValue() ){
			me.bus_volts = battery.get_output_volts();
			me.src = "battery";
		} else {
			me.bus_volts = 0.0;
		}
		
		var load = me.update_consumers();
		load += engine_bus.on_update( me.bus_volts );
		load += engine_avionics_bus.on_update( me.bus_volts );
		
		battery.apply_load( load );
		
		me.extra_load = 0.0;
		me.volts.setDoubleValue( me.bus_volts );
		me.amps.setDoubleValue( load );
	},
	[
	],
);

#	Engine Bus
var engine_bus = bus.new(
	"engine-bus",
	func( bv ){
		me.src = "";
		me.bus_volts = bv;
		
		var load = me.update_consumers();
		
		# Calculate engine load
		var power_kw = math.max( eng_power.getDoubleValue() * 0.7457, 0 );
		# P = U * I -> I = P / U
		load += ( power_kw * 1000 ) / me.bus_volts;
		
		me.volts.setDoubleValue( me.bus_volts );
		me.amps.setDoubleValue( load );
		
		return load;
	},
	[
	],
);

#	Engine Avionics Bus
var engine_avionics_bus = bus.new(
	"engine-avionics-bus",
	func( bv ){
		me.src = "";
		me.bus_volts = bv * 0.1;
		
		var load = me.update_consumers();
		
		me.volts.setDoubleValue( me.bus_volts );
		me.amps.setDoubleValue( load );
		
		return load * 0.1 ;
	},
	[
		consumer.new( "sdi", nil, 0.25, 999 ),	# 50mA (manual p.98) at 5V
	],
);

var update_electrical = func {
	
	battery_bus.on_update();
	
}

var electrical_updater = maketimer( 0.0, update_electrical );
electrical_updater.simulatedTime = 1;
electrical_updater.start();
