/*
   Copyright (C) nerdctl authors.
   Copyright (C) containerd authors.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
*/

package main

import (
	"os"
	"os/exec"
	"strings"

	"github.com/pkg/errors"
	"github.com/sirupsen/logrus"
	"github.com/urfave/cli/v2"
)

var buildCommand = &cli.Command{
	Name:   "build",
	Usage:  "Build an image from a Dockerfile. Needs buildkitd to be running.",
	Action: buildAction,
	Flags: []cli.Flag{
		&cli.StringSliceFlag{
			Name:    "tag",
			Aliases: []string{"t"},
			Usage:   "Name and optionally a tag in the 'name:tag' format",
		},
		&cli.StringFlag{
			Name:    "file",
			Aliases: []string{"f"},
			Usage:   "Name of the Dockerfile",
		},
		&cli.StringFlag{
			Name:  "target",
			Usage: "Set the target build stage to build",
		},
		&cli.StringSliceFlag{
			Name:  "build-arg",
			Usage: "Set build-time variables",
		},
		&cli.BoolFlag{
			Name:  "no-cache",
			Usage: "Do not use cache when building the image",
		},
		&cli.StringFlag{
			Name:  "progress",
			Usage: "Set type of progress output (auto, plain, tty). Use plain to show container output",
			Value: "auto",
		},
		&cli.StringSliceFlag{
			Name:  "secret",
			Usage: "Secret file to expose to the build: id=mysecret,src=/local/secret",
		},
		&cli.StringSliceFlag{
			Name:  "ssh",
			Usage: "SSH agent socket or keys to expose to the build (format: default|<id>[=<socket>|<key>[,<key>]])",
		},
	},
}

func buildAction(clicontext *cli.Context) error {
	buildctlBinary := "buildctl"
	buildctlArgs, err := generateBuildctlArgs(clicontext)
	if err != nil {
		return err
	}

	buildctlCheckCmd := exec.Command(buildctlBinary, "debug", "workers")
	buildctlCheckCmd.Env = os.Environ()
	if out, err := buildctlCheckCmd.CombinedOutput(); err != nil {
		logrus.Error(string(out))
		return errors.Wrap(err, "`buildctl` needs to be installed and `buildkitd` needs to be running, see https://github.com/moby/buildkit")
	}

	logrus.Debugf("running %s %v", buildctlBinary, buildctlArgs)
	buildctlCmd := exec.Command(buildctlBinary, buildctlArgs...)
	buildctlCmd.Env = os.Environ()

	buildctlStdout, err := buildctlCmd.StdoutPipe()
	if err != nil {
		return err
	}
	buildctlCmd.Stderr = clicontext.App.ErrWriter

	if err := buildctlCmd.Start(); err != nil {
		return err
	}

	if err = loadImage(buildctlStdout, clicontext); err != nil {
		return err
	}

	if err = buildctlCmd.Wait(); err != nil {
		return err
	}

	return nil
}

func generateBuildctlArgs(clicontext *cli.Context) ([]string, error) {
	if clicontext.NArg() < 1 {
		return nil, errors.New("context needs to be specified")
	}
	buildContext := clicontext.Args().First()
	if buildContext == "-" || strings.Contains(buildContext, "://") {
		return nil, errors.Errorf("unsupported build context: %q", buildContext)
	}

	output := "--output=type=docker"
	if tagSlice := clicontext.StringSlice("tag"); len(tagSlice) > 0 {
		if len(tagSlice) > 1 {
			return nil, errors.Errorf("specifying multiple -t is not supported yet")
		}
		output += ",name=" + tagSlice[0]
	}

	buildctlArgs := []string{
		"build",
		"--progress=" + clicontext.String("progress"),
		"--frontend=dockerfile.v0",
		"--local=context=" + buildContext,
		"--local=dockerfile=" + buildContext,
		output,
	}

	if filename := clicontext.String("file"); filename != "" {
		buildctlArgs = append(buildctlArgs, "--opt=filename="+filename)
	}

	if target := clicontext.String("target"); target != "" {
		buildctlArgs = append(buildctlArgs, "--opt=target="+target)
	}

	for _, ba := range clicontext.StringSlice("build-arg") {
		buildctlArgs = append(buildctlArgs, "--opt=build-arg:"+ba)
	}

	if clicontext.Bool("no-cache") {
		buildctlArgs = append(buildctlArgs, "--no-cache")
	}

	for _, s := range clicontext.StringSlice("secret") {
		buildctlArgs = append(buildctlArgs, "--secret="+s)
	}

	for _, s := range clicontext.StringSlice("ssh") {
		buildctlArgs = append(buildctlArgs, "--ssh="+s)
	}

	return buildctlArgs, nil
}
