// Copyright 2020 OpenTelemetry Authors
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package data

import (
	otlpcommon "github.com/open-telemetry/opentelemetry-proto/gen/go/common/v1"
	otlpresource "github.com/open-telemetry/opentelemetry-proto/gen/go/resource/v1"
)

// Resource information.
//
// Must use NewResource functions to create new instances.
// Important: zero-initialized instance is not valid for use.
type Resource struct {
	// Set of attributes that describe the resource.
	attributes AttributesMap
}

func NewResource() *Resource {
	return &Resource{}
}

func newResource(orig *otlpresource.Resource) *Resource {
	if orig == nil {
		return &Resource{}
	}
	origAttr := orig.Attributes
	attr := make(AttributesMap, len(origAttr))
	for i := range origAttr {
		attr[origAttr[i].GetKey()] = AttributeValue{origAttr[i]}
	}
	return &Resource{attr}
}

func (r *Resource) Attributes() AttributesMap {
	return r.attributes
}

func (r *Resource) SetAttributes(v AttributesMap) {
	r.attributes = v
}

func (r *Resource) toOrig() *otlpresource.Resource {
	if r.attributes == nil {
		return nil
	}
	attrs := make([]otlpcommon.AttributeKeyValue, len(r.attributes))
	wrappers := make([]*otlpcommon.AttributeKeyValue, len(r.attributes))
	i := 0
	for k, v := range r.attributes {
		attrs[i] = *v.orig
		attrs[i].Key = k
		wrappers[i] = &attrs[i]
		i++
	}
	return &otlpresource.Resource{Attributes: wrappers}
}
