// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/polymer/v3_0/iron-icon/iron-icon.js';
import 'chrome://resources/polymer/v3_0/paper-spinner/paper-spinner-lite.js';
import 'chrome://resources/cr_elements/icons.html.js';
import './signin_shared.css.js';
import './signin_vars.css.js';
import './strings.m.js';

import {CrButtonElement} from 'chrome://resources/cr_elements/cr_button/cr_button.js';
import {loadTimeData} from 'chrome://resources/js/load_time_data.m.js';
import {WebUIListenerMixin} from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import {afterNextRender, PolymerElement} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

import {getTemplate} from './dice_web_signin_intercept_app.html.js';
import {DiceWebSigninInterceptBrowserProxy, DiceWebSigninInterceptBrowserProxyImpl, InterceptionParameters} from './dice_web_signin_intercept_browser_proxy.js';

const DiceWebSigninInterceptAppElementBase = WebUIListenerMixin(PolymerElement);

export interface DiceWebSigninInterceptAppElement {
  $: {
    cancelButton: CrButtonElement,
    acceptButton: CrButtonElement,
  };
}

export class DiceWebSigninInterceptAppElement extends
    DiceWebSigninInterceptAppElementBase {
  static get is() {
    return 'dice-web-signin-intercept-app';
  }

  static get template() {
    return getTemplate();
  }

  static get properties() {
    return {
      interceptionParameters_: {
        type: Object,
        value: null,
      },

      acceptButtonClicked_: {
        type: Boolean,
        value: false,
      },

      guestLink_: {
        type: String,
        value() {
          return loadTimeData.getString('guestLink');
        },
      },
    };
  }

  private interceptionParameters_: InterceptionParameters;
  private acceptButtonClicked_: boolean;
  private guestLink_: string;
  private diceWebSigninInterceptBrowserProxy_:
      DiceWebSigninInterceptBrowserProxy =
          DiceWebSigninInterceptBrowserProxyImpl.getInstance();

  override connectedCallback() {
    super.connectedCallback();

    this.addWebUIListener(
        'interception-parameters-changed',
        this.handleParametersChanged_.bind(this));
    this.diceWebSigninInterceptBrowserProxy_.pageLoaded().then(
        parameters => this.onPageLoaded_(parameters));
  }

  private onPageLoaded_(parameters: InterceptionParameters) {
    this.handleParametersChanged_(parameters);
    afterNextRender(this, () => {
      // |showGuestOption| is constant during the lifetime of this bubble,
      // therefore it's safe to set the listener only during initialization.
      if (this.interceptionParameters_.showGuestOption) {
        this.shadowRoot!.querySelector('#footer-description a')!
            .addEventListener('click', () => this.onGuest_());
      }
    });
  }

  private onAccept_() {
    this.acceptButtonClicked_ = true;
    this.diceWebSigninInterceptBrowserProxy_.accept();
  }

  private onCancel_() {
    this.diceWebSigninInterceptBrowserProxy_.cancel();
  }

  private onGuest_() {
    if (this.acceptButtonClicked_) {
      return;
    }
    this.acceptButtonClicked_ = true;
    this.diceWebSigninInterceptBrowserProxy_.guest();
  }

  /** Called when the interception parameters are updated. */
  private handleParametersChanged_(parameters: InterceptionParameters) {
    this.interceptionParameters_ = parameters;
    this.style.setProperty(
        '--intercepted-profile-color', parameters.interceptedProfileColor);
    this.style.setProperty(
        '--primary-profile-color', parameters.primaryProfileColor);
    this.style.setProperty('--header-text-color', parameters.headerTextColor);
    this.notifyPath('interceptionParameters_.interceptedAccount.isManaged');
    this.notifyPath('interceptionParameters_.primaryAccount.isManaged');
  }
}

declare global {
  interface HTMLElementTagNameMap {
    'dice-web-signin-intercept-app': DiceWebSigninInterceptAppElement;
  }
}

customElements.define(
    DiceWebSigninInterceptAppElement.is, DiceWebSigninInterceptAppElement);
