// Copyright (C) 2016 The Qt Company Ltd.
// SPDX-License-Identifier: LicenseRef-Qt-Commercial OR GPL-3.0-only WITH Qt-GPL-exception-1.0

#pragma once

#include "searchsymbols.h"

#include <coreplugin/find/ifindfilter.h>
#include <coreplugin/find/searchresultitem.h>
#include <coreplugin/find/searchresultwindow.h>

#include <QFutureWatcher>
#include <QPointer>
#include <QWidget>
#include <QCheckBox>
#include <QRadioButton>

namespace Core { class SearchResult; }

namespace CppEditor {
class CppModelManager;

namespace Internal {

class SymbolsFindFilter : public Core::IFindFilter
{
    Q_OBJECT

public:
    using SearchScope = SymbolSearcher::SearchScope;

public:
    explicit SymbolsFindFilter(CppModelManager *manager);

    QString id() const override;
    QString displayName() const override;
    bool isEnabled() const override;

    void findAll(const QString &txt, Core::FindFlags findFlags) override;

    QWidget *createConfigWidget() override;
    void writeSettings(QSettings *settings) override;
    void readSettings(QSettings *settings) override;

    void setSymbolsToSearch(const SearchSymbols::SymbolTypes &types) { m_symbolsToSearch = types; }
    SearchSymbols::SymbolTypes symbolsToSearch() const { return m_symbolsToSearch; }

    void setSearchScope(SearchScope scope) { m_scope = scope; }
    SearchScope searchScope() const { return m_scope; }

signals:
    void symbolsToSearchChanged();

private:
    void openEditor(const Core::SearchResultItem &item);

    void addResults(QFutureWatcher<Core::SearchResultItem> *watcher, int begin, int end);
    void finish(QFutureWatcher<Core::SearchResultItem> *watcher);
    void cancel(Core::SearchResult *search);
    void setPaused(Core::SearchResult *search, bool paused);
    void onTaskStarted(Utils::Id type);
    void onAllTasksFinished(Utils::Id type);

    QString label() const;
    QString toolTip(Core::FindFlags findFlags) const;
    void startSearch(Core::SearchResult *search);

    CppModelManager *m_manager;
    bool m_enabled;
    QMap<QFutureWatcher<Core::SearchResultItem> *, QPointer<Core::SearchResult> > m_watchers;
    QPointer<Core::SearchResult> m_currentSearch;
    SearchSymbols::SymbolTypes m_symbolsToSearch;
    SearchScope m_scope;
};

class SymbolsFindFilterConfigWidget : public QWidget
{
    Q_OBJECT
public:
    explicit SymbolsFindFilterConfigWidget(SymbolsFindFilter *filter);

private:
    void setState() const;
    void getState();

    SymbolsFindFilter *m_filter;

    QCheckBox *m_typeClasses;
    QCheckBox *m_typeMethods;
    QCheckBox *m_typeEnums;
    QCheckBox *m_typeDeclarations;

    QRadioButton *m_searchGlobal;
    QRadioButton *m_searchProjectsOnly;
    QButtonGroup *m_searchGroup;
};

} // Internal
} // CppEditor
